/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.IdentityImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.variable.ConditionalDerivedVariableImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.variable.DerivedVariableImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.variable.ParameterGroupImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.variable.ResourcePropertyImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Identity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ConditionalDerivedVariable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.DerivedVariable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ParameterGroup;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ResourceProperty;
import org.w3c.dom.Element;

public abstract class ContentUnitImpl extends SPIDataObject implements ContentUnit {
	private static final String VARIABLES_TYPE = "sdd-dd:VariablesType";
	
	public ContentUnitImpl(Element data, SPISessionImpl session, String sddSchemaType) {
		super(data, session, sddSchemaType);
	}
		
	public String getContentUnitID() {
	    return getAttribute("id");
	}

	public Identity getIdentityData() {
		Identity identityData = null;
        
        if (isSetElement("Identity")) {
        	identityData = new IdentityImpl(getChild("Identity"), session);
        }

        return identityData;	
	}

	public Collection<DerivedVariable> getDerivedVariables() {
		Collection<DerivedVariable> derivedVariables = new ArrayList<DerivedVariable>();
		Collection<Element> derivedVariableElements = session.getChildren(getChild("Variables"), "DerivedVariable");
		
		for (Element e : derivedVariableElements) {
			if (session.getChild(e, "Expression") != null) {
				derivedVariables.add(new DerivedVariableImpl(e, session));
			}
		}		

		return derivedVariables;
	}
	
	public Collection<ConditionalDerivedVariable> getConditionalDerivedVariables() {
		Collection<ConditionalDerivedVariable> derivedVariables = new ArrayList<ConditionalDerivedVariable>();
		Collection<Element> derivedVariableElements = session.getChildren(getChild("Variables"), "DerivedVariable");
		
		for (Element e : derivedVariableElements) {
			if (session.getChild(e, "Expression") == null) {
				derivedVariables.add(new ConditionalDerivedVariableImpl(e, session));
			}
		}		

		return derivedVariables;
	}
	
	public Collection<ResourceProperty> getResourceProperties() {
		Collection<ResourceProperty> resourceProperties = new ArrayList<ResourceProperty>();
		Collection<Element> resourcePropertyElements = session.getChildren(getChild("Variables"), "ResourceProperty");
		
		for (Element e : resourcePropertyElements) {
			resourceProperties.add(new ResourcePropertyImpl(e, session));
		}
				
		return resourceProperties;
	}
	
	public Collection<ParameterGroup> getParameterGroups() {
		Collection<ParameterGroup> parameterGroups = new ArrayList<ParameterGroup>();
		Collection<Element> parameterElements = session.getChildren(getChild("Variables"), "Parameters");
		
		for (Element e : parameterElements) {
			parameterGroups.add(new ParameterGroupImpl(e, session));
		}
		
		return parameterGroups;
	}

	public void setContentUnitID(String contentUnitID) {
		session.testParameter(contentUnitID, 1, null);

		setAttribute("id", contentUnitID);
	}

	public void setIdentityData(Identity contentUnitIdentity) {
		if (contentUnitIdentity == null) {
            unSetElement("Identity");
        }
        else {
            setContent((SPIDataObject)contentUnitIdentity);
        }
	}
	
	public void setResourceProperties(Collection<ResourceProperty> newResourceProperties) {
		Element variablesElement = null;
		
		if (isSetElement("Variables")) {
			variablesElement = getChild("Variables");
			session.removeChildren(variablesElement, "ResourceProperty");
		}
		else {
			variablesElement = session.createDeploymentElement("Variables");
		}
		
		if (newResourceProperties != null && !newResourceProperties.isEmpty()) {
			for (ResourceProperty rp : newResourceProperties) {
				session.insertNode(variablesElement, (SPIDataObject)rp, VARIABLES_TYPE);
			}
		}
		
		setContent(variablesElement);
	}	
	
	public void setDerivedVariables(Collection<DerivedVariable> derivedVariables) {
		session.testParameter(derivedVariables, 1, null);
		
		Element variablesElement = null;
		
		if (isSetElement("Variables")) {
			variablesElement = getChild("Variables");
			Collection<Element> derivedVariableElements = session.getChildren(variablesElement, "DerivedVariable");
			
			for (Element e : derivedVariableElements) {
				if (session.getChild(e, "Expression") != null) {
					variablesElement.removeChild(e);
				}
			}
		}
		else {
			variablesElement = session.createDeploymentElement("Variables");
		}
		
		if (derivedVariables != null && !derivedVariables.isEmpty()) {
			for (DerivedVariable dv : derivedVariables) {
				session.insertNode(variablesElement, (SPIDataObject)dv, VARIABLES_TYPE);
			}
		}
		
		setContent(variablesElement);
	}
	
	public void setConditionalDerivedVariables(Collection<ConditionalDerivedVariable> conditionalDerivedVariables) {
		Element variablesElement = null;
		
		if (isSetElement("Variables")) {
			variablesElement = getChild("Variables");
			Collection<Element> derivedVariableElements = session.getChildren(variablesElement, "DerivedVariable");
			
			for (Element e : derivedVariableElements) {
				if (session.getChild(e, "Expression") == null) 	{
					variablesElement.removeChild(e);
				}
			}
		}
		else {
			variablesElement = session.createDeploymentElement("Variables");
		}
		
		if (conditionalDerivedVariables != null && !conditionalDerivedVariables.isEmpty()) {
			for (ConditionalDerivedVariable dv : conditionalDerivedVariables) {
				session.insertNode(variablesElement, (SPIDataObject)dv, VARIABLES_TYPE);
			}
		}
		
		setContent(variablesElement);
	}

	
	public void setParameterGroups(Collection<ParameterGroup> newParameterGroups) {
		Element variablesElement = null;
		
		if (isSetElement("Variables")) {
			variablesElement = getChild("Variables");
			session.removeChildren(variablesElement, "Parameters");
		}
		else {
			variablesElement = session.createDeploymentElement("Variables");
		}
		
		if (newParameterGroups != null && !newParameterGroups.isEmpty()) {
			for (ParameterGroup pg : newParameterGroups) {
				session.insertNode(variablesElement, (SPIDataObject)pg, VARIABLES_TYPE);
			}
		}
		
		setContent(variablesElement);
	}
	
	public String toString() {
		String tmp = new String();
		tmp = "ID: " + getContentUnitID() + "\n";
		return tmp;
	}	
}