/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.util.Collection;
import java.util.Date;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;


/**
 * A BaseFactory is used to create some of the basic container objects, which
 * group together lower level SPI objects.
 *
 */
public interface BaseFactory {
	
	/**
	 * Creates a new DeploymentDescriptor.  
	 * 
	 * @param schemaVersion The SDD schema version complied with, typically "1.0"
	 * @param descriptorID A unique identifier for the descriptor
	 * @param lastModified The date and time that the descriptor was most recently modified
	 * @param content The content unit to be contained by the descriptor
	 * @param topology Resources that may be referred to in the content unit
	 * @return A new DeploymentDescriptor object
	 * @throws InvalidInputException if any of the parameters is null
	 */
	DeploymentDescriptor createDeploymentDescriptor(
			String schemaVersion, byte[] descriptorID, Date lastModified,
			ContentUnit content, Collection<Resource> topology);
	
	/**
	 * Creates a new Resource.
	 * 
	 * @param resourceID An identifier of the resource scoped to the descriptor
	 * @param resourceType A well-known resource type
	 * @return A new Resource object with element name "Resource"
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Resource createResource(String resourceID, String resourceType);
	
	/**
	 * Creates a new HostedResource.
	 * 
	 * @param resourceID An identifier of the resource scoped to the descriptor
	 * @param resourceType A well-known resource type
	 * @return A new Resource object with element name "HostedResource"
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Resource createHostedResource(String resourceID, String resourceType);
	
	
	/**
	 * Creates a new Identity.
	 * 
	 * @param name Identity's translatable name
	 * @return An Identity object, which may contain various descriptive
	 * information about another object
	 * @throws InvalidInputException if the parameter is null
	 */
	Identity createIdentity(TranslatableString name);
	
	/**
	 * Creates a new ManufacturerInformation.
	 * 
	 * @param name The Manufacturer's name
	 * @return a ManufacturerInformation object
	 * @throws InvalidInputException if the parameter is null
	 */
	ManufacturerInformation createManufacturerInformation(TranslatableString name);
	
	/**
	 * Creates a new MaintenanceInformation.
	 * 
	 * @return a MaintenanceInformation object with element name "MaintenanceInformation"
	 */
	MaintenanceInformation createMaintenanceInformation();
	
	/**
	 * Creates a new Supersedes.
	 * 
	 * @return a MaintenanceInformation object with element name "Supersedes"
	 */
	MaintenanceInformation createSupersedes();
	
	/**
	 * Creates a new FixIdentity.
	 * 
	 * @param fixName The name of the fix to be described
	 * @return a FixIdentity object, which may contain various descriptive
	 * information about a fix
	 * @throws InvalidInputException if the parameter is null
	 */
	FixIdentity createFixIdentity(String fixName);
	
	
	/**
	 * Creates a new PackageDescriptor.  
	 *  
	 * @param schemaVersion The SDD schema version complied with, typically "1.0"
	 * @param descriptorID A unique identifier for the descriptor
	 * @param lastModified The date and time that the descriptor was most recently modified
	 * @param identity A PackageIdentity object, describing the package
	 * @param packageContents a PackageContent collection, list the contents that are
	 * part of the package.
	 * @return a new PackageDescriptor object
	 * @throws InvalidInputException if any of the parameters is null
	 */
	PackageDescriptor createPackageDescriptor(
			String schemaVersion, byte[] descriptorID, Date lastModified,
			PackageIdentity identity, Collection<PackageContent> packageContents);
	
	/**
	 * Creates a new PackageContent.
	 * 
	 * @param contentID An identifier that refers to the content definition in 
	 * a package descriptor
	 * @param pathName The absolute or relative path of the content
	 * @return a new PackageContent object
	 * @throws InvalidInputException if any of the parameters is null
	 */
	PackageContent createPackageContent(String contentID, String pathName);
	
	/**
	 * Creates a new PackageIdentity.
	 * 
	 * @return a new PackageIdentity object, which may contain various
	 * descriptive information about a package
	 */
	PackageIdentity createPackageIdentity();
	
	/**
	 * Creates a new ReferencedPackage.
	 * 
	 * @param contentRef Reference to the identifier of the package Content defined in 
	 * the package descriptor which identifies the package descriptor of the referenced package
	 * @param id Identifier for the referenced package element that is unique within 
	 * the deployment descriptor
	 * @return a new ReferencedPackage object with element name "ReferencedPackage"
	 * @throws InvalidInputException if any of the parameters is null
	 */
	ReferencedPackage createReferencedPackage(String contentRef, String id);
	
	/**
	 * Creates a new ContainedPackage.
	 * 
	 * @param contentRef Reference to the identifier of the package Content defined in 
	 * the package descriptor which identifies the package descriptor of the referenced package
	 * @param id Identifier for the referenced package element that is unique within 
	 * the deployment descriptor
	 * @return a new ReferencedPackage object with element name "ContainedPackage"
	 * @throws InvalidInputException if any of the parameters is null
	 */
	ReferencedPackage createContainedPackage(String contentRef, String id);
	
	/**
	 * Creates a new ContainedLocalizationPackage.
	 * 
	 * @param contentRef Reference to the identifier of the package Content defined in 
	 * the package descriptor which identifies the package descriptor of the referenced package
	 * @param id Identifier for the referenced package element that is unique within 
	 * the deployment descriptor
	 * @return a new ReferencedPackage object with element name "ContainedLocalizationPackage"
	 * @throws InvalidInputException if any of the parameters is null
	 */
	ReferencedPackage createContainedLocalizationPackage(String contentRef, String id);
	
	/**
	 * Creates a new ResultingChangeMap.
	 * 
	 * @param resourceRef Reference to a resource defined in the deployment descriptor
	 * @return a new ResultingChangeMap object
	 * @throws InvalidInputException if the parameter is null
	 */
	ResultingChangeMap createResultingChangeMap(String resourceRef);
	
	/**
	 * Creates a new ResultingResourceMap.
	 * 
	 * @param resourceRef Reference to a resource defined in the deployment descriptor
	 * @return a new ResultingResourceMap object
	 * @throws InvalidInputException if the parameter is null
	 */
	ResultingResourceMap createResultingResourceMap(String resourceRef);
	
	/**
	 * Creates a new ResourceMap.
	 * 
	 * @param resourceRef Reference to a resource defined in the deployment descriptor
	 * @return a new ResourceMap object
	 * @throws InvalidInputException if the parameter is null
	 */
	ResourceMap createRequiredResourceMap(String resourceRef);
}
