/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.math.BigInteger;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact.ArtifactArgument;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentSelectionFeature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Language;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.OutputVariable;

/**
 * A referenced package identifies an aggregated SDD and describes the conditions of its aggregation. ReferencedPackageType provides 
 * the type definition for ContainedPackage and Requisite elements. ContainedPackage elements identify an SDD package that is treated 
 * like a content element of the defining SDD. Requisite elements identify an SDD package that can be deployed, if necessary, to satisfy
 * resource constraints.
 */
public interface ReferencedPackage {
	
    /**
     * Gets the required groups.
     * 
     * Certain Groups may need to be selected when deploying the referenced package. These can be identified in the 
     * RequiredContentSelection element.<br><br>
     * If one particular aggregated SDD requires the selection of different groups, depending on other choices made 
     * during a particular deployment, different Requisite or ContainedPackage elements can be defined in a way that will cause the 
     * correct combination of Groups to be used in each situation.
     * 
     * @return Groups that MUST be selected when the referenced package is deployed.
     */
    String getRequiredGroup();
    
    /**
     * Gets the required features.
     * 
     * Certain Features may need to be selected when deploying the referenced package. These can be identified in the 
     * RequiredContentSelection element.<br><br>
     * If one particular aggregated SDD requires the selection of different features, depending on other choices made 
     * during a particular deployment, different Requisite or ContainedPackage elements can be defined in a way that will cause the 
     * correct combination of Features to be used in each situation.
     * 
     * @return A collection of features that MUST be selected when the referenced package is deployed.
     */
    Collection<ContentSelectionFeature> getRequiredFeatures();
    
    /**
     * Gets the ArtifactArguments.
     * 
     * Arguments are used to provide values for input variables defined in the deployment descriptor of the referenced package. The 
     * argument name specified MUST reference the id of a parameter in the referenced package.
     * 
     * @return A collection of ArtifactArguments.
     */
    Collection<ArtifactArgument> getArguments();
    
    /**
     * Gets the OutputVariables.
     * 
     * The output variable mapping can be used to set variables to outputs created by processing the referenced SDD. The output 
     * variables in the referenced package are mapped to output variables in the aggregating SDD.<br><br>
     * Each output variable value specified MUST reference the id of an output variable in the referenced package. This can be an 
     * output variable from an artifact or an output variable from a referenced package defined within the referenced SDD.
     * 
     * @return A collection of OutputVariables from the referenced package.
     */
    Collection<OutputVariable> getOutputVariables();
    
    /**
     * Gets the Requirements.
     * 
     * When the aggregating SDD has stricter requirements for the use of the referenced SDD than are defined by the referenced SDD 
     * itself, those requirements can be defined in Requirements. This is not intended to repeat requirements expressed in the 
     * referenced SDD, but rather to add additional stricter requirements.<br><br>
     * Requirements expressed in the referenced SDD need to be satisfied, in addition to the requirements expressed in the Requisite 
     * or ContainedPackage element of the aggregating SDD.<br><br>
     * Requirements expressed in the aggregating SDD MUST NOT conflict with requirements expressed in the referenced SDD. The 
     * requirements specified MUST further constrain the referenced package.
     * 
     * @return A collection of additional requirements for deploying the referenced package as part of the aggregation.
     */
    Collection<Requirement> getRequirements();
    
    /**
     * Gets the ResultingChangeMaps.
     * 
     * Resources configured by the referenced package may be resources that are defined in the aggregating SDD. The ResultingChangeMap 
     * is used to identify the correspondence between resource definitions in the aggregating SDD and changed resources defined in 
     * ResultingChange elements of the aggregated SDD.<br><br>
     * Characteristics of resources that are changed by the referenced SDD MAY be exposed in the ResultingChangeMap. These are 
     * correlated with ResourceConstraints on the changed resource in the same manner as the exposed characteristics of a resulting 
     * resource. See the property usage notes for ResultingResourceMap above.
     * 
     * @return A collection of ResultingChangeMaps.
     */
    Collection<ResultingChangeMap> getResultingChangeMaps();
    
    /**
     * Gets the ResultingResourceMaps.
     * 
     * Resources created by the referenced package may be resources that are defined in the aggregating SDD. The ResultingResourceMap 
     * is used to identify the correspondence between resource definitions in the aggregating SDD and resulting resource definitions 
     * in the aggregated SDD.<br><br>
     * Characteristics of the resulting resources MAY be exposed in the ResultingResourceMap element. ResourceConstraints defined on 
     * those resources anywhere in the aggregation are mapped to the resource properties exposed in the resulting maps of the referenced 
     * package to determine if the referenced package will satisfy the constraints. Each individual constraint is considered met by the 
     * referenced package if a property exposed in the resulting resource map that is in scope for the particular deployment satisfies 
     * the constraint.<br><br>
     * <ul>For example, a property constraint in a ResourceConstraint element states that the property named FileAttributes has the value 
     * Writeable. The resourceRef in the ResourceConstraint identifies a resource defined in Topology that is also identified in the 
     * ResultingResourceMap of a Requisite or ContainedPackage element that is in scope for the particular deployment. If the 
     * ResultingResourceMap element contains a statement that the property named FileAttributes has the value Writeable, then the 
     * ResourceConstraint is met when the Requisite or ContainedPackage is deployed.</ul>
     * This same logic applies to ResourceConstraints in aggregated packages. If the SDD in the preceding example also aggregates 
     * another SDD and maps the same resource to a required resource in that aggregated SDD, then all ResourceConstraints in the 
     * aggregated SDD are met only if the ResultingResourceMap of the referenced SDD that creates that resource contains a Name, 
     * Version or Property definition that satisfies the constraint. 
     * 
     * @return A collection of ResultingResourceMaps.
     */
    Collection<ResultingResourceMap> getResultingResourceMaps();
    
    /**
     * Gets the ResourceMaps.
     * 
     * ResourceMapType is used in the definition of elements that map resources in an SDD to resources in a referenced SDD. 
     * The purpose of a resource map is to identify when two resources in separate SDDs MUST resolve to the same resource instance 
     * during any particular deployment. The characteristics of a mapped resource that are defined in the topology sections of the 
     * two SDDs MUST NOT conflict. 
     * For example, if a Name is defined for the resource in both topologies, it MUST be the same in both definitions and if a Property 
     * definition is included for the same property in both places, the value MUST be the same.
     * Additional characteristics of a mapped resource may be constrained by Requirements or Conditions in either SDD. All constraints 
     * on a mapped resource that are in scope for a particular deployment MUST NOT conflict.
     * Resources that are not mapped between the two SDDs MAY resolve to the same instance when their characteristics defined in topology 
     * do not conflict and when the constraints in scope for any particular deployment do not conflict.
     * The RequiredResourceMap, ResultingResourceMap and ResultingChangeMap elements all use ResourceMapType, either directly 
     * or as a base type that is extended.
     * 
     * @return A collection of ResourceMaps.
     */
    Collection<ResourceMap> getRequiredResourceMaps();
    
    /**
     * Gets the Languages.
     * 
     * Languages supported by the referenced package MAY be identified here. This list does not identify mandatory versus optional 
     * languages; it is for informational purposes only. The SDD author is not limiting use of the referenced package to deployments 
     * where all in-scope languages are found in this list. There may be cases where aggregated packages are deployed even though they 
     * cannot support all of the languages supported by the aggregation as a whole.<br><br>
     * Each language specified MUST match a language in the referenced package.
     * 
     * @return A collection of Languages supported by the referenced package.
     */
    Collection<Language> getLanguages();
    
    /**
     * Gets the referenced package ID.
     * 
     * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
     * 
     * @return Identifier for the referenced package element that is unique within the deployment descriptor.
     */
    String getReferencedPackageID();
    
    /**
     * Gets the content reference.
     * 
     * The package descriptor of an SDD that aggregates other SDDs, either through ContainedPackage elements or Requisite elements, 
     * will list the package descriptor files of the aggregated SDDs in its content list. The contentRef attribute of a referenced 
     * package element MUST be a reference to the id of a Content element in the aggregating SDDs package descriptor that defines 
     * the aggregated package descriptor.
     * 
     * @return Reference to the identifier of the package Content defined in the package descriptor which identifies the package 
     * descriptor of the referenced package.
     */
    String getContentReference();
    
    /**
     * Gets the operation.
     * 
     * The referenced SDD may support more than one deployment lifecycle operation. The operation attribute MUST include the operations that 
     * are applicable when this is the case.
     * 
     * @return Operation in the referenced SDD performed. 
     */
    String getOperation();
    
    /**
     * Gets the weight.
     * 
     * Defining weights for all artifacts and referenced packages in an SDD provides useful information to software that manages 
     * deployment. The weight of the referenced package refers to the relative time taken to deploy the referenced package with 
     * respect to other packages in this SDD. 
     * <ul>For example, if the referenced package takes twice as long to deploy as a particular install artifact whose weight is 4, 
     * then the weight of the referenced package would be 8. The weight numbers have no meaning in isolation and do not describe 
     * ]actual time elapsed. They simply provide an estimate of relative time.</ul>
     * 
     * @return The time required to process the referenced package relative to all artifacts and other referenced packages in the SDD.
     */
    BigInteger getWeight();
    
    /**
     * Sets the required group.
     * 
     * Certain Groups may need to be selected when deploying the referenced package. These can be identified in the 
     * RequiredContentSelection element.<br><br>
     * If one particular aggregated SDD requires the selection of different groups, depending on other choices made 
     * during a particular deployment, different Requisite or ContainedPackage elements can be defined in a way that will cause the 
     * correct combination of Groups to be used in each situation.
     * 
     * @param group Groups that MUST be selected when the referenced package is deployed.
     */
    void setRequiredGroup(String group);
    
    /**
     * Sets the required features.
     * 
     * Certain Features may need to be selected when deploying the referenced package. These can be identified in the 
     * RequiredContentSelection element.<br><br>
     * If one particular aggregated SDD requires the selection of different features, depending on other choices made 
     * during a particular deployment, different Requisite or ContainedPackage elements can be defined in a way that will cause the 
     * correct combination of Features to be used in each situation.
     * 
     * @param features A collection of features that MUST be selected when the referenced package is deployed.
     */
    void setRequiredFeatures(Collection<ContentSelectionFeature> features);
    
    
    /**
     * Sets the arguments.
     * 
     * Arguments are used to provide values for input variables defined in the deployment descriptor of the referenced package. The 
     * argument name specified MUST reference the id of a parameter in the referenced package.
     * 
     * @param arguments A collection of ArtifactArguments.
     */
    void setArguments(Collection<ArtifactArgument> arguments);
    
    /**
     * Sets the OutputVariables.
     * 
     * The output variable mapping can be used to set variables to outputs created by processing the referenced SDD. The output 
     * variables in the referenced package are mapped to output variables in the aggregating SDD.<br><br>
     * Each output variable value specified MUST reference the id of an output variable in the referenced package. This can be an 
     * output variable from an artifact or an output variable from a referenced package defined within the referenced SDD.
     * 
     * @param variables A collection of OutputVariables from the referenced package.
     */
    void setOutputVariables(Collection<OutputVariable> variables);
    
    /**
     * Sets the Requirements.
     * 
     * When the aggregating SDD has stricter requirements for the use of the referenced SDD than are defined by the referenced SDD 
     * itself, those requirements can be defined in Requirements. This is not intended to repeat requirements expressed in the 
     * referenced SDD, but rather to add additional stricter requirements.<br><br>
     * Requirements expressed in the referenced SDD need to be satisfied, in addition to the requirements expressed in the Requisite 
     * or ContainedPackage element of the aggregating SDD.<br><br>
     * Requirements expressed in the aggregating SDD MUST NOT conflict with requirements expressed in the referenced SDD. The 
     * requirements specified MUST further constrain the referenced package.
     * 
     * @param requirements A collection of additional requirements for deploying the referenced package as part of the aggregation.
     */
    void setRequirements(Collection<Requirement> requirements);
    
    /**
     * Sets the ResultingChangeMaps.
     * 
     * Resources configured by the referenced package may be resources that are defined in the aggregating SDD. The ResultingChangeMap 
     * is used to identify the correspondence between resource definitions in the aggregating SDD and changed resources defined in 
     * ResultingChange elements of the aggregated SDD.<br><br>
     * Characteristics of resources that are changed by the referenced SDD MAY be exposed in the ResultingChangeMap. These are 
     * correlated with ResourceConstraints on the changed resource in the same manner as the exposed characteristics of a resulting 
     * resource. See the property usage notes for ResultingResourceMap above.
     * 
     * @param resultingChangeMaps A collection of ResultingChangeMaps.
     */
    void setResultingChangeMaps(Collection<ResultingChangeMap> resultingChangeMaps);
    
    /**
     * Sets the ResultingResourceMaps.
     * 
     * Resources created by the referenced package may be resources that are defined in the aggregating SDD. The ResultingResourceMap 
     * is used to identify the correspondence between resource definitions in the aggregating SDD and resulting resource definitions 
     * in the aggregated SDD.<br><br>
     * Characteristics of the resulting resources MAY be exposed in the ResultingResourceMap element. ResourceConstraints defined on 
     * those resources anywhere in the aggregation are mapped to the resource properties exposed in the resulting maps of the referenced 
     * package to determine if the referenced package will satisfy the constraints. Each individual constraint is considered met by the 
     * referenced package if a property exposed in the resulting resource map that is in scope for the particular deployment satisfies 
     * the constraint.<br><br>
     * <ul>For example, a property constraint in a ResourceConstraint element states that the property named FileAttributes has the value 
     * Writeable. The resourceRef in the ResourceConstraint identifies a resource defined in Topology that is also identified in the 
     * ResultingResourceMap of a Requisite or ContainedPackage element that is in scope for the particular deployment. If the 
     * ResultingResourceMap element contains a statement that the property named FileAttributes has the value Writeable, then the 
     * ResourceConstraint is met when the Requisite or ContainedPackage is deployed.</ul>
     * This same logic applies to ResourceConstraints in aggregated packages. If the SDD in the preceding example also aggregates 
     * another SDD and maps the same resource to a required resource in that aggregated SDD, then all ResourceConstraints in the 
     * aggregated SDD are met only if the ResultingResourceMap of the referenced SDD that creates that resource contains a Name, 
     * Version or Property definition that satisfies the constraint. 
     * 
     * @param resultingResourceMaps A collection of ResultingResourceMap.
     */
    void setResultingResourceMaps(Collection<ResultingResourceMap> resultingResourceMaps);
    
    /**
     * Sets the ResourceMaps.
     * 
     * ResourceMapType is used in the definition of elements that map resources in an SDD to resources in a referenced SDD. 
     * The purpose of a resource map is to identify when two resources in separate SDDs MUST resolve to the same resource instance 
     * during any particular deployment. The characteristics of a mapped resource that are defined in the topology sections of the 
     * two SDDs MUST NOT conflict. 
     * For example, if a Name is defined for the resource in both topologies, it MUST be the same in both definitions and if a Property 
     * definition is included for the same property in both places, the value MUST be the same.
     * Additional characteristics of a mapped resource may be constrained by Requirements or Conditions in either SDD. All constraints 
     * on a mapped resource that are in scope for a particular deployment MUST NOT conflict.
     * Resources that are not mapped between the two SDDs MAY resolve to the same instance when their characteristics defined in topology 
     * do not conflict and when the constraints in scope for any particular deployment do not conflict.
     * The RequiredResourceMap, ResultingResourceMap and ResultingChangeMap elements all use ResourceMapType, either directly 
     * or as a base type that is extended.
     * 
     * @param requiredResourceMaps A collection of ResourceMaps.
     */
    void setRequiredResourceMaps(Collection<ResourceMap> requiredResourceMaps);
    
    /**
     * Sets the languages.
     * 
     * Languages supported by the referenced package MAY be identified here. This list does not identify mandatory versus optional 
     * languages; it is for informational purposes only. The SDD author is not limiting use of the referenced package to deployments 
     * where all in-scope languages are found in this list. There may be cases where aggregated packages are deployed even though they 
     * cannot support all of the languages supported by the aggregation as a whole.<br><br>
     * Each language specified MUST match a language in the referenced package.
     * 
     * @param languages A collection of Languages supported by the referenced package.
     */
    void setLanguages(Collection<Language> languages);
    
    /**
     * Sets the referenced package ID.
     * 
     * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
     * 
     * @param id Identifier for the referenced package element that is unique within the deployment descriptor.
     * @throws InvalidInputException if the parameter is null
     */
    void setReferencedPackageID(String id);
    
    /**
     * Sets the content reference.
     * 
     * The package descriptor of an SDD that aggregates other SDDs, either through ContainedPackage elements or Requisite elements, 
     * will list the package descriptor files of the aggregated SDDs in its content list. The contentRef attribute of a referenced 
     * package element MUST be a reference to the id of a Content element in the aggregating SDDs package descriptor that defines 
     * the aggregated package descriptor.
     * 
     * @param ref Reference to the identifier of the package Content defined in the package descriptor which identifies the package 
     * descriptor of the referenced package.
     * @throws InvalidInputException if the parameter is null
     */
    void setContentReference(String ref);
    
    /**
     * Sets the operation.
     * 
     * The referenced SDD may support more than one deployment lifecycle operation. The operation attribute MUST include the operations that 
     * are applicable when this is the case.
     * 
     * @param operation Operation in the referenced SDD performed. 
     */
    void setOperation(String operation);
    
    /**
     * Sets the weight.
     * 
     * Defining weights for all artifacts and referenced packages in an SDD provides useful information to software that manages 
     * deployment. The weight of the referenced package refers to the relative time taken to deploy the referenced package with 
     * respect to other packages in this SDD. 
     * <ul>For example, if the referenced package takes twice as long to deploy as a particular install artifact whose weight is 4, 
     * then the weight of the referenced package would be 8. The weight numbers have no meaning in isolation and do not describe 
     * ]actual time elapsed. They simply provide an estimate of relative time.</ul>
     * 
     * @param weight The time required to process the referenced package relative to all artifacts and other referenced packages 
     * in the SDD.
     */
    void setWeight(BigInteger weight);
    
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
