/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * SAS - Fix for defect 236866
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.core;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.StringTokenizer;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.SchemaFactory;

import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.PSVIProvider;
import org.apache.xerces.xs.XSModel;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.cosmos.rm.internal.validation.artifacts.DOMStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementSchemaModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.MappedNamespaceContext;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ResourceWrapper;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ValidationSet;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil.RemoteRetrievalException;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.IdentityDataBuilder;
import org.eclipse.cosmos.rm.internal.validation.util.Base64Util;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Implements a custom SAX Default Handler that will delegate ContentHandler
 * calls to IDataBuilder instances.  This class behaves differently depending
 * on the following modes it runs in: <br/>
 * 
 * <ul>
 * 	<li> {@link DataBuilderRegistry#TOP_LEVEL_MODE} </li>
 * 	<li> {@link DataBuilderRegistry#INSTANCE_LEVEL_MODE} </li>
 * </ul>
 * 
 * <p>
 * TOP_LEVEL_MODE: The input is expected to be an SML-IF document or a set
 * of SML documents.  The documents are SAX parsed using the data builders
 * registered at top level.
 * </p>
 * 
 * <p>
 * INSTANCE_LEVEL_MODE: The input is a {@link ValidationSet} consisting of
 * instance and definition documents.  The instance documents are SAX parsed
 * along with the definition documents using data builders registered at 
 * instance level.
 * </p>
 * 
 * @author Sheldon Lee-Loy
 * @author Ali Mehregani
 * @author Mark McCraw
 * @author John Arwe
 */
public class FoundationBuilder extends DefaultHandler implements IFoundationBuilder
{
	/**
	 * The SAX parser
	 */
	private SAXParser saxParser;

	/**
	 * List of IDataBuilder instances
	 */
	private IDataBuilder<?>[] dataStructureBuilders;

	/**
	 * The validation logger
	 */
	private AbstractValidationOutput validationLogger;

	/**
	 * Indicates whether the XML file should be validated
	 */
	private boolean validateXML;

	/**
	 * The locator used to indicate the column and line number of a problem
	 */
	private Locator locator;
	
	/**
	 * A flag used to indicate when a document element has been hit
	 */
	private boolean documentElementHit;
	
	/**
	 * Indicates whether the input is a set of SML unit document or an SML-IF 
	 * document.  
	 */
	private boolean smlIfInput;
	
	/**
	 * Indicates the status of this main data structure builder.  true indicates
	 * a passed status and false indicates a failed status.
	 */
	private boolean status;
	
	/**
	 * The validation listeners
	 */
	private Collection<IValidationListener> validationListeners;

	/**
	 * Set to true if the validator is running inside the Eclipse environment
	 */
	private boolean inEclipse;
	
	/**
	 * The validation attributes
	 */
	private Map<String, Object> attributes;
	
	/**
	 * The processed import statements
	 */
	private Map<String, String> processedImports;
	
	/**
	 * The imported schema files that are pending to be processed
	 */
	private Map<String, String> pendingProcess;
	
	/**
	 * The absolute path of the current resource
	 */
	private String currentResource;
	
	/**
	 * The (possibly partial) URI to a referenced remote document
	 */
	private StringBuffer remoteURI;
	
	/**
	 * The mode this foundation builder is in.  Valid value are
	 * {@link DataBuilderRegistry#INSTANCE_LEVEL_MODE} and
	 * {@link DataBuilderRegistry#TOP_LEVEL_MODE}
	 */
	private int mode;
	
	/**
	 * Set when the locator element is hit
	 */
	private boolean locatorElementHit;
	
	/**
	 * Set when document URI element is hit
	 */
	private boolean documentURIElementHit;
	
	/**
	 * The base line number 
	 */
	private int baseLineNumber;
	
	/**
	 * The set requiring validation
	 */
	private ValidationSet validationSet;
	
	/**
	 * Set to ignore syntactical error caused by
	 * schema validation
	 */
	private boolean ignoreSyntactical;

	/**
	 * The PSVI provider
	 */
	private PSVIProvider psvi;
	
	/**
	 * Indicates if it has found the XS model for the current 
	 * validation set being processed
	 */
	private boolean foundXSModel;
	
	/**
	 * String buffer for storing base64 data
	 */
	private StringBuffer base64data;
	
	/**
	 * Set when base64Data element is hit
	 */
	private boolean base64DataHit = false;
	
	
	/**
	 * The constructor
	 * 
	 * @param mode The mode this foundation builder is
	 * expected to run in.  Valid values are 
	 * {@link DataBuilderRegistry#TOP_LEVEL_MODE} and
	 * {@link DataBuilderRegistry#INSTANCE_LEVEL_MODE}
	 */	
	public FoundationBuilder(int mode) 
	{
		super();
		status = true;
		processedImports = new Hashtable<String,String>();
		pendingProcess = new Hashtable<String, String>();
		this.mode = mode;
		this.foundXSModel = false;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.core.IValidator#reset()
	 */
	public void reset()
	{
		status = true;
		processedImports.clear();
		pendingProcess.clear();
	}

	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.core.IValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init)
	{
		initialize(init, mode == DataBuilderRegistry.TOP_LEVEL_MODE);
	}

	
	public void initialize(Map<String, Object> init, boolean initBuilders)
	{
		// Initialize instance variables based on 
		// the initialization attributes
		this.validationLogger = (AbstractValidationOutput) init.get(ATTRIBUTE_OUTPUT);
		this.smlIfInput = !(init.get(IValidator.ATTRIBUTE_INPUT_TYPE) instanceof String) || !IValidator.VALUE_SML_UNIT.equals(init.get(IValidator.ATTRIBUTE_INPUT_TYPE));				
		this.inEclipse = IValidator.VALUE_ENV_ECLIPSE.equals(init.get(ATTRIBUTE_ENV));			
		this.attributes = init;		
		this.validationSet = (ValidationSet)init.get(IValidator.ATTRIBUTE_VALIDATION_SET);
		this.ignoreSyntactical = Boolean.TRUE.equals((Boolean)init.get(IValidator.ATTRIBUTE_IGNORE_SYNTACTICAL_ERRORS));
		this.psvi = (PSVIProvider)init.get(IValidator.ATTRIBUTE_PSVI_INSTANCE);
		
		// If no validation class then we should do no XML schema validation
		validateXML = true;
		createSAXParser(smlIfInput ? (String) init.get(ATTRIBUTE_SML_IF_SCHEMA) : null);
		
		// Initialize the data builders
		retrieveBuilders();
		if (initBuilders)
		{
			for (int i = 0; i < dataStructureBuilders.length; i++)
			{
				dataStructureBuilders[i].initialize(init);
			}
		}
		
		// The base line number
		Integer lineNo = (Integer)init.get(IValidator.ATTRIBUTE_BASE_LINE_NUMBER);
		baseLineNumber = lineNo == null ? -1: lineNo;
	}

	
	/**
	 * Configures the default handler and associates the schema with the SAX
	 * Parser
	 * 
	 * @param xsd file location of the schema file to validate
	 */
	protected void createSAXParser(String xsd)
	{
		try
		{
			// if there is no schema then we should not validate
			if (xsd == null)
				validateXML = false;

			InputStream schemaAsStream = (xsd == null) ? null : this.getClass().getClassLoader().getResourceAsStream(xsd);
			saxParser = SMLValidatorUtil.createSAXParser(schemaAsStream, validateXML, this, validationLogger);
		} 
		catch (ParserConfigurationException e)
		{
			fatalError(e);
		} 
		catch (SAXNotRecognizedException e)
		{			
			fatalError(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.errorUnsupportedSAXProperty1, e.getLocalizedMessage()));
		} 
		catch (SAXNotSupportedException e)
		{			
			fatalError(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.errorUnsupportedSAXProperty1, e.getLocalizedMessage()));
		} 
		catch (SAXException e)
		{
			fatalError(e);
		}
	}

	public void error(SAXParseException exception)
	{		
		if (ignoreSyntactical)
		{
			return;
		}
		
		status = false;
		int lineNo = baseLineNumber >= 0 ? baseLineNumber + exception.getLineNumber() - 1 : exception.getLineNumber();
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNo, exception.getLocalizedMessage()));
	}

	protected void fatalError(Exception exception)
	{
		fatalError(ValidationMessage.NO_LINE_NUMBER, exception.getLocalizedMessage());
	}

	protected void fatalError(int lineNumber, String message)
	{
		status = false;
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNumber, message));
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.core.IValidator#validate()
	 */
	public boolean validate()
	{
		IProgressMonitor monitor = (IProgressMonitor)attributes.get(IValidator.ATTRIBUTE_PROGRESS_MONITOR);
		if (monitor != null)
		{
			monitor.setTaskName(SMLValidationMessages.validationBuildingStructures);
		}
		
		// Depending on the mode, this process will
		// be different
		Exception exception = null;
		int lineNumber = -1;
		
		try
		{
			switch (this.mode)
			{
				case DataBuilderRegistry.TOP_LEVEL_MODE:
					return handleTopLevelMode();
				case DataBuilderRegistry.INSTANCE_LEVEL_MODE:
					return handleInstanceLevelMode();						
			}
		} 
		catch (SAXParseException e)
		{			
			lineNumber = baseLineNumber >= 0 ? baseLineNumber + e.getLineNumber() - 1 : e.getLineNumber();
			exception = e;			
		} 
		catch (Exception e)
		{			
			exception = e;			
		} 


		if (exception != null)
		{
			exception.printStackTrace();
			fatalError(lineNumber, exception.getLocalizedMessage());
		}			
		
		return false;
	}

	
	/**
	 * Invoked when this foundation builder is running
	 * in top level mode
	 * 
	 * @return The status
	 * @throws IOException 
	 * @throws SAXException 
	 * @throws FileNotFoundException 
	 * @throws ParserConfigurationException 
	 * @throws CoreException 
	 */
	@SuppressWarnings("unchecked")
	private boolean handleTopLevelMode() throws FileNotFoundException, SAXException, IOException, CoreException, ParserConfigurationException
	{		
		String currentDocument = null;
										
		// The input is an SML-IF document 
		String input = (String)attributes.get(IValidator.ATTRIBUTE_INSTANCE);
		if (smlIfInput)
		{
			currentDocument = input;
			saxParser.parse(new InputSource(new FileInputStream(new File(currentDocument))), this);
		}
		// Otherwise the input is expected to be a set of comma separated files and directories
		else
		{
			StringTokenizer inputs = new StringTokenizer(input, ISMLConstants.COMMA); //$NON-NLS-1$
			List<ResourceWrapper> instanceList = new ArrayList<ResourceWrapper>(); 					
			
			// Look up the identity
			SMLIFIdentity identity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, IdentityDataBuilder.ID);
			SMLIFIdentity metaIdentity = (SMLIFIdentity)attributes.get(IValidator.ATTRIBUTE_IDENTITY);
			if (identity != null) 
			{
				if (metaIdentity != null)
				{
					identity.copy(metaIdentity);
				}
				else
				{
					identity.setBaseURI(IValidationConstants.DUMMY_BASE_URI);
				}
			}
			
			// Send the rule binding data
			Map<String, List<String>> ruleBindings = (Map<String, List<String>>)attributes.get(IValidator.ATTRIBUTE_RULE_BINDINGS);
			if (ruleBindings != null && ruleBindings.size() > 0)
			{
				elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.RULEBINDINGS_ELEMENT, ISMLConstants.RULEBINDINGS_ELEMENT, 
						IValidationConstants.EMPTY_ATTRIBUTES, true);
				for (Iterator<String> aliases = ruleBindings.keySet().iterator(); aliases.hasNext();)
				{
					String alias =  aliases.next();
					List<String> rules = ruleBindings.get(alias);
					if (rules != null)
					{
						for (int i = 0, ruleCount = rules.size(); i < ruleCount; i++)
						{
							ruleNotification(alias, (String)rules.get(i));
						}
					}
				}
				elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.RULEBINDINGS_ELEMENT, ISMLConstants.RULEBINDINGS_ELEMENT, 
						IValidationConstants.EMPTY_ATTRIBUTES, false);
			}
			
			
			
			// Start the definition element			
			elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.DEFINITIONS_ELEMENT, ISMLConstants.DEFINITIONS_ELEMENT, 
					IValidationConstants.EMPTY_ATTRIBUTES, true);
			
			String contextRoot = (String)attributes.get(IValidator.ATTRIBUTE_REPOSITORY_ROOT);
			contextRoot = contextRoot == null ? "" : contextRoot; //$NON-NLS-1$
			
			// For every input
			while (inputs.hasMoreTokens())
			{
				currentDocument = inputs.nextToken();
				ResourceWrapper resource = retrieveResource(currentDocument);
				if (resource == null)
					return false;

				String parentFolder = findParentFolder(contextRoot, resource.getFullPath());
				traverseRecursively(instanceList, resource, true, parentFolder);
				Map<String,String> processPending = (Map<String, String>)((Hashtable)pendingProcess).clone();
				
				if (processPending == null)
				{
					continue;
				}
									
				for (Iterator<String> namespaces = processPending.keySet().iterator(); namespaces.hasNext();) 
				{
					String namespace = (String) namespaces.next();
					String path = (String)processPending.get(namespace);
					
					if (path == null || processedImports.get(namespace) != null)
					{
						continue;
					}
					
					try
					{
						traverseRecursively(instanceList, new ResourceWrapper (path, inEclipse), true, parentFolder);														
					}
					catch (MissingResourceException mre)
					{
						validationLogger.reportMessage(ValidationMessageFactory.createWarningMessage(ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.warningMissingResource, path)));							
					}						
				}
				
				processedImports.putAll(processPending);
				for (Iterator<String> namespaces = processPending.keySet().iterator(); namespaces.hasNext();)
				{
					String namespace = namespaces.next();
					processedImports.put(namespace, pendingProcess.get(namespace));
					pendingProcess.remove(namespace);
				}
			}

			// Terminate the definition element
			elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.DEFINITIONS_ELEMENT, ISMLConstants.DEFINITIONS_ELEMENT, 
					IValidationConstants.EMPTY_ATTRIBUTES, false);
			
			// Start the instance element
			elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.INSTANCES_ELEMENT, ISMLConstants.DEFINITIONS_ELEMENT, 
					IValidationConstants.EMPTY_ATTRIBUTES, true);
			
			// For every instance document
			for (int i = 0, instanceCount = instanceList.size(); i < instanceCount; i++)
			{
				ResourceWrapper currentInstanceResource = (ResourceWrapper)instanceList.get(i);
				traverseRecursively(instanceList, currentInstanceResource, false, findParentFolder(contextRoot, currentInstanceResource.getFullPath()));
			}
			
			// Terminate the instance element
			elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.INSTANCES_ELEMENT, ISMLConstants.DEFINITIONS_ELEMENT, 
					IValidationConstants.EMPTY_ATTRIBUTES, false);
		}
		
		return true;		
	}

	
	/**
	 * Invoked when this foundation builder is running
	 * in instance mode
	 * 
	 * @return status
	 * @throws SAXException 
	 * @throws ParserConfigurationException 
	 * @throws IOException 
	 */
	private boolean handleInstanceLevelMode() throws SAXException, ParserConfigurationException, IOException
	{
		// Using the validation set, construct the schemas
		// the instance documents should be validated against
		ElementSchemaModel[] definitions = validationSet.getDefinitions();					
		InputStream[] schemas = SMLValidatorUtil.toInputStream(definitions);		
		this.attributes.put(IValidator.ATTRIBUTE_NAMESPACE_CONTEXT, createNamespace(definitions));
		this.attributes.remove(IValidator.ATTRIBUTE_XS_MODEL);
		this.foundXSModel = false;
		
		boolean validateInstanceDoc = false;
		if (schemas.length != 0) {
			validateInstanceDoc = true;
		}
		SAXParser newSaxParser = SMLValidatorUtil.createSAXParser(schemas, validateInstanceDoc, null, validationLogger);
		
		// Retrieve the PSVI provider for this parser
		PSVIProvider psviProvider = (PSVIProvider)newSaxParser.getXMLReader();
		this.attributes.put(IValidator.ATTRIBUTE_PSVI_INSTANCE, psviProvider);
		
		// Register document builder as an instance builder
		DocumentDOMBuilder documentBuilder = new DocumentDOMBuilder();
		DataBuilderRegistry.getInstanceLevelRegistry().registerDataStructureBuilder(DocumentDOMBuilder.ID, documentBuilder);
		DOMStructure domDocumentStructure = documentBuilder.getDataStructure();	
			
		// For each instance documents
		ElementModel[] instances = validationSet.getInstances();
		
		// Create a dummy instance document if none exists
		if (instances.length == 0)
		{
			final String DUMMY_ELEMENT = "<dummyElement/>"; //$NON-NLS-1$
			this.attributes.put(IValidator.ATTRIBUTE_BASE_LINE_NUMBER, 0);				
			this.attributes.put(IValidator.ATTRIBUTE_DOCUMENT_ALIASES, new String[0]);
			this.attributes.put(IValidator.ATTRIBUTE_IGNORE_SYNTACTICAL_ERRORS, true);
			
			this.initialize(this.attributes, true);					
			newSaxParser.parse(new ByteArrayInputStream(DUMMY_ELEMENT.getBytes()), this);
		}
		
		for (int i = 0; i < instances.length; i++)
		{				
			// Construct the foundation builder that will be used to invoke the 
			// data builder registered at instance level.								
			this.attributes.put(IValidator.ATTRIBUTE_BASE_LINE_NUMBER, instances[i].getLineNumber());				
			this.attributes.put(IValidator.ATTRIBUTE_DOCUMENT_ALIASES, instances[i].getAliases());		
			this.attributes.put(IValidator.ATTRIBUTE_IGNORE_SYNTACTICAL_ERRORS, false);
			this.initialize(this.attributes, true);
			
			// Parse the current instance document			
			InputStream source = new ByteArrayInputStream(instances[i].getSource().getBytes());
			newSaxParser.parse(source, this);
			
			// Build the DOM representation of the instance document
			Node documentNode = buildDOMDocument (schemas, source); 
			domDocumentStructure.add(documentNode, instances[i]);
		}

		
		return true;
	}


	/**
	 * Combines the namespace context of all the definition
	 * documents passed in
	 * 
	 * @param definitions The definition documents to be used to construct the
	 * namespace context.
	 * @return The namespace context of all definition documents passed in
	 */
	private MappedNamespaceContext createNamespace(ElementSchemaModel[] definitions)
	{
		MappedNamespaceContext namespaceContext = new MappedNamespaceContext();
		for (int i = 0; i < definitions.length; i++)
		{
			namespaceContext.addEntries(definitions[i].getNamespaceContext());
		}
		
		return namespaceContext;
	}

	

	
	
	/**
	 * Builds a DOM representation at of the instance document
	 * passed in.  The DOM representation is PSVI aware.
	 * 
	 * @param schemas The schemas that are bound to instance
	 * @param instance The instance document
	 * @return A DOM node representation of the instance document
	 */
	private Node buildDOMDocument(InputStream[] schemas, InputStream instance)
	{
		try
		{
			// Reset the input stream
			for (int i = 0; i < schemas.length; i++)
			{
				schemas[i].reset();
			}
			instance.reset();
			
			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
		    docBuilderFactory.setNamespaceAware(true);
		    docBuilderFactory.setAttribute(ISMLConstants.DOM_PARSER_PROPERTY_CLASS_NAME, ISMLConstants.DOM_PARSER_VALUE_CLASS_NAME);
		    
		    Source[] schemaSource = new Source[schemas.length];
		    for (int i = 0; i < schemaSource.length; i++)
			{
				schemaSource[i] = new StreamSource(schemas[i]);
			}
		    
		    if (schemas.length > 0) {
			    docBuilderFactory.setSchema(SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI).newSchema(schemaSource));	    
		    }
		    
		    javax.xml.parsers.DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
		    Document document = docBuilder.parse(instance);
		    if (document == null)
		    {
		    	return null;
		    }
		    
		    // Find the first element node
		    NodeList children = document.getChildNodes();
		    for (int i = 0, childCount = children.getLength(); i < childCount; i++)
			{
		    	Node child = children.item(i);
		    	if (Node.ELEMENT_NODE == child.getNodeType())
		    	{
		    		return child;
		    	}
			}
		    
			return null;
		}
		catch (Exception e)
		{
			e.printStackTrace();
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(
							ValidationMessage.NO_LINE_NUMBER, 
							SMLValidationMessages.errorDOMDocument + " " + e.getMessage())); //$NON-NLS-1$
					
			return null;
		}
	}
	

	private void ruleNotification(String alias, String ruleAlias) throws SAXException
	{
		fireNotifications(ISMLConstants.SMLIF_URI, new String[]{
				ISMLConstants.RULEBINDING_ELEMENT,
				ISMLConstants.DOCUMENTALIAS_ELEMENT
			}, true);		
		characters(alias.toCharArray(), 0, alias.length());		
		elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.DOCUMENTALIAS_ELEMENT, ISMLConstants.DOCUMENTALIAS_ELEMENT, 
				IValidationConstants.EMPTY_ATTRIBUTES, false);				
		elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.RULEALIAS_ELEMENT, ISMLConstants.DOCUMENTALIAS_ELEMENT, 
				IValidationConstants.EMPTY_ATTRIBUTES, true);
		characters(ruleAlias.toCharArray(), 0, ruleAlias.length());		
		fireNotifications(ISMLConstants.SMLIF_URI, new String[]{
				ISMLConstants.RULEALIAS_ELEMENT,
				ISMLConstants.RULEBINDING_ELEMENT
			}, false);	
	}

	/**
	 * Given a root path and a resource path, the following method will determine the
	 * resource path minus the intersection and the filename: <br/>
	 *                
	 *                ***
	 * |_____________|___|____________|
	 *  -------root-------
	 *               -----resource----
	 * 
	 * <br/>
	 * The intersection is marked by '***'              
	 * @param root The root path
	 * @param resource The resource path
	 * @return The resource path minus the intersection and the file name
	 */
	private String findParentFolder(String root, String resource)
	{
		// Make sure that both paths use forward slashes
		root = root.replace('\\', '/');
		resource = resource.replace('\\', '/');
		
		IPath rootPath = new Path(root);
		IPath resourcePath = new Path(resource);
		
		if (inEclipse)
		{
			// Find the work-space directory and exclude it from the root path
			IPath workspace = ResourcesPlugin.getWorkspace().getRoot().getLocation();			
			rootPath = rootPath.removeFirstSegments(workspace.segmentCount());			
		}
		
		for (int i = 0, segCount = rootPath.segmentCount(); i < segCount; i++)
		{
			if (resourcePath.segment(0).equals(rootPath.segment(i)))
			{
				resourcePath = resourcePath.removeFirstSegments(1);
			}
			else
			{
				break;
			}
		}
		
		String parentFolder = resourcePath.removeLastSegments(1).toString();
		String device = resourcePath.getDevice();
		if (device != null && parentFolder.startsWith(device))
		{
			parentFolder = parentFolder.substring(device.length());
		}
		
		return parentFolder;
	}

	private void traverseRecursively(List<ResourceWrapper> instanceList, ResourceWrapper resource, boolean isDefinition, String parentFolder) throws CoreException, IOException, SAXException, ParserConfigurationException
	{
		if (resource.isDirectory())
		{
			ResourceWrapper[] children = resource.getChildren();
			parentFolder = (parentFolder.length() > 0 ? parentFolder + "/" : IValidationConstants.EMPTY_STRING) + resource.getName(); //$NON-NLS-1$
			for (int i = 0; i < children.length; i++)
			{
				traverseRecursively(instanceList, children[i], isDefinition, parentFolder);
			}			
		}
		else if (resource.isFile())
		{	
			int type = SMLValidatorUtil.identifyDocumentType(resource);
			currentResource = resource.getAbsolutePath();
			
			switch (type)
			{
				// definition
				case ISMLConstants.TYPE_DEFINITION:
					addDocument (parentFolder, resource);
					break;
				
				// instance 
				case ISMLConstants.TYPE_INSTANCE:
					if (isDefinition)
					{
						instanceList.add(resource);
						return;
					}
					addDocument (parentFolder, resource);					
					break;
				default:
					break;
			}
		}
	}

	private void addDocument(String parentFolder, ResourceWrapper resource) throws SAXException, ParserConfigurationException, IOException, CoreException
	{
		// Add the aliases
		addAliases(parentFolder, resource);
		
		// Add the data	 
		processData(resource);
		
		elementNotification(ISMLConstants.SMLIF_URI, 
				ISMLConstants.DOCUMENT_ELEMENT, 
				ISMLConstants.DOCUMENT_ELEMENT, 
				IValidationConstants.EMPTY_ATTRIBUTES, 
				false);
	}

	@SuppressWarnings("unchecked")
	private void addAliases(String parentFolder, ResourceWrapper resource) throws SAXException
	{
		// The document and the doc info section
		fireNotifications(ISMLConstants.SMLIF_URI, new String[]{
				ISMLConstants.DOCUMENT_ELEMENT,
				ISMLConstants.DOC_INFO_ELEMENT, 
				ISMLConstants.ALIASES_ELEMENT							
			}, true);
		
		Map<String, List<String>> fileAliasMap = (Map<String, List<String>>)attributes.get(IValidator.ATTRIBUTE_ALIASES);
		boolean definedAliases = false;
		if (fileAliasMap != null)
		{
			List<String> aliases = fileAliasMap.get(resource.getName());						
			if (aliases != null)
			{	
				definedAliases = aliases.size() > 0;
				for (int i = 0, aliasCount = aliases.size(); i < aliasCount; i++)
				{
					String alias = aliases.get(i);
					notifyAlias(alias);							
				}
			}
		}
		
		// If there are no aliases defined in the meta-data, then use the
		// path as the alias
		if (!definedAliases)
		{
			notifyAlias(((parentFolder.length() > 0 ? parentFolder + "/" : IValidationConstants.DUMMY_BASE_URI) + resource.getName())); //$NON-NLS-1$
		}
							
		fireNotifications(ISMLConstants.SMLIF_URI, new String[]{						
				ISMLConstants.ALIASES_ELEMENT,						
				ISMLConstants.DOC_INFO_ELEMENT					
			}, false);
	}

	private void notifyAlias(String alias) throws SAXException
	{
		elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.ALIAS_ELEMENT, 
				ISMLConstants.ALIAS_ELEMENT, IValidationConstants.EMPTY_ATTRIBUTES, true);
		characters(alias.toCharArray(), 0, alias.length());
		elementNotification(ISMLConstants.SMLIF_URI, ISMLConstants.ALIAS_ELEMENT, 
				ISMLConstants.ALIAS_ELEMENT, IValidationConstants.EMPTY_ATTRIBUTES, false);
	}

	private void processData(ResourceWrapper resource) throws SAXException, ParserConfigurationException, IOException, CoreException
	{
		InputStream fileInputStream = resource.getContents();
		try
		{
			elementNotification(ISMLConstants.SMLIF_URI, 
					ISMLConstants.DATA_ELEMENT, 
					ISMLConstants.DATA_ELEMENT, 
					IValidationConstants.EMPTY_ATTRIBUTES, true);
			
			validationLogger.setAssociatedResource(resource);
			for (int i = 0; i < dataStructureBuilders.length; i++)
			{
				IDataBuilder<?> dataBuilder = dataStructureBuilders[i];
				dataBuilder.setFilePath(resource.getFullPath());
			}
			
			saxParser.parse(fileInputStream, this);			
			elementNotification(ISMLConstants.SMLIF_URI, 
					ISMLConstants.DATA_ELEMENT, 
					ISMLConstants.DATA_ELEMENT, 
					IValidationConstants.EMPTY_ATTRIBUTES, false);
		}
		finally
		{
			if (fileInputStream != null)
				fileInputStream.close();
		}
	}

	private void fireNotifications(String uri, String[] elements, boolean open) throws SAXException
	{
		for (int i = 0; i <elements.length; i++)
		{
			elementNotification(uri, elements[i], elements[i], IValidationConstants.EMPTY_ATTRIBUTES, open);
		}
	}

	private ResourceWrapper retrieveResource(String path)
	{
		try
		{
			return new ResourceWrapper (path, inEclipse);
		}
		catch (MissingResourceException mre)
		{
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(ValidationMessage.NO_LINE_NUMBER, mre.getMessage()));
			return null;			
		}
	}
	
	
	/**
	 * Retrieves the data builders registered.  The data builders
	 * are retrieved based on the mode this foundation builder
	 * has been instantiated with.
	 */
	private void retrieveBuilders()
	{		
		DataBuilderRegistry registry = null;
		switch (this.mode)
		{
			case DataBuilderRegistry.TOP_LEVEL_MODE:
				registry = DataBuilderRegistry.getTopLevelRegistry();
				break;
			case DataBuilderRegistry.INSTANCE_LEVEL_MODE:
				registry = DataBuilderRegistry.getInstanceLevelRegistry();
				break;
			default:
				return;
		}
		
		Collection<IDataBuilder<?>> dataBuilders = registry.getDataStructureBuilders();
		dataStructureBuilders = new IDataBuilder[dataBuilders.size()];
		dataBuilders.toArray(dataStructureBuilders);
	}
	

	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{		
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{	
			if (ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = true;
			}
			else if (documentElementHit && ISMLConstants.LOCATOR_ELEMENT.equals(localName))
			{
				locatorElementHit = true;
			}			
			else if (locatorElementHit && ISMLConstants.DOCUMENTURI_ELEMENT.equals(localName)) 
			{
				remoteURI = new StringBuffer();
				documentURIElementHit = true;
			}
			else if (documentElementHit && ISMLConstants.BASE64DATA_ELEMENT.equals(localName))
			{
				base64data = new StringBuffer();
				base64DataHit = true;
			}
		}
		else if (ISMLConstants.SCHEMA_URI.equals(uri))
		{
			if (ISMLConstants.IMPORT_ELEMENT.equals(localName))
			{
				String namespace = attributes.getValue(ISMLConstants.NAMESPACE_ATTRIBUTE);
				String location = attributes.getValue(ISMLConstants.SCHEMA_LOCATION_ATTRIBUTE);
				
				if (namespace != null && location != null && currentResource != null 
					&& processedImports.get(namespace) == null && location.indexOf("//") < 0) //$NON-NLS-1$
				{
					int inx = Math.max(currentResource.lastIndexOf('/'), currentResource.lastIndexOf('\\')); 
					String parentFolder = inx >= 0 ? currentResource.substring(0, inx) : currentResource;
					location = location.startsWith(parentFolder) ? location : parentFolder + "/" + location; //$NON-NLS-1$
					pendingProcess.put(namespace, location);					
				}
			}
		}
		
		// build data structures
		elementNotification (uri, localName, qName, attributes, true);
	}

	private void retrieveRemoteDocument(String href) throws SAXException
	{
		Node node = null;
		Exception exception = null;
		try
		{
			node = SMLValidatorUtil.retrieveRemoteDocument(href);			
			attributes.put(IValidator.ATTRIBUTE_REMOTE_DOCUMENT, node);
			Node dataNode = node.getOwnerDocument().createElementNS(ISMLConstants.SMLIF_URI, ISMLConstants.DATA_ELEMENT);
			dataNode.appendChild(node);
			node = dataNode;						
		} 
		catch (RemoteRetrievalException e)
		{
			exception = e;
		}
		
		if (exception != null || node == null)
		{
			int lineNo = baseLineNumber >= 0 ? baseLineNumber + locator.getLineNumber() - 1 : locator.getLineNumber();
			String theCause = ""; // default //$NON-NLS-1$
			if (exception != null) {
				String eCause = exception.getMessage(); // MAY be null
				if (eCause == null) {
					theCause = SMLValidationMessages.messageNoCauseProvidedInException;
				} else {
					theCause = eCause;
				}
			}
			String messageParameters[] = {theCause}; // removed duplicate line number, since it is always provided on the logger call
			String message = NLS.bind(SMLValidationMessages.warningMissingDocument, messageParameters); 						
			validationLogger.reportMessage(ValidationMessageFactory.createWarningMessage(lineNo, message));						
			return;
		}
		walkDocument(node);
		attributes.remove(IValidator.ATTRIBUTE_REMOTE_DOCUMENT);
	}

	private void walkDocument(Node node) 
	{
		Attributes atts = toAttributes(node.getAttributes());
		
		try 
		{
			domNotification(node, atts, true);
		} 
		catch (SAXException e) 
		{			
			e.printStackTrace();
		}
		
		NodeList children = node.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			walkDocument (children.item(i));			
		}
		
		try 
		{
			domNotification(node, atts, false);
		} 
		catch (SAXException e) 
		{		
			e.printStackTrace();
		}
	}

	private void domNotification(Node node, Attributes atts, boolean start) throws SAXException
	{
		switch (node.getNodeType())
		{
			case Node.ELEMENT_NODE:
				elementNotification(node.getNamespaceURI(), node.getLocalName(), node.getNodeName(), 
						atts, start);
				break;
			case Node.TEXT_NODE:
			case Node.COMMENT_NODE:
				if (!start)
					break;
				String value = node.getNodeValue();
				if (value == null)
					break;				
				characters(value.toCharArray(), 0, value.length());
				break;
			default:
				break;
		}
	}

	private Attributes toAttributes(final NamedNodeMap attributes)
	{				
		AttributeWrapper attributeWrapper = new AttributeWrapper (attributes);
		return attributeWrapper;
	}

	public void endElement(String uri, String localName, String qName) throws SAXException
	{
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{		
			if (ISMLConstants.DOCUMENT_ELEMENT.equals(localName))
			{
				documentElementHit = false;				
			}
			else if (ISMLConstants.DOCUMENTURI_ELEMENT.equals(localName))
			{
				retrieveRemoteDocument(remoteURI.toString().trim()); // trim any whitespace around the URI
				documentURIElementHit = false;
			}
			else if (ISMLConstants.BASE64DATA_ELEMENT.equals(localName))
			{
				processBase64Data(base64data.toString());
				base64DataHit = false;
			}
			
		}
		
		if (!foundXSModel && psvi != null)
		{
			ElementPSVI elementPSVI = psvi.getElementPSVI();
			XSModel xsModel = elementPSVI == null ? null : elementPSVI.getSchemaInformation();
			if (xsModel != null)
			{
				foundXSModel = true;
				attributes.put(IValidator.ATTRIBUTE_XS_MODEL, xsModel);
			}
		}
		
		elementNotification (uri, localName, qName, null, false);
	}
	
	/**
	 * Decode and validate document encoded in base64
	 */
	private void processBase64Data(String base64data)
	{
		String decodedString = null;
		// decode base64 data
		try {
			base64data = Base64Util.removeWhiteSpaces(base64data);
			byte[] decodedByteArray = Base64Util.decode(base64data.getBytes());
			decodedString = new String(decodedByteArray);
		} catch (Exception e) {
			int lineNo = baseLineNumber >= 0 ? baseLineNumber + locator.getLineNumber() - 1 : locator.getLineNumber();
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNo, e.getMessage()));
			return;
		}
		// parse document
		if (decodedString != null) {
			try {
				// The SAX parser created here is not validating because we don't need to 
				// validate SML-IF syntax in the instance document and the instance document
				// will be validated against its associated schema and DTD in the instance
				// level validation.
				attributes.put(IValidator.ATTRIBUTE_DECODED_DATA, decodedString);
				SAXParser newSaxParser = SMLValidatorUtil.createSAXParser(null, false, null, validationLogger);
				Locator locator = this.locator;
				newSaxParser.parse(new InputSource(new StringReader(decodedString)), this);
				setDocumentLocator(locator);
				attributes.remove(IValidator.ATTRIBUTE_DECODED_DATA);
			} catch (Exception e) {
				// IOException, SAXException, ParserConfigurationException
				int lineNo = baseLineNumber >= 0 ? baseLineNumber + locator.getLineNumber() - 1 : locator.getLineNumber();
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(lineNo, e.getMessage()));
			}
		}
	}
	
	
	private void elementNotification (String uri, String localName, String qName, 
									  Attributes attributes, boolean start) throws SAXException
	{
		for (int i = 0; i < dataStructureBuilders.length; i++)
		{
			IDataBuilder<?> dataBuilder = dataStructureBuilders[i];
			
			if (start) 
			{
				dataBuilder.startElement(uri, localName, qName, attributes);
			} 
			else 
			{
				dataBuilder.endElement(uri, localName, qName);
			}			
		}
	}

	public void characters(char[] ch, int start, int len) throws SAXException
	{
		eventNotification (EVENT_CHARACTER, ch, new int[]{start, len}, null);
		if (documentURIElementHit) 
		{
			remoteURI.append(new String(ch).substring(start, start + len));
		}

		else if (base64DataHit)
		{
			base64data.append(new String(ch).substring(start, start+len));
		}
	}

	public void endDocument() throws SAXException
	{
		eventNotification (EVENT_END_DOCUMENT, null, null, null);
	}

	public void endPrefixMapping(String arg0) throws SAXException
	{
		eventNotification (EVENT_END_PREFIX, null, null, new String[]{arg0});
	}

	public void unparsedEntityDecl(String arg0, String arg1, String arg2, String arg3) throws SAXException
	{
		super.unparsedEntityDecl(arg0, arg1, arg2, arg3);
	}

	public void ignorableWhitespace(char[] arg0, int arg1, int arg2) throws SAXException
	{
		eventNotification (EVENT_IGNORABLE_WHITESPACE, arg0, new int[]{arg1, arg2}, null);
	}

	public void processingInstruction(String arg0, String arg1) throws SAXException
	{
		eventNotification (EVENT_PROCESSING_INSTRUCTIONS, null, null, new String[]{arg0, arg1});
	}

	public void skippedEntity(String arg0) throws SAXException
	{
		eventNotification (EVENT_SKIPPED_ENTITY, null, null, new String[]{arg0});
	}

	public void startDocument() throws SAXException
	{
		eventNotification (EVENT_START_DOCUMENT, null, null, null);
	}

	public void startPrefixMapping(String arg0, String arg1) throws SAXException
	{
		eventNotification(EVENT_START_PREFIX, null, null, new String[]{arg0, arg1});
	}
	
	public void setDocumentLocator(Locator locator)
	{
		this.locator = locator;
		for (int i = 0; i < dataStructureBuilders.length; i++)
		{
			dataStructureBuilders[i].setDocumentLocator(locator);
		}
	}
	
	private void eventNotification (int event, char[] characters, int[] integerParameters, String[] stringParameters) throws SAXException
	{
		for (int i = 0; i < dataStructureBuilders.length; i++)
		{
			if (isEventSupported(dataStructureBuilders[i], event))
			{
				handleEvent (dataStructureBuilders[i], event, characters, integerParameters, stringParameters);				
			}
		}		
	}
	
	
	public void comment(char[] ch, int start, int len) throws SAXException 
	{
		eventNotification (EVENT_COMMENT, ch, new int[]{start, len}, null);
	}
	
	
	/**
	 * Returns true if dataBuilder supports the event passed
	 * in.
	 * 
	 * @param dataBuilder The data builder
	 * @param event The event to be checked
	 * @return true if dataBuilder supports event; false otherwise
	 */
	private boolean isEventSupported(IDataBuilder<?> dataBuilder, int event)
	{
		int supportedEvents = dataBuilder.getSupportedEvents();
		return (supportedEvents & (1 << event)) > 0;
	}

	private void handleEvent (ContentHandler contentHandler, int event, char[] characters, int[] integerParameters, String[] stringParameters) throws SAXException
	{
		switch (event)
		{
			case EVENT_COMMENT:
				((IDataBuilder<?>)contentHandler).comment(characters, integerParameters[0], integerParameters[1]);
				break;
			case EVENT_CHARACTER:
				contentHandler.characters(characters, integerParameters[0], integerParameters[1]);
				break;
			case EVENT_START_DOCUMENT:
				contentHandler.startDocument();
				break;
			case EVENT_END_DOCUMENT:
				contentHandler.endDocument();
				break;
			case EVENT_START_PREFIX:
				contentHandler.startPrefixMapping(stringParameters[0], stringParameters[1]);
				break;
			case EVENT_END_PREFIX:
				contentHandler.endPrefixMapping(stringParameters[0]);
				break;
			case EVENT_IGNORABLE_WHITESPACE:
				contentHandler.ignorableWhitespace(characters, integerParameters[0], integerParameters[1]);
				break;
			case EVENT_SKIPPED_ENTITY:
				contentHandler.skippedEntity(stringParameters[0]);
				break;
			case EVENT_PROCESSING_INSTRUCTIONS:
				contentHandler.processingInstruction(stringParameters[0], stringParameters[1]);
				break;
			default:
				break;
		}
	}
	
	
	private class AttributeWrapper implements Attributes
	{
		private NamedNodeMap attributes;
		private Map<String, Integer> qNameInx;
		private Map<String, Integer> uriInx;
		
		public AttributeWrapper(NamedNodeMap attributes)
		{
			this.attributes = attributes;
			qNameInx = new Hashtable<String, Integer>();
			uriInx = new Hashtable<String, Integer>();
			
			if (attributes == null)
				return;
			for (int i = 0, attributeCount = attributes.getLength(); i < attributeCount; i++)
			{
				Node att = attributes.item(i);
				Integer inx = new Integer(i);
				qNameInx.put(att.getNodeName(), inx);
				String key = att.getNamespaceURI() == null ? IValidationConstants.EMPTY_STRING : att.getNamespaceURI();
				key += att.getLocalName();
				uriInx.put(key, inx);
			}
		}
		
		public int getIndex(String qName)
		{			
			Integer inx = (Integer)qNameInx.get(qName);
			return inx == null ? -1 : inx.intValue();
		}

		public int getIndex(String uri, String localName)
		{				
			Integer inx = (Integer)uriInx.get(uri == null ? localName : uri+localName);
			return inx == null ? -1 : inx.intValue();
		}

		public int getLength()
		{
			return attributes.getLength();
		}

		public String getLocalName(int index)
		{
			return attributes.item(index).getLocalName();
		}

		public String getQName(int index)
		{				
			return attributes.item(index).getNodeName();
		}

		public String getType(int index)
		{
			return null;
		}

		public String getType(String qName)
		{
			return null;
		}

		public String getType(String uri, String localName)
		{				
			return null;
		}

		public String getURI(int index)
		{
			return attributes.item(index).getNamespaceURI();
		}

		public String getValue(int index)
		{
			Node node = index >= 0 && index < attributes.getLength() ? attributes.item(index) : null;
			return node == null ? null : node.getNodeValue();
		}

		public String getValue(String qName)
		{
			Node node = attributes.getNamedItem(qName);
			return node == null ? null : node.getNodeValue();
		}

		public String getValue(String uri, String localName)
		{
			Node node = attributes.getNamedItemNS(uri, localName); 
			return node == null ? null : node.getNodeValue();
		}
		
	}
	
	public boolean getStatus()
	{
		return status;
	}

	public void addValidationListener(IValidationListener listener) {
		if (listener == null) {
			throw new NullPointerException();
		}
		if (validationListeners == null) {
			validationListeners = new ArrayList<IValidationListener>();
		}
		if (!validationListeners.contains(listener)) {
			validationListeners.add(listener);
		}
	}

	public void endCDATA() throws SAXException {
		// do nothing
	}

	public void endDTD() throws SAXException {
		// do nothing
	}

	public void endEntity(String arg0) throws SAXException {
		// do nothing
	}

	public void startCDATA() throws SAXException {
		// do nothing
	}

	public void startDTD(String arg0, String arg1, String arg2) throws SAXException {
		// do nothing
	}

	public void startEntity(String arg0) throws SAXException {
		// do nothing
	}


	public Map<String, Object> getAttributes() {
		return attributes;
	}
}
