/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.RelationshipConstraint;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.UniquenessConstraint;

/**
 * ConditionalResourceConstraintType provides the type definitions for the ResourceConstraint elements used in conditions. These 
 * constraints do not represent requirements for deployment. They identify the resource characteristics associated with a condition. 
 * Name, version, property and the existence or absence of the resource can be specified with a resource constraint used in a condition.
 */
public interface ConditionalResourceConstraint extends DisplayNameable {
	
    /**
     * Gets the name.
     * 
     * The name of the resource identified by resourceRef. If the resource name is defined in topology it SHOULD NOT be defined here. 
     * If it is defined in both places, the one defined in the condition is used when evaluating the condition.
     * 
     * @return Name of the resource constraint.
     */
    String getName();
    
    /**
     * Gets the constraint ID.
     * 
     * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
     * 
     * @return Identifier for the resource constraint that is unique within the deployment descriptor.
     */
    String getConstraintID();

	/**
	 * Gets the resource reference.
	 * 
	 * The version and property constraints defined here all apply to the one resource specification in topology identified by this 
	 * attribute.<br><br>
	 * The value MUST reference the id of that resource element in Topology.
	 * 
	 * @return The resource to which the conditions apply.
	 */
	String getResourceReference();
	
	/**
	 * Gets the test value.
	 * 
	 * When the result of evaluating Name and all of the constraints defined in the ResourceConstraint matches the value of testValue, 
	 * the ResourceConstraint is considered met.<br><br>
	 * When no name, version or property constraints are defined, and testValue is true, the constraint is met if the resource exists 
	 * as defined in topology.<br><br>
	 * When no name, version or property constraints are defined, and testValue is false, the constraint is met if the resource, as 
	 * defined in topology, does not exist.
	 * 
	 * @return The result of evaluating the contained constraints, which will result in the ResourceConstraint being met.
	 */
	Boolean getTestValue();
	
	/**
	 * Gets the VersionRanges.
	 * 
	 * A VersionRange is specified with a minimum and maximum version value and a list of required fixes associated with that range. 
	 * The method of comparing version strings in a version range is resource-specific.
	 *  
	 * @return A collection of VersionRanges with associated fixes.
	 */
	Collection<VersionRange> getVersionRanges();
	
	/**
	 * Gets the VersionValues.
	 * 
	 * Discrete version values can be defined when the set of required versions includes versions that do not fall within a range. 
	 * There is no assumption by this specification that version values are numerically comparable. The method of comparing version 
	 * values may be resource-specific. 
	 * 
	 * @return A collection of VersionValues with associated fixes.
	 */
	Collection<VersionValue> getVersionValues();
	
	/**
	 * Gets the single value conditional property constraints.
	 * 
	 * The actual value of the property MUST match the value defined here for the condition to be considered met. 
	 * 
	 * @return A collection of ConditionalPropertyConstraint objects. 
	 */
	Collection<ConditionalPropertyConstraint> getConditionalSingleValuePropertyConstraints();
	
	/**
	 * Gets the multiple value conditional property constraints.
	 * 
	 * The actual value of the property MUST match the value defined here for the condition to be considered met. 
	 * 
	 * @return A collection of ConditionalPropertyConstraint objects. 
	 */
	Collection<ConditionalPropertyConstraint> getConditionalMultiValuePropertyConstraints();
	
	/**
	 * Gets the UniquenessConstraints.
	 * 
	 * UniquenessConstraint elements are used in ResourceConstraints to indicate when two resources defined in topology MUST or 
	 * MUST NOT resolve to the same resource instance during a particular deployment.
	 * 
	 * @return A collection of UniquenessConstraint objects.
	 */
	Collection<UniquenessConstraint> getUniquenessConstraints();
	
	/**
	 * Gets the RelationshipConstraints.
	 * 
	 * RelationshipConstraint elements are used in ResourceConstraints to indicate a constraint on a particular relationship 
	 * between resources.
	 * 
	 * @return A collection of RelationshipConstraint objects.
	 */
	Collection<RelationshipConstraint> getRelationshipConstraints();
	
	/**
	 * Sets the name.
	 * 
	 * The name of the resource identified by resourceRef. If the resource name is defined in topology it SHOULD NOT be defined here. 
     * If it is defined in both places, the one defined in the condition is used when evaluating the condition.
     * 
	 * @param name Name of the resource constraint.
	 */
	void setName(String name);
	
	/**
	 * Sets the constraint ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @param id Identifier for the resource constraint that is unique within the deployment descriptor.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setConstraintID(String id);
	
	/**
	 * Sets the resource reference.
	 * 
	 * * The version and property constraints defined here all apply to the one resource specification in topology identified by this 
	 * attribute.<br><br>
	 * The value MUST reference the id of that resource element in Topology.
	 * 
	 * @param resourceReference The resource to which the conditions apply.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setResourceReference(String resourceReference);
	
	/**
	 * Sets the test value.
	 * 
	 * When the result of evaluating Name and all of the constraints defined in the ResourceConstraint matches the value of testValue, 
	 * the ResourceConstraint is considered met.<br><br>
	 * When no name, version or property constraints are defined, and testValue is true, the constraint is met if the resource exists 
	 * as defined in topology.<br><br>
	 * When no name, version or property constraints are defined, and testValue is false, the constraint is met if the resource, as 
	 * defined in topology, does not exist.
	 * 
	 * @param testValue The result of evaluating the contained constraints, which will result in the ResourceConstraint being met.
	 * <ul>**default value=true</ul>
	 * 
	 */
	void setTestValue(Boolean testValue);

	/**
	 * Sets the VersionRanges.
	 * 
	 * A VersionRange is specified with a minimum and maximum version value and a list of required fixes associated with that range. 
	 * The method of comparing version strings in a version range is resource-specific.
	 *  
	 * @param versionRanges A collection of VersionRanges with associated fixes. 
	 */
	void setVersionRanges(Collection<VersionRange> versionRanges);
	
	/**
	 * Sets the VersionValues.
	 * 
	 * Discrete version values can be defined when the set of required versions includes versions that do not fall within a range. 
	 * There is no assumption by this specification that version values are numerically comparable. The method of comparing version 
	 * values may be resource-specific. 
	 * 
	 * @param versionValues A collection of VersionValues with associated fixes.
	 */
	void setVersionValues(Collection<VersionValue> versionValues);
	
	/**
	 * Sets the single value conditional property constraints.
	 * 
	 * The actual value of the property MUST match the value defined here for the condition to be considered met. 
	 * 
	 * @param propertyConstraints A resource property name and required value. 
	 * 
	 */
	void setConditionalSingleValuePropertyConstraints(
	        Collection<ConditionalPropertyConstraint> propertyConstraints);
	
	/**
	 * Sets the multiple value conditional property constraints.
	 * 
	 * The actual value of the property MUST match the value defined here for the condition to be considered met.  
	 * 
	 * @param propertyConstraints A resource property name and a list of required values.
	 */
	void setConditionalMultiValuePropertyConstraints(
	        Collection<ConditionalPropertyConstraint> propertyConstraints);
	
	/**
	 * Sets the UniquenessConstraints.
	 * 
	 * UniquenessConstraint elements are used in ResourceConstraints to indicate when two resources defined in topology MUST or 
	 * MUST NOT resolve to the same resource instance during a particular deployment.
	 * 
	 * @param uniquenessConstraints A collection of UniquenessConstraint objects.
	 */
	void setUniquenessConstraints(Collection<UniquenessConstraint> uniquenessConstraints);
	
	/**
	 * Sets the RelationshipConstraints.
	 * 
	 * RelationshipConstraint elements are used in ResourceConstraints to indicate a constraint on a particular relationship 
	 * between resources.
	 * 
	 * @param relationshipConstraints A collection of RelationshipConstraint objects. 
	 * resource in the topology.
	 */
	void setRelationshipConstraints(Collection<RelationshipConstraint> relationshipConstraints);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}
