/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.DerivedVariable;


/**
 * NestedFeatureType is identical to FeatureType except that NestedFeatureType does not define a required attribute. All 
 * features other than those defined directly below SelectableContent use the NestedFeatureType.
 *
 */
public interface NestedFeature extends DisplayNameable {
	
    /**
     * Gets the condition.
     * 
     * If the features and its nested features are only applicable in certain environments, a Condition can be defined. 
     * When the Condition is not met, the feature and its nested features are not in scope. 
     * <ul>For example, some features may be available only on a Linux operating system, even though the software can be 
     * applied on other operating systems. In this case, a Condition can be defined to cause the feature to be ignored 
     * when the operating system is not Linux.</ul>
     * 
     * @return A condition that determines if the feature is relevant to a particular deployment.
     */
    Condition getCondition();
    
    /**
     * Gets the multiplicity.
     * 
     * When multiple instances of a feature can be selected, a Multiplicity element MUST be defined. 
     * <ul>For example, a solution that includes a server and a client may allow the deployment of multiple clients. 
     * In this situation, a feature that defines a Multiplicity element would select the content elements that deploy 
     * the client software.</ul>
     * 
     * @return Both an indication that multiple instances of the feature can be selected and the specification of their constraints.
     */
    Multiplicity getMultiplicity();
    
    /**
     * Gets the mandatory languages.
     * 
     * The deployer has no ability to determine if a mandatory language will be deployed.
     * 
     * @return The collection of languages that will be deployed.
     */
    Collection<Language> getMandatoryLanguages();
    
    /**
     * Gets the optional languages.
     * 
     * Each language group in the list of optional languages defines a list of one or more languages that can be selected together.<br><br>
     * Language groups defined in LanguageSelections MAY be used to allow the deployer to select individual languages or to allow 
     * selection of multiple languages as a single choice.
     * 
     * @return The collection of language selections available to the deployer.
     */
    Collection<Language> getOptionalLanguages();
    
    /**
     * Gets the optional language sets.
     * 
     * When the SDD author allows the deployer to select languages for deployment as a set, it is defined in a LanguageSet element 
     * within OptionalLanguages.
     * <ul>One example of a reason to define optional languages in a set rather than individually is for a group of languages that 
     * are packaged together and whose deployment cannot be separated.</ul>
     * 
     * @return The collection of languages that can be chosen together.
     */
    Collection<LanguageSet> getOptionalLanguageSets();
    
    /**
     * Gets the nested features.
     * 
     * A NestedFeature must be explicitly selected. It is not assumed to be selected when the parent feature is selected. 
     * Selection of a nested feature causes its parent feature to be selected, but not vice-versa. The definition of a 
     * NestedFeature indicates that application of the NestedFeature is dependent on application of the parent feature.
     * 
     * @return A collection of nested features.
     */
    Collection<NestedFeature> getNestedFeatures();
    
    /**
     * Gets the content elements.
     * 
     * The ContentElement referred to MUST be in the selectable content hierarchy defined by the SelectableContent element.<br><br>
     * When the content reference is to a CompositeUnit, the composite and all content elements below it in the content hierarchy 
     * are considered to be in scope when the feature is selected. Ease of referencing a group of content from a feature can be 
     * one reason for using a composite in the content hierarchy.
     * 
     * @return A collection of references to content elements to be deployed when the feature is selected.
     */
    Collection<String> getContentElements();
    
    /**
     * Gets the required features.
     * 
     * When the selection of one feature requires the selection of another feature, the RequiredFeature can be used to 
     * specify this requirement.<br><br>
     * When two features identify each other as required features, they are always selected together.<br><br>
     * The selection of the defining feature MUST cause the required feature to be selected.
     * 
     * @return A collection of references to features that are required when the defining feature is selected and so is selected automatically.
     */
    Collection<String> getRequiredFeature();
    
    /**
     * Gets the package feature references.
     * 
     * Selection of a feature may result in selection of an aggregated packages feature identified by a ContainedPackage 
     * element anywhere in the BaseContent or SelectableContent hierarchies. A PackageFeatureReference identifies both the 
     * ContainedPackage and the applicable features to be selected in that package.
     * 
     * @return A collection of references to features to be selected in a ContainedPackage defined in either the BaseContent or 
     * SelectableContent hierarchies.
     */
    Collection<PackageFeatureReference> getPackageFeatureReferences();
    
    /**
     * Gets the variables.
     * 
     * Variables defined in features are useful when inputs to an artifact need to vary based on which features are selected 
     * for a particular deployment. Artifact arguments can be defined in terms of feature Variables to allow for this variation. 
     * When an artifact deploys selectable content, inputs to the artifact that indicate the selections for a particular deployment 
     * can be associated with feature selection in the SDD via feature Variables.<br><br>
     * <ul>For example, a Feature that deploys a trace facility might define a Variable called TraceSettings. The value of an 
     * argument to a base content artifact might define its value as $(TraceSettings). If the feature is selected, this argument 
     * would be used and its value would be taken from the feature Variable. If the feature is not selected, the argument would be 
     * ignored.</ul>
     * A Variable defined in a feature differs from Variable elements defined in content elements in one important way. A reference 
     * to an undefined feature Variable is treated as an empty string and is considered to be defined.
     * 
     * @return The collection of variables that can be used anywhere in any variable expression in the SDD.
     */
    Collection<DerivedVariable> getVariables();
    
    /**
     * Gets the feature ID.
     * 
     * Provides the means to reference a feature from other features.<br><br>
     * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
     * 
     * @return ID used within the SDD to refer to the feature.
     */
    String getFeatureID();
    
    /**
     * Gets whether feature can be added to a deployed instance of the solution.
     * 
     * When a solution and the artifacts that deploy the various parts of the solution are designed in a way that 
     * supports the addition of a particular feature at a later time (after the deployment of the base solution), the addOn 
     * attribute is set to true.
     * 
     * @return A true value indicates that the feature can be added to a deployed instance of the solution.
     */
    Boolean getAddOn();
    
    /**
     * Sets the condition.
     * 
     * If the features and its nested features are only applicable in certain environments, a Condition can be defined. 
     * When the Condition is not met, the feature and its nested features are not in scope. 
     * <ul>For example, some features may be available only on a Linux operating system, even though the software can be 
     * applied on other operating systems. In this case, a Condition can be defined to cause the feature to be ignored 
     * when the operating system is not Linux.</ul>
     * 
     * @param condition A condition that determines if the feature is relevant to a particular deployment.
     */
    void setCondition(Condition condition);
    
    /**
     * Sets the multilplicity.
     * 
     * When multiple instances of a feature can be selected, a Multiplicity element MUST be defined. 
     * <ul>For example, a solution that includes a server and a client may allow the deployment of multiple clients. 
     * In this situation, a feature that defines a Multiplicity element would select the content elements that deploy 
     * the client software.</ul>
     * 
     * @param multiplicity Both an indication that multiple instances of the feature can be selected and the specification 
     * of their constraints.
     */
    void setMultiplicity(Multiplicity multiplicity);
    
    /**
     * Sets the mandatory languages.
     * 
     * The deployer has no ability to determine if a mandatory language will be deployed.
     * 
     * @param languages The collection of languages that will be deployed.
     */
    void setMandatoryLanguages(Collection<Language> languages);
    
    /**
     * Sets the optional languages.
     * 
     * Each language group in the list of optional languages defines a list of one or more languages that can be selected together.<br><br>
     * Language groups defined in LanguageSelections MAY be used to allow the deployer to select individual languages or to allow 
     * selection of multiple languages as a single choice.
     * 
     * @param languages The collection of language selections available to the deployer.
     */
    void setOptionalLanguages(Collection<Language> languages);
    
    /**
     * Sets the optional language sets.
     * 
     * When the SDD author allows the deployer to select languages for deployment as a set, it is defined in a LanguageSet element 
     * within OptionalLanguages.
     * <ul>One example of a reason to define optional languages in a set rather than individually is for a group of languages that 
     * are packaged together and whose deployment cannot be separated.</ul>
     * 
     * @param languageSets The collection of languages that can be chosen together.
     */
    void setOptionalLanguageSets(Collection<LanguageSet> languageSets);
    
    /**
     * Sets the nested features.
     * 
     * A NestedFeature must be explicitly selected. It is not assumed to be selected when the parent feature is selected. 
     * Selection of a nested feature causes its parent feature to be selected, but not vice-versa. The definition of a 
     * NestedFeature indicates that application of the NestedFeature is dependent on application of the parent feature.
     * 
     * @param features A collection of nested features.
     */
    void setNestedFeatures(Collection<NestedFeature> features);
    
    /**
     * Sets the content elements.
     * 
     * The ContentElement referred to MUST be in the selectable content hierarchy defined by the SelectableContent element.<br><br>
     * When the content reference is to a CompositeUnit, the composite and all content elements below it in the content hierarchy 
     * are considered to be in scope when the feature is selected. Ease of referencing a group of content from a feature can be 
     * one reason for using a composite in the content hierarchy.
     * 
     * @param content A collection of references to content elements to be deployed when the feature is selected.
     */
    void setContentElements(Collection<String> content);
    
    /**
     * Sets the required features.
     * 
     * When the selection of one feature requires the selection of another feature, the RequiredFeature can be used to 
     * specify this requirement.<br><br>
     * When two features identify each other as required features, they are always selected together.<br><br>
     * The selection of the defining feature MUST cause the required feature to be selected.
     * 
     * @param features A collection of features that are required when the defining feature is selected and so is selected automatically.
     */
    void setRequiredFeature(Collection<String> features);
    
    /**
     * Sets the package feature references.
     * 
     * Selection of a feature may result in selection of an aggregated packages feature identified by a ContainedPackage 
     * element anywhere in the BaseContent or SelectableContent hierarchies. A PackageFeatureReference identifies both the 
     * ContainedPackage and the applicable features to be selected in that package.
     * 
     * @param packageFeatureReferences A collection of features to be selected in a ContainedPackage defined in either the 
     * BaseContent or SelectableContent hierarchies.
     */
    void setPackageFeatureReferences(Collection<PackageFeatureReference> packageFeatureReferences);
    
    /**
     * Sets the variables.
     * 
     * Variables defined in features are useful when inputs to an artifact need to vary based on which features are selected 
     * for a particular deployment. Artifact arguments can be defined in terms of feature Variables to allow for this variation. 
     * When an artifact deploys selectable content, inputs to the artifact that indicate the selections for a particular deployment 
     * can be associated with feature selection in the SDD via feature Variables.<br><br>
     * <ul>For example, a Feature that deploys a trace facility might define a Variable called TraceSettings. The value of an 
     * argument to a base content artifact might define its value as $(TraceSettings). If the feature is selected, this argument 
     * would be used and its value would be taken from the feature Variable. If the feature is not selected, the argument would be 
     * ignored.</ul>
     * A Variable defined in a feature differs from Variable elements defined in content elements in one important way. A reference 
     * to an undefined feature Variable is treated as an empty string and is considered to be defined.
     * 
     * @param variables The collection of variables that can be used anywhere in any variable expression in the SDD.
     */
    void setVariables(Collection<DerivedVariable> variables);
    
    /**
     * Sets the feature ID.
     * 
     * Provides the means to reference a feature from other features.<br><br>
     * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
     * 
     * @param id Used within the SDD to refer to the feature.
     * @throws InvalidInputException if the parameter is null
     */
    void setFeatureID(String id);
    
    /**
     * Sets whether feature can be added to a deployed instance of the solution.
     * 
     * When a solution and the artifacts that deploy the various parts of the solution are designed in a way that 
     * supports the addition of a particular feature at a later time (after the deployment of the base solution), the addOn 
     * attribute is set to true.
     * 
     * @param addOn A true value indicates that the feature can be added to a deployed instance of the solution.
     */
    void setAddOn(Boolean addOn);
    
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
    public Object clone();
}
