/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

public class ValidatorUtils {
	private static final String SCHEMATRON_TEXT = "iso_schematron_text.xsl";
	private static final String SCHEMATRON_SKELETON = "iso_schematron_skeleton.xsl";
	
	public static File getTempFileFromSchemaName(String schemaFileName) {
		// Try to load the Schema from the jar file
        InputStream in = ValidatorUtils.class.getResourceAsStream("/sdd_schema/" + schemaFileName); //$NON-NLS-1$
        
        // If the jar is missing, load from the file system
        if (in == null) {
            try {
                in = new File("sdd_schema/" + schemaFileName).toURL().openStream(); //$NON-NLS-1$
            } catch (Exception e) {
                return null;
            }
        }
        
        // Get a temporary file and delete it on exit
        File tempFile = null;
        try {
			tempFile = File.createTempFile(schemaFileName, null);

		} catch (IOException e) {
			return null;
		}
		tempFile.deleteOnExit();
		
		// Get an OutputStream to write to the temp file
		OutputStream out = null;
		try {
			out = new FileOutputStream(tempFile);
		} catch (FileNotFoundException e) {
			// Unpossible, the file was just created
			return null;
		}
		
		// Copy the contents of the schema to the temp file
		byte[] copyBuffer = new byte[1024];
        int bytesRead = 0;
        
		try {
			while ((bytesRead = in.read(copyBuffer)) != -1) {
				out.write(copyBuffer, 0, bytesRead);
			}
			in.close();
			out.close();
		} catch (IOException e) {
			return null;
		}
		
		return tempFile;        
	}
	
	public static File getTempFileFromSchematronText() {
		// Try to load the Schematron files from the jar file
		InputStream skeletonIn = ValidatorUtils.class.getResourceAsStream("/schematron/" + SCHEMATRON_SKELETON); //$NON-NLS-1$
        InputStream textIn = ValidatorUtils.class.getResourceAsStream("/schematron/" + SCHEMATRON_TEXT); //$NON-NLS-1$
        
        // If the jar is missing, load from the file system
        if (skeletonIn == null) {
            try {
                skeletonIn = new File("schematron/" + SCHEMATRON_SKELETON).toURL().openStream(); //$NON-NLS-1$
            } catch (Exception e) {
                return null;
            }
        }
        
        if (textIn == null) {
            try {
                textIn = new File("schematron/" + SCHEMATRON_TEXT).toURL().openStream(); //$NON-NLS-1$
            } catch (Exception e) {
                return null;
            }
        }
        
        // Get temporary files and delete them on exit
        File skeletonTempFile = null;
        try {
			skeletonTempFile = File.createTempFile(SCHEMATRON_SKELETON, null);
		} catch (IOException e) {
			return null;
		}
		skeletonTempFile.deleteOnExit();
        
		File textTempFile = null;
        try {
			textTempFile = File.createTempFile(SCHEMATRON_TEXT, null);
		} catch (IOException e) {
			return null;
		}
		textTempFile.deleteOnExit();

		// Get OutputStreams to write to the temp files
		OutputStream skeletonOut = null;
		try {
			skeletonOut = new FileOutputStream(skeletonTempFile);
		} catch (FileNotFoundException e) {
			// Unpossible, the file was just created
			return null;
		}
		
		OutputStream textOut = null;
		try {
			textOut = new FileOutputStream(textTempFile);
		} catch (FileNotFoundException e) {
			// Unpossible, the file was just created
			return null;
		}

		// Copy the contents of the Skeleton file to the temp skeleton file
		byte[] copyBuffer = new byte[1024];
        int bytesRead = 0;
		try {
			while ((bytesRead = skeletonIn.read(copyBuffer)) != -1) {
				skeletonOut.write(copyBuffer, 0, bytesRead);
			}
			skeletonIn.close();
			skeletonOut.close();
		} catch (IOException e) {
			return null;
		}
		
		// Read in the contents of the Text file
        bytesRead = 0;
        String textContents = new String();
		try {
			while ((bytesRead = textIn.read(copyBuffer)) != -1) {
				textContents += new String(copyBuffer, 0, bytesRead);
			}
			textIn.close();
		} catch (IOException e) {
			return null;
		}
		
		// Replace the Schematron Skeleton import with the correct absolute path
		String skeletonPath = skeletonTempFile.getAbsolutePath().replaceAll("\\\\", "\\\\\\\\");
		textContents = textContents.replaceFirst("<xsl:import href=\"iso_schematron_skeleton.xsl\"/>",
				"<xsl:import href=\"" + skeletonPath + "\"/>");
		// Write the updated file contents to the temp file
		try {
			textOut.write(textContents.getBytes());
		} catch (IOException e) {
			return null;
		}
		
		return textTempFile;		
	}
	
	public static Element findElement(Node rootElement, String elementName) {
		// If the rootElement is a Document, use the Document Element instead
		if (rootElement instanceof Document) {
			return findElement(((Document)rootElement).getDocumentElement(), elementName);
		}
		
		// Is this the Element we want?
		if (rootElement instanceof Element && nodeNameEquals(rootElement, elementName)) {
			return (Element)rootElement;
		}
		
		// Get a list of children
		NodeList nodes = rootElement.getChildNodes();
		
		// Check each child to see if it's the correct Element
		for (int i = 0; i < nodes.getLength(); i++) {
			Node node = nodes.item(i);
			
			if (node instanceof Element) {
				// If this Element matches the given element name, return it
				if (node.getNodeName() != null && nodeNameEquals(node, elementName)) {
					return (Element)node;
				}
				// If this Node has children, check them for the given element name
				else if (node.getChildNodes().getLength() > 0) {
					NodeList childNodes = node.getChildNodes();
					
					for (int j = 0; j < childNodes.getLength(); j++) {
						// Recurse!
						Node childNode = findElement(childNodes.item(j), elementName);
						
						// Return the correct Element if recursing found the given element name
						if (childNode != null) {
							return (Element)childNode;
						}
					}
				}
			}
		}
		
		// Given element name not found
		return null;
	}
	
	public static boolean nodeNameEquals(Node node, String nodeName) {
		String name = node.getNodeName();
		
		// String out the prefix if present
		if (name.indexOf(':') != -1) {
			name = name.substring(name.indexOf(':') + 1, name.length());
		}
		
		return nodeName.equals(name);
	}
}