/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.examples.common.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.FileChannel;
import java.util.Collection;
import java.util.Vector;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.SDD_DASImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.util.ValidatorUtils;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.SDD_DAS;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.ValidationRule;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.XMLValidationError;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/*
 *  ValidationRule Sample
 *  
 *  Eric S Rose, esrose@us.ibm.com
 *  
 *  The ValidationRule interface provides a way to write custom XML validation code beyond what is provided
 *  by standard XML Schema.  Any number of rules can be added to an instance of the SDD_DAS and the SDD_DAS
 *  will execute those rules against its associated XML.  The validation rule should return a Collection of
 *  XMLValidationErrors for cases where the XML is not valid according to that rule.
 *  
 *  This sample shows how to create a ValidationRule that provides some constraints across two
 *  separate XML files.  This type of validation could be used to ensure referential integrity.
 *  
 *  The ValidationRule is implemented below as an inner class called CustomValidationRule.
 */


public class ValidationRule_Sample {
    // Use platform independent file separator
    private static final String FSEP = System.getProperty("file.separator");

    // Path to the location of files needed for the sample to run
    private static final String filesPath = "src" + FSEP + "org" + FSEP + "eclipse" + FSEP + "cosmos" + FSEP + "me"
    	+ FSEP + "internal" + FSEP + "deployment" + FSEP + "sdd" + FSEP + "examples" + FSEP + "common" + FSEP
    	+ "validation" + FSEP;

	// This sample depends on the following files
	private static final String DD_INPUT_FILE = filesPath + "DeploymentDescriptor.xml";
	private static final String PD_INPUT_FILE = filesPath + "PackageDescriptor.xml";


	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// Copy over the original XML files so we start with known descriptors
		prepFiles();

		// Create a DAS for both the Package and Deployment Descriptors
		SDD_DAS deploymentDescriptorDAS = new SDD_DASImpl();
		SDD_DAS packageDescriptorDAS = new SDD_DASImpl();
		
		// Load Document for the Package and Deployment Descriptors
		Document pdDocument = null;
		Document ddDocument = null;
		try {
			pdDocument = packageDescriptorDAS.loadDocument(new FileInputStream(PD_INPUT_FILE));
			ddDocument = deploymentDescriptorDAS.loadDocument(new FileInputStream(DD_INPUT_FILE));
		} catch (FileNotFoundException e) {
			// Thrown if the XML file is missing
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		} catch (IOException e) {
			// Thrown if the XML file is present but unreadable
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		} catch (XMLValidationException e) {
			// Thrown if the XML is not well-formed or no valid according to the SDD schema
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		}
		
		// Create an instance of CustomValidationRule with the Package Descriptor Document
		ValidationRule customValidationRule = new CustomValidationRule(pdDocument);
		// Add the rule to the DAS instance representing the Deployment Descriptor
		deploymentDescriptorDAS.addValidation(customValidationRule);
		
		// Validate the Deployment Descriptor Document using the CustomValidationRule
		Collection<XMLValidationError> validationErrors = deploymentDescriptorDAS.validate(ddDocument);

		// The example files provided are valid according to the CustomValidationRule
		// There should be no validation errors
		for (XMLValidationError error : validationErrors) {
			System.err.println(error.getErrorMessage());
		}
	}
	
	/*
	 * This ValidationRule will check that the <InstallArtifact> in the Deployment Descriptor has a
	 * corresponding <File> in the Package Descriptor.  The rule will be added to the XML_DAS
	 * representing the Deployment Descriptor and it will be given a reference to the Document for the
	 * Package Descriptor.
	 */
	private static class CustomValidationRule implements ValidationRule {
		private Document pdDocument;
		
		public CustomValidationRule(Document document) {
			// This Document should represent the Package Descriptor
			pdDocument = document;
		}
		
		/*
		 * The validate() method will perform these steps:
		 * 
		 * - Create a Document representing the given InputStream, which contains the Deployment Descriptor
		 * - For the <InstallArtifact> element in the Deployment Descriptor, check for a matching <File>
		 *     element in the PackageDescriptor
		 * - Create an XMLValidationError if the <InstallArtifact> that has no matching <File>
		 * - Return the error list to the DAS
		 * 
		 */
		public Collection<XMLValidationError> validate(InputStream xmlStream) {
			Collection<XMLValidationError> validationErrors = new Vector<XMLValidationError>();
			
			// Create a new DAS instance and use it to get a Document for the given XML
			SDD_DAS sddDas = new SDD_DASImpl();
			Document ddDocument = null;
			try {
				ddDocument = sddDas.loadDocument(xmlStream);
			} catch (IOException e) {
				// Thrown if the InputStream is unreadable
				e.printStackTrace();
				// No point in continuing - bail out
				System.exit(1);
			} catch (XMLValidationException e) {
				// Thrown if the XML is not well-formed or no valid according to the SDD schema
				e.printStackTrace();
				// No point in continuing - bail out
				System.exit(1);
			}
			
			// Get the <InstallArtifact> from the Deployment Descriptor
			Element installArtifact = ValidatorUtils.findElement(ddDocument, "InstallArtifact");
			// Get the <InstallArtifact>'s contentRef
			String contentRef = installArtifact.getAttribute("contentRef");
			
			// Create a local copy of the Package Descriptor to work with
			Element pdCopy = (Element)pdDocument.getDocumentElement().cloneNode(true);
			
			// The contentRef attribute of the <InstallArtifact> must match the id attribute of a <Content>
			boolean foundMatch = false;
			Element contentElement = ValidatorUtils.findElement(pdCopy, "Content");
			// Check each <Content> element until a match is found or there are no more <Content> elements
			while (contentElement != null && !foundMatch) {
				if (contentRef.equals(contentElement.getAttribute("id"))) {
					foundMatch = true;
				}
				else {
					// This <Content> element didn't match, so remove it and check the next one
					contentElement.getParentNode().removeChild(contentElement);
					contentElement = ValidatorUtils.findElement(pdCopy, "Content");
				}
			}
			
			// If no match was found, then the Package Descriptor/Deployment Descriptor pair is invalid
			// according to this rule.  Add an error to the validation error list.
			if (!foundMatch) {
				String errorMsg =
					"No matching Content element in Package Descriptor for InstallArtifact element in Deployment Descriptor.";
				validationErrors.add(new XMLValidationError(errorMsg, -1, -1, this, XMLValidationError.ERROR));
			}
			
			// Return the error list (which is empty if no errors occurred)
			return validationErrors;
		}		
	}

	private static void prepFiles() {
		final File deploymentDescriptor = new File(filesPath + "DeploymentDescriptor.xml");
		final File packageDescriptor = new File(filesPath + "PackageDescriptor.xml");
		final File ddOriginal = new File(filesPath + "DeploymentDescriptor.xml_Original");
		final File pdOriginal = new File(filesPath + "PackageDescriptor.xml_Original");

		// Overwrite the descriptor files with the originals so we start from a known state
		try {
			
			// Copy the Deployment Descriptor
			FileChannel ddSrcChannel = new FileInputStream(ddOriginal).getChannel();
			FileChannel ddDestChannel = new FileOutputStream(deploymentDescriptor).getChannel();
			ddDestChannel.transferFrom(ddSrcChannel, 0, ddSrcChannel.size());
			
			// Copy the Package Descriptor
			FileChannel pdSrcChannel = new FileInputStream(pdOriginal).getChannel();
			FileChannel pdDestChannel = new FileOutputStream(packageDescriptor).getChannel();
			pdDestChannel.transferFrom(pdSrcChannel, 0, pdSrcChannel.size());
			
		} catch (FileNotFoundException e) {
			// Thrown if one of the original files is missing
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		} catch (IOException e) {
			// Thrown if there's a problem reading or writing one of the files
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		}
	}	
}
