/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact.ArtifactArgument;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit.ContentSelectionFeatureImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit.LanguageImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.requirement.RequirementImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.variable.OutputVariableImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResourceMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResultingChangeMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResultingResourceMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentSelectionFeature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Language;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.OutputVariable;
import org.w3c.dom.Element;

public class ReferencedPackageImpl extends SPIDataObject implements ReferencedPackage {
	private static final String REFERENCED_PACKAGE_TYPE = "sdd-dd:ReferencedPackageType";
	private static final String ARGUMENT_LIST_TYPE = "sdd-dd:ArgumentListType";
	private static final String LANGUAGES_TYPE = "sdd-dd:LanguagesType";
	private static final String OUTPUT_VARIABLE_LIST_TYPE = "sdd-dd:OutputVariableListType";
	private static final String REQUIRED_CONTENT_SELECTION_TYPE = "sdd-dd:RequiredContentSelectionType";
	private static final String REQUIREMENTS_TYPE = "sdd-dd:RequirementsType";
    
	public ReferencedPackageImpl(Element rpdo, SPISessionImpl session) {
        super(rpdo, session, REFERENCED_PACKAGE_TYPE);
    }
    
    public Collection<ArtifactArgument> getArguments() {
        Collection<ArtifactArgument> arguments = new ArrayList<ArtifactArgument>();

        if (isSetElement("Arguments")) {
        	Element argumentsElement = getChild("Arguments");
        	Collection<Element> argumentElements = session.getChildren(argumentsElement, "Argument");
        	
        	for (Element argumentElement : argumentElements)
            {
        		String name = argumentElement.getAttribute("name");
        		String value = argumentElement.getAttribute("value");
        		Boolean required = new Boolean(argumentElement.getAttribute("required"));
        		
        		ArtifactArgument argument = new ArtifactArgument(name, value, required);
        		arguments.add(argument);
        	}
        }

        return arguments;
    }

    public String getReferencedPackageID() {
   		return getAttribute("id");
    }

    public String getOperation() {
   		return getAttribute("operation");
    }

    public String getContentReference() {
   		return getAttribute("contentRef");
    }

    public Collection<Language> getLanguages() {
        Collection<Language> languages = new ArrayList<Language>();

        if (isSetElement("Languages")) {
        	Element languagesElement = getChild("Languages");
        	Collection<Element> languageElements = session.getChildren(languagesElement, "Language");
        	
        	for (Element element : languageElements)
            {
        		Language language = new LanguageImpl(element, session);
        		languages.add(language);
        	}
        }

        return languages;
    }

    public Collection<OutputVariable> getOutputVariables() {
        Collection<OutputVariable> outputVariables = new ArrayList<OutputVariable>();

        if (isSetElement("OutputVariables")) {
        	Element variablesElement = getChild("OutputVariables");
        	Collection<Element> variableElements = session.getChildren(variablesElement, "OutputVariable");
        	
        	for (Element element : variableElements)
            {
        		OutputVariable outputVariable = new OutputVariableImpl(element, session);
        		outputVariables.add(outputVariable);
        	}
        }

        return outputVariables;
    }

    public Collection<ResultingChangeMap> getResultingChangeMaps() {
    	Collection<ResultingChangeMap> resultingChangeMaps = new ArrayList<ResultingChangeMap>();
    	
    	if (isSetElement("ResultingChangeMap")) {
    		Collection<Element> resultingChangeElements = getChildren("ResultingChangeMap");
    		
    		for (Element resultingChangeElement : resultingChangeElements) {
    			ResultingChangeMap resultingChangeMap = new ResultingChangeMapImpl(resultingChangeElement, session);
    			resultingChangeMaps.add(resultingChangeMap);
    		}
    	}
    	
    	return resultingChangeMaps;
    }
    
    public Collection<ResultingResourceMap> getResultingResourceMaps() {
    	Collection<ResultingResourceMap> resultingResourceMaps = new ArrayList<ResultingResourceMap>();
    	
    	if (isSetElement("ResultingResourceMap")) {
    		Collection<Element> resultingResourceElements = getChildren("ResultingResourceMap");
    		
    		for (Element resultingResourceElement : resultingResourceElements) {
    			ResultingResourceMap resultingResourceMap = 
    				new ResultingResourceMapImpl(resultingResourceElement, session);
    			resultingResourceMaps.add(resultingResourceMap);
    		}
    	}
    	
    	return resultingResourceMaps;
    }
    
    public Collection<ResourceMap> getRequiredResourceMaps() {
    	Collection<ResourceMap> requiredResourceMaps = new ArrayList<ResourceMap>();
    	
    	if (isSetElement("RequiredResourceMap")) {
    		Collection<Element> requiredResourceElements = getChildren("RequiredResourceMap");
    		
    		for (Element requiredResourceElement : requiredResourceElements) {
    			ResourceMap requiredResourceMap = new ResourceMapImpl(requiredResourceElement, session);
    			requiredResourceMaps.add(requiredResourceMap);
    		}
    	}
    	
    	return requiredResourceMaps;
    }
    
    public Collection<ContentSelectionFeature> getRequiredFeatures() {
        Collection<ContentSelectionFeature> requiredFeatures = new ArrayList<ContentSelectionFeature>();

        if (isSetElement("RequiredContentSelection")) {
        	Element contentSelectionElement = getChild("RequiredContentSelection");
        	Collection<Element> featureElements = session.getChildren(contentSelectionElement, "Feature");
        	
        	for (Element element : featureElements)
            {
        		ContentSelectionFeature contentSelectionFeature = new ContentSelectionFeatureImpl(element, session);
        		requiredFeatures.add(contentSelectionFeature);
        	}
        }

        return requiredFeatures;    	
    }

    public String getRequiredGroup() {
        String requiredGroup = null;
        
        if (isSetElement("RequiredContentSelection")) {
        	Element contentSelectionElement = getChild("RequiredContentSelection");
        	Collection<Element> groupElements = session.getChildren(contentSelectionElement, "Group");
        	
        	if (groupElements != null && !groupElements.isEmpty()) {
        		requiredGroup = groupElements.iterator().next().getTextContent();
        	}
        }
        	
        return requiredGroup;
    }

    public Collection<Requirement> getRequirements() {
        Collection<Requirement> requirements = new ArrayList<Requirement>();

        if (isSetElement("Requirements")) {
        	Element requirementsElement = getChild("Requirements");
        	Collection<Element> requirementElements = session.getChildren(requirementsElement, "Requirement");
        	
        	for (Element element : requirementElements)
            {
        		Requirement requirement = new RequirementImpl(element, session);
        		requirements.add(requirement);
        	}
        }

        return requirements;
    }

    public BigInteger getWeight() {
    	if (isSetAttribute("weight")) {
    		BigInteger weight = new BigInteger(getAttribute("weight"));
    		
    		return weight;
    	}
    	else {
    		return null;
    	}
    }
    
    public void setArguments(Collection<ArtifactArgument> arguments) {
        if (arguments == null) {
        	unSetElement("Arguments");
        }
        else {
        	Element argumentListElement = session.createDeploymentElement("Arguments");
        	
        	for (ArtifactArgument argument : arguments) {
        		Element argumentElement = session.createDeploymentElement("Argument");
        		argumentElement.setAttribute("name", argument.getName());
        		if (argument.getValue() != null) {
        			argumentElement.setAttribute("value", argument.getValue());
        		}
        		if (argument.getRequired() != null) {
        			argumentElement.setAttribute("required", argument.getRequired().toString());
        		}
        		
        		session.insertNode(argumentListElement, argumentElement, ARGUMENT_LIST_TYPE);
        	}
        	
        	setContent(argumentListElement);
        }
    }

    public void setReferencedPackageID(String id) {
        session.testParameter(id, 1, null);

        setAttribute("id", id);
    }

    public void setOperation(String operation) {
       	setAttribute("operation", operation);
    }

    public void setContentReference(String ref) {
        session.testParameter(ref, 1, null);

        setAttribute("contentRef", ref);
    }

    public void setLanguages(Collection<Language> languages) {
        if (languages == null) {
        	unSetElement("Languages");
        }
        else {
        	Element languagesElement = session.createDeploymentElement("Languages");
        	
        	for (Language language : languages) {
        	    session.insertNode(languagesElement, (SPIDataObject) language, LANGUAGES_TYPE);
        	}
        	
        	setContent(languagesElement);
        }
    }

    public void setOutputVariables(Collection<OutputVariable> outputVariables) {
        if (outputVariables == null) {
        	unSetElement("OutputVariables");
        }
        else {
        	Element outputVariableListElement = session.createDeploymentElement("OutputVariables");
        	
        	for (OutputVariable outputVariable : outputVariables) {
        	    session.insertNode(outputVariableListElement, (SPIDataObject) outputVariable, OUTPUT_VARIABLE_LIST_TYPE);
        	}
        	
        	setContent(outputVariableListElement);
        }
    }

    public void setRequiredFeatures(Collection<ContentSelectionFeature> features) {
        if (features != null) {
       		if (!isSetElement("RequiredContentSelection")) {
    			Element contentSelectionElement = session.createDeploymentElement("RequiredContentSelection");
    			setContent(contentSelectionElement);
    		}
       		
       		Element contentSelectionElement = getChild("RequiredContentSelection");
			Collection<Element> featureElements = session.getChildren(contentSelectionElement, "Feature");
			
			for (Element element : featureElements)
            {
				contentSelectionElement.removeChild(element);
			}
       		
       		for (ContentSelectionFeature feature : features) {
       		    session.insertNode(contentSelectionElement, (SPIDataObject) feature, REQUIRED_CONTENT_SELECTION_TYPE);
       		}
       		
       		setContent(contentSelectionElement);
        }    	
    }

    public void setRequiredGroup(String group) {
    	if (group != null) {
    		if (!isSetElement("RequiredContentSelection")) {
    			Element contentSelectionElement = session.createDeploymentElement("RequiredContentSelection");
    			setContent(contentSelectionElement);
    		}
    		
    		Element contentSelectionElement = getChild("RequiredContentSelection");
			Collection<Element> groupElements = session.getChildren(contentSelectionElement, "Group");
			
			if (groupElements != null && !groupElements.isEmpty()) 
			{
				contentSelectionElement.removeChild(groupElements.iterator().next());
			}
    		
    		Element groupElement = session.createDeploymentElement("Group");
    		groupElement.setTextContent(group);
    		
    		session.insertNode(contentSelectionElement, groupElement, REQUIRED_CONTENT_SELECTION_TYPE);
    		
    		setContent(contentSelectionElement);
    	}
    }

    public void setRequirements(Collection<Requirement> requirements) {
        if (requirements == null) {
        	unSetElement("Requirements");
        }
        else {
        	Element requirementsElement = session.createDeploymentElement("Requirements");
        	
        	for (Requirement requirement : requirements) {
        	    session.insertNode(requirementsElement, (SPIDataObject) requirement, REQUIREMENTS_TYPE);
        	}
        	
        	setContent(requirementsElement);
        }
    }

    public void setResultingChangeMaps(Collection<ResultingChangeMap> resultingChangeMaps) {
    	if (resultingChangeMaps == null) {
    		unSetElement("ResultingChangeMap");
    	}
    	else {
        	removeContent(getChildren("ResultingChangeMap", false));
        	
            addContent(resultingChangeMaps);
        }
    }

    public void setResultingResourceMaps(Collection<ResultingResourceMap> resultingResourceMaps) {
    	if (resultingResourceMaps == null) {
    		unSetElement("ResultingResourceMap");
    	}
    	else {
        	removeContent(getChildren("ResultingResourceMap", false));
        	
            addContent(resultingResourceMaps);
        }
    }

    public void setRequiredResourceMaps(Collection<ResourceMap> requiredResourceMaps) {
    	if (requiredResourceMaps == null) {
    		unSetElement("RequiredResourceMap");
    	}
    	else {
        	removeContent(getChildren("RequiredResourceMap", false));
        	
            addContent(requiredResourceMaps);
        }
    }

    public void setWeight(BigInteger weight) {
        if (weight == null) {
            unSetAttribute("weight");
        }
        else {
            setAttribute("weight", weight.toString());
        }
    }

    public Object clone() {
        return new ReferencedPackageImpl((Element)getElement().cloneNode(true), session);        
    }
}