/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.Artifact;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Completion;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.CompositeInstallable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.CompositeLocalizationUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.CompositeUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ConfigurationUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ConstrainedResource;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentSelectionFeature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentUnitFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Feature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Group;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.InstallableUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Language;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.LanguageSet;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.LocalizationUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.MultiFeatureSelect;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Multiplicity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.NestedFeature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.PackageFeatureReference;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Relationship;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ResultingChange;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ResultingResource;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.DerivedVariable;
import org.w3c.dom.Element;

public class ContentUnitFactoryImpl extends SPIObject implements ContentUnitFactory {
	private SPISessionImpl session;
	private static final String CONFIGURATION_UNIT_TYPE = "sdd-dd:ConfigurationUnitType";
	private static final String INSTALLABLE_UNIT_TYPE = "sdd-dd:InstallableUnitType";
	private static final String LOCALIZATION_UNIT_TYPE = "sdd-dd:LocalizationUnitType";
	
	public ContentUnitFactoryImpl(SPISessionImpl session) {
	    super(session);
		this.session = session;
	}

	public Completion createCompletion(String completionType, String resourceReference) {
		session.testParameter(completionType, 1, null);
		session.testParameter(resourceReference, 2, null);

		Element completionElement = session.createDeploymentElement("Completion");
		
		Completion c = new CompletionImpl(completionElement, session);
		c.setCompletionType(completionType);
		c.setCompletionResourceReference(resourceReference);
		
		return c;
	}

	public ConfigurationUnit createConfigurationUnit(String id, String targetResourceReference, Artifact configArtifact) {
		session.testParameter(id, 1, null);
		session.testParameter(targetResourceReference, 2, null);
		session.testParameter(configArtifact, 3, null);

		Element configElement = session.createDeploymentElement("ConfigurationUnit");
		
		session.insertNode(configElement, session.createDeploymentElement("Artifacts"), CONFIGURATION_UNIT_TYPE);
		
		ConfigurationUnit cu = new ConfigurationUnitImpl(configElement, session);
		
		cu.setContentUnitID(id);
		cu.setTargetResourceReference(targetResourceReference);
		
		cu.setConfigurationArtifact(configArtifact);
		
		return cu;
	}

	public InstallableUnit createInstallableUnit(String id, String targetResourceReference, Artifact installArtifact) {
		session.testParameter(id, 1, null);
		session.testParameter(targetResourceReference, 2, null);
		session.testParameter(installArtifact, 3, null);

		Element iuElement = session.createDeploymentElement("InstallableUnit");
		
		session.insertNode(iuElement, session.createDeploymentElement("Artifacts"), INSTALLABLE_UNIT_TYPE);
		
		InstallableUnit iu = new InstallableUnitImpl(iuElement, session);
		iu.setContentUnitID(id);
		iu.setTargetResourceReference(targetResourceReference);
		
		iu.setInstallArtifact(installArtifact);
		
		return iu;
	}

	public InstallableUnit createInstallableUnit(String id, String targetResourceReference, Artifact initialArtifact,
			int artifactType) {
		session.testParameter(id, 1, null);
		session.testParameter(targetResourceReference, 2, null);
		session.testParameter(initialArtifact, 3, null);

		int[] validValues = {
				ContentUnitFactory.INSTALL_ARTIFACT,
				ContentUnitFactory.UNDO_ARTIFACT,
				ContentUnitFactory.UNINSTALL_ARTIFACT,
				ContentUnitFactory.UPDATE_ARTIFACT
		};
				
		session.testIntParameter(artifactType, 4, validValues);
		
		Element iuElement = session.createDeploymentElement("InstallableUnit");
		
		session.insertNode(iuElement, session.createDeploymentElement("Artifacts"), INSTALLABLE_UNIT_TYPE);
		
		InstallableUnit iu = new InstallableUnitImpl(iuElement, session);
		iu.setContentUnitID(id);
		iu.setTargetResourceReference(targetResourceReference);
				
		switch (artifactType) {
			case ContentUnitFactory.INSTALL_ARTIFACT:
				iu.setInstallArtifact(initialArtifact);
				break;
			case ContentUnitFactory.UNDO_ARTIFACT:
				iu.setUndoArtifact(initialArtifact);
				break;
			case ContentUnitFactory.UNINSTALL_ARTIFACT:
				iu.setUninstallArtifact(initialArtifact);
				break;
			case ContentUnitFactory.UPDATE_ARTIFACT:
				iu.setUpdateArtifact(initialArtifact);
				break;
		}
		
		return iu;
	}
	

	public LocalizationUnit createLocalizationUnit(String id, String targetResourceReference, Artifact installArtifact) {
		session.testParameter(id, 1, null);
		session.testParameter(targetResourceReference, 2, null);
		session.testParameter(installArtifact, 3, null);

		Element luElement = session.createDeploymentElement("LocalizationUnit");
		
		session.insertNode(luElement, session.createDeploymentElement("Artifacts"), LOCALIZATION_UNIT_TYPE);
		
		LocalizationUnit lu = new LocalizationUnitImpl(luElement, session);
		lu.setContentUnitID(id);
		lu.setTargetResourceReference(targetResourceReference);
		
		lu.setInstallArtifact(installArtifact);

		return lu;
	}

	public LocalizationUnit createLocalizationUnit(String id, String targetResourceReference, Artifact initialArtifact,
			int artifactType) {
		session.testParameter(id, 1, null);
		session.testParameter(targetResourceReference, 2, null);
		session.testParameter(initialArtifact, 3, null);

		int[] validValues = {
				ContentUnitFactory.INSTALL_ARTIFACT,
				ContentUnitFactory.UNDO_ARTIFACT,
				ContentUnitFactory.UNINSTALL_ARTIFACT,
				ContentUnitFactory.UPDATE_ARTIFACT
		};
				
		session.testIntParameter( artifactType, 4, validValues);
		
		Element luElement = session.createDeploymentElement("LocalizationUnit");
		
		session.insertNode(luElement, session.createDeploymentElement("Artifacts"), LOCALIZATION_UNIT_TYPE);
		
		LocalizationUnit lu = new LocalizationUnitImpl(luElement, session);
		lu.setContentUnitID(id);
		lu.setTargetResourceReference(targetResourceReference);
		
		switch (artifactType) {
			case ContentUnitFactory.INSTALL_ARTIFACT:
				lu.setInstallArtifact(initialArtifact);
				break;
			case ContentUnitFactory.UNDO_ARTIFACT:
				lu.setUndoArtifact(initialArtifact);
				break;
			case ContentUnitFactory.UNINSTALL_ARTIFACT:
				lu.setUninstallArtifact(initialArtifact);
				break;
			case ContentUnitFactory.UPDATE_ARTIFACT:
				lu.setUpdateArtifact(initialArtifact);
				break;
		}

        return lu;
    }

    public Language createLanguage(String type) {
        session.testParameter(type, 1, null);

        Element languageElement = session.createDeploymentElement("Language");
		
        Language l = new LanguageImpl(languageElement, session);
		l.setType(type);
		
        return l;
    }

    public ResultingChange createResultingChange() {
    	Element resultingChangeElement = session.createDeploymentElement("ResultingChange");
				
		return new ResultingChangeImpl(resultingChangeElement, session);
	}

	public ResultingResource createResultingResource(String resultingResourceReference) {
		session.testParameter(resultingResourceReference, 1, null);
		
		Element resultingResElement = session.createDeploymentElement("ResultingResource");
		
		ResultingResource rr = new ResultingResourceImpl(resultingResElement, session);
		rr.setResourceReference(resultingResourceReference);
		
		return rr;
	}

    public Relationship createRelationship(String relatedResourceRef, String type) throws InvalidInputException {
        log.entering(getClass().getName(), "createRelationship");
        session.testParameter(relatedResourceRef, 1, null);
        session.testParameter(type, 2, null);

        Element relationshipElement = session.createDeploymentElement("Relationship");
        
        Relationship r = new RelationshipImpl(relationshipElement, session);
        r.setRelatedResourceReference(relatedResourceRef);
        r.setType(type);
        
        log.exiting(getClass().getName(), "createRelationship");
        return r;
    }

	public CompositeInstallable createCompositeInstallable(String id, String operation) throws InvalidInputException {
        log.entering(getClass().getName(), "createCompositeInstallable");
        session.testParameter(id, 1, null);
        session.testParameter(operation, 2, null);

        Element element = session.createDeploymentElement("CompositeInstallable");
        
        CompositeInstallable ci = new CompositeInstallableImpl(element, session);
        ci.setContentUnitID(id);
        ci.setOperation(operation);

        log.exiting(getClass().getName(), "createCompositeInstallable");
        return ci;
    }
	

    public CompositeLocalizationUnit createCompositeLocalizationUnitWithLocalizationUnits(String id,
    		Collection<LocalizationUnit> localizationUnits) throws InvalidInputException {
        log.entering(getClass().getName(), "createCompositeLocalizationUnitWithLocalizationUnits");
        session.testParameter(id, 1, null);
        session.testParameter(localizationUnits, 2, null);
        
        Element element = session.createDeploymentElement("CompositeLocalizationUnit");
        
        CompositeLocalizationUnit cl = new CompositeLocalizationUnitImpl(element, session);
        cl.setContentUnitID(id);
        cl.setLocalizationUnits(localizationUnits);
        
        log.exiting(getClass().getName(), "createCompositeLocalizationUnitWithLocalizationUnits");
        return cl;
    }
    
    public CompositeLocalizationUnit createCompositeLocalizationUnitWithCompositeLocalizationUnits(String id,
    		Collection<CompositeLocalizationUnit> compositeLocalizationUnits) throws InvalidInputException {
        log.entering(getClass().getName(), "createCompositeLocalizationUnitWithCompositeLocalizationUnits");
        session.testParameter(id, 1, null);
        session.testParameter(compositeLocalizationUnits, 2, null);

        Element element = session.createDeploymentElement("CompositeLocalizationUnit");
        
        CompositeLocalizationUnit cl = new CompositeLocalizationUnitImpl(element, session);
        cl.setContentUnitID(id);
        cl.setCompositeLocalizationUnits(compositeLocalizationUnits);
        
        log.exiting(getClass().getName(), "createCompositeLocalizationUnitWithCompositeLocalizationUnits");
        return cl;
    }
    
    public CompositeLocalizationUnit createCompositeLocalizationUnitWithContainedLocalizationPackages(String id,
    		Collection<ReferencedPackage> containedLocalizationPackages) throws InvalidInputException {
        log.entering(getClass().getName(), "createCompositeLocalizationUnitWithContainedLocalizationPackages");
        session.testParameter(id, 1, null);
        session.testParameter(containedLocalizationPackages, 2, null);

        Element element = session.createDeploymentElement("CompositeLocalizationUnit");
        
        CompositeLocalizationUnit cl = new CompositeLocalizationUnitImpl(element, session);
        cl.setContentUnitID(id);
        cl.setContainedLocalizationPackages(containedLocalizationPackages);
        
        log.exiting(getClass().getName(), "createCompositeLocalizationUnitWithContainedLocalizationPackages");
        return cl;
    }
    

    public CompositeUnit createCompositeUnit(String id) throws InvalidInputException {
        session.testParameter(id, 1, null);
        
        Element element = session.createDeploymentElement("CompositeUnit");
        
        CompositeUnit cu = new CompositeUnitImpl(element, session);
        cu.setContentUnitID(id);
        
        return cu;
    }
    
    public Group createGroup(String id, Collection<String> selectedFeatures) throws InvalidInputException {
        log.entering(getClass().getName(), "createGroup");
        session.testParameter(id, 1, null);
        session.testParameter(selectedFeatures, 2, null);

        Element element = session.createDeploymentElement("Group");
        
        Group group = new GroupImpl(element, session);
        group.setGroupID(id);
        group.setSelectedFeatures(selectedFeatures);
        
        log.exiting(getClass().getName(), "createGroup");
        return group;
    }
    
    public LanguageSet createLanguageSet(Collection<Language> languages) throws InvalidInputException {
        log.entering(getClass().getName(), "createLanguageSet");
        session.testParameter(languages, 1, null);

        Element element = session.createDeploymentElement("LanguageSet");
        
        LanguageSet languageSet = new LanguageSetImpl(element, session);
        languageSet.setLanguages(languages);
        
        log.exiting(getClass().getName(), "createLanguageSet");
        return languageSet;
    }

    public MultiFeatureSelect createMultiFeatureSelect(Collection<String> features) throws InvalidInputException {
        log.entering(getClass().getName(), "createMultiFeatureSelect");
        session.testParameter(features, 1, null);
        
        Element element = session.createDeploymentElement("MultiSelect");
        
        MultiFeatureSelect mfs = new MultiFeatureSelectImpl(element, session);
        mfs.setFeatures(features);
        
        log.exiting(getClass().getName(), "createMultiFeatureSelect");
        return mfs;
    }
    
    public Multiplicity createMultiplicity(Boolean multiplesAllowed) throws InvalidInputException {
        log.entering(getClass().getName(), "createMultiplicity");
        session.testParameter(multiplesAllowed, 1, null);

        Element element = session.createDeploymentElement("Multiplicity");
        
        Multiplicity mfs = new MultiplicityImpl(element, session);
        mfs.setMultiplesAllowed(multiplesAllowed);
        
        log.exiting(getClass().getName(), "createMultiplicity");
        return mfs;
    }

    public Feature createFeatureWithRequiredFeatures(String id, Collection<String> features)
    		throws InvalidInputException {
        log.entering(getClass().getName(), "createFeatureWithRequiredFeatures");
        session.testParameter(id, 1, null);
        session.testParameter(features, 2, null);

        Element element =
			session.createDeploymentElement("Feature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setRequiredFeature(features);

        log.exiting(getClass().getName(), "createFeatureWithRequiredFeatures");
        return feature;
    }

    public Feature createFeatureWithContentElements(String id, Collection<String> contentReferences)
    		throws InvalidInputException {
        log.entering(getClass().getName(), "createFeatureWithContentElements");
        session.testParameter(id, 1, null);
        session.testParameter(contentReferences, 2, null);

        Element element = session.createDeploymentElement("Feature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setContentElements(contentReferences);
        
        log.exiting(getClass().getName(), "createFeatureWithContentElements");
        return feature;
    }

    public Feature createFeatureWithNestedFeatures(String id, Collection<NestedFeature> features)
    		throws InvalidInputException {
        log.entering(getClass().getName(), "createFeatureWithNestedFeatures");
        session.testParameter(id, 1, null);
        session.testParameter(features, 1, null);
        
        Element element = session.createDeploymentElement("Feature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setNestedFeatures(features);
        
        log.exiting(getClass().getName(), "createFeatureWithNestedFeatures");
        return feature;
    }

    public Feature createFeatureWithDerivedVariables(String id, Collection<DerivedVariable> derivedVariables)
    		throws InvalidInputException {
        log.entering(getClass().getName(), "createFeatureWithDerivedVariables");
        session.testParameter(id, 1, null);
        session.testParameter(derivedVariables, 1, null);

        Element element = session.createDeploymentElement("Feature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setVariables(derivedVariables);
        
        log.exiting(getClass().getName(), "createFeatureWithDerivedVariables");
        return feature;
    }

    public Feature createFeatureWithPackageFeatureReferences(String id,
    		Collection<PackageFeatureReference> packageFeatureReferences) throws InvalidInputException {
        log.entering(getClass().getName(), "createFeatureWithPackageFeatureReferences");
        session.testParameter(id, 1, null);
        session.testParameter(packageFeatureReferences, 1, null);
        
        Element element = session.createDeploymentElement("Feature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setPackageFeatureReferences(packageFeatureReferences);
        
        log.exiting(getClass().getName(), "createFeatureWithPackageFeatureReferences");
        return feature;
    }

    public NestedFeature createNestedFeatureWithNestedFeatures(String id, Collection<NestedFeature> features)
    		throws InvalidInputException {
        log.entering(getClass().getName(), "createNestedFeatureWithNestedFeatures");
        session.testParameter(id, 1, null);
        session.testParameter(features, 1, null);

        Element element = session.createDeploymentElement("NestedFeature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setNestedFeatures(features);
        
        log.exiting(getClass().getName(), "createNestedFeatureWithNestedFeatures");
        return feature;
    }

    public NestedFeature createNestedFeatureWithContentElements(String id, Collection<String> contentReferences)
    		throws InvalidInputException {
        log.entering(getClass().getName(), "createNestedFeatureWithContentElements");
        session.testParameter(id, 1, null);
        session.testParameter(contentReferences, 2, null);

        Element element = session.createDeploymentElement("NestedFeature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setContentElements(contentReferences);

        log.exiting(getClass().getName(), "createNestedFeatureWithContentElements");
        return feature;
    }

    public NestedFeature createNestedFeatureWithRequiredFeatures(String id, Collection<String> features)
    		throws InvalidInputException {
        log.entering(getClass().getName(), "createNestedFeatureWithRequiredFeatures");
        session.testParameter(id, 1, null);
        session.testParameter(features, 2, null);

        Element element = session.createDeploymentElement("NestedFeature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setRequiredFeature(features);
        
        log.exiting(getClass().getName(), "createNestedFeatureWithRequiredFeatures");
        return feature;
    }

    public NestedFeature createNestedFeatureWithDerivedVariables(String id,
    		Collection<DerivedVariable> derivedVariables) throws InvalidInputException {
        log.entering(getClass().getName(), "createNestedFeatureWithDerivedVariables");
        session.testParameter(id, 1, null);
        session.testParameter(derivedVariables, 1, null);

        Element element = session.createDeploymentElement("NestedFeature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setVariables(derivedVariables);

        log.exiting(getClass().getName(), "createNestedFeatureWithDerivedVariables");
        return feature;
    }

    public NestedFeature createNestedFeatureWithPackageFeatureReferences(String id,
    		Collection<PackageFeatureReference> packageFeatureReferences) throws InvalidInputException {
        log.entering(getClass().getName(), "createNestedFeatureWithPackageFeatureReferences");
        session.testParameter(id, 1, null);
        session.testParameter(packageFeatureReferences, 1, null);

        Element element = session.createDeploymentElement("NestedFeature");
        
        Feature feature = new FeatureImpl(element, session);
        
        feature.setFeatureID(id);
        feature.setPackageFeatureReferences(packageFeatureReferences);

        log.exiting(getClass().getName(), "createNestedFeatureWithPackageFeatureReferences");
        return feature;
    }
    
    public PackageFeatureReference createPackageFeatureReference(String contentElementRef, String packageFeatureRef)
    		throws InvalidInputException {
        log.entering(getClass().getName(), "createPackageFeatureReference");
        session.testParameter(contentElementRef, 1, null);
        session.testParameter(packageFeatureRef, 2, null);

        Element element = session.createDeploymentElement("PackageFeatureReference");
        
        PackageFeatureReference pfr = new PackageFeatureReferenceImpl(element, session);
        
        pfr.setContentElementRef(contentElementRef);
        pfr.setPackageFeatureRef(packageFeatureRef);
        
        log.exiting(getClass().getName(), "createPackageFeatureReference");
        return pfr;
    }

    public ConstrainedResource createConstrainedResource(String resourceRef) throws InvalidInputException {
        log.entering(getClass().getName(), "createConstrainedResource");
        session.testParameter(resourceRef, 1, null);
        
        Element element = session.createDeploymentElement("ConstrainedResource");
        
        ConstrainedResource cr = new ConstrainedResourceImpl(element, session);
        cr.setResourceRef(resourceRef);

        log.exiting(getClass().getName(), "createConstrainedResource");
        return cr;
    }

    public ContentSelectionFeature createContentSelectionFeature(String feature) throws InvalidInputException {
        log.entering(getClass().getName(), "createContentSelectionFeature");
        session.testParameter(feature, 1, null);

        Element element = session.createDeploymentElement("Feature");
        
        ContentSelectionFeature csf = new ContentSelectionFeatureImpl(element, session);
        csf.setFeature(feature);
        
        log.exiting(getClass().getName(), "createContentSelectionFeature");
        return csf;
    }
}