/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.util.ConversionUtil;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.BaseFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DeploymentDescriptor;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.FixIdentity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Identity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.MaintenanceInformation;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ManufacturerInformation;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.PackageContent;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.PackageDescriptor;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.PackageIdentity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Resource;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResourceMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResultingChangeMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ResultingResourceMap;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.TranslatableString;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public class BaseFactoryImpl extends SPIObject implements BaseFactory {

	public BaseFactoryImpl(SPISessionImpl sessionImpl) {
		super(sessionImpl);
	}
	
	public DeploymentDescriptor createDeploymentDescriptor (
			String schemaVersion, byte[] descriptorID, Date lastModified,
			ContentUnit content, Collection<Resource> resources) {
		log.entering(getClass().getName(), "createDeploymentDescriptor");

		session.testParameter(schemaVersion, 1, null);
		session.testParameter(descriptorID, 2, null);
		session.testParameter(lastModified, 3, null);
		session.testParameter(content, 4, null);
		session.testParameter(resources, 5, null);
		
		Document ddDocument = session.getDocumentBuilder().newDocument();
		Element ddRoot = session.createDeploymentElement("DeploymentDescriptor");
		
        ddRoot.setAttribute("xmlns:sdd-common", "http://docs.oasis-open.org/sdd/ns/common");
		ddRoot.setAttribute("schemaVersion", schemaVersion);
		
		if (descriptorID != null && descriptorID.length != 16) {
			throw new InvalidInputException(2, InvalidInputException.INVALID_VALUE);
		}
		
		ddRoot.setAttribute("descriptorID", ConversionUtil.bytesToHex(descriptorID));
		
        DateFormat tzf = new SimpleDateFormat("Z");
        String timeZone = tzf.format(lastModified);
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ss");
        String formattedDate = df.format(lastModified) + timeZone.substring(0, 3) + ":" + timeZone.substring(3);

		ddRoot.setAttribute("lastModified", formattedDate);
		
		session.insertNode(ddDocument, ddRoot, null);
        
        DeploymentDescriptor dd = new DeploymentDescriptorImpl(ddDocument, session);
        dd.setResources(resources);
        dd.setContentUnit(content);        

		log.exiting(getClass().getName(), "createDeploymentDescriptor");
		
		return dd;
	}

	public FixIdentity createFixIdentity(String fixName) {
		log.entering(getClass().getName(), "createFixIdentity");
		
		session.testParameter(fixName, 1, null);
		
		Element fixIdentityElement = session.createCommonElement("Fix");
		FixIdentity fixIdentity = new FixIdentityImpl(fixIdentityElement, session);
		fixIdentity.setFixName(fixName);
		
		log.exiting(getClass().getName(), "createFixIdentity");
		
		return fixIdentity;
	}

	public Identity createIdentity(TranslatableString name) {
		log.entering(getClass().getName(), "createIdentity");
		
		session.testParameter(name, 1, null);
		
		Element identityElement = session.createCommonElement("Identity");
		Identity identity = new IdentityImpl(identityElement, session);
		identity.setName(name);
		
		log.exiting(getClass().getName(), "createIdentity");
		
		return identity;
	}

	public ManufacturerInformation createManufacturerInformation(TranslatableString name) {
		log.entering(getClass().getName(), "createManufacturerInformation");
		
		session.testParameter(name, 1, null);
		
		Element manuInfoElement = session.createCommonElement("Manufacturer");
		ManufacturerInformation manuInfo = new ManufacturerInformationImpl(manuInfoElement, session);
		manuInfo.setName(name);

		log.exiting(getClass().getName(), "createManufacturerInformation");
		
		return manuInfo;
	}
	
	public MaintenanceInformation createMaintenanceInformation() {
		log.entering(getClass().getName(), "createMaintenanceInformation");
		
		Element maintInfoElement = session.createCommonElement("MaintenanceInformation");
		MaintenanceInformation maintInfo = new MaintenanceInformationImpl(maintInfoElement, session);
		
		log.exiting(getClass().getName(), "createMaintenanceInformation");
		
		return maintInfo;
	}

	public MaintenanceInformation createSupersedes() {
		log.entering(getClass().getName(), "createSupersedes");
		
		Element maintInfoElement = session.createCommonElement("Supersedes");
		MaintenanceInformation maintInfo = new MaintenanceInformationImpl(maintInfoElement, session);
		
		log.exiting(getClass().getName(), "createSupersedes");
		
		return maintInfo;
	}
	
	public PackageDescriptor createPackageDescriptor(String schemaVersion,
            byte[] descriptorID, Date lastModified, PackageIdentity identity,
            Collection<PackageContent> packageContents) {
        log.entering(getClass().getName(), "createPackageDescriptor");
        
        session.testParameter(schemaVersion, 1, null);
        session.testParameter(descriptorID, 2, null);
        session.testParameter(lastModified, 3, null);
        session.testParameter(identity, 4, null);
        session.testParameter(packageContents, 5, null);
        
		Document pdDocument = session.getDocumentBuilder().newDocument();
        Element pdRoot = session.createDeploymentElement("DeploymentDescriptor");
        
        pdRoot.setAttribute("xmlns:sdd-common", "http://docs.oasis-open.org/sdd/ns/common");
        pdRoot.setAttribute("schemaVersion", schemaVersion);
        pdRoot.setAttribute("descriptorID", new String(descriptorID));
        
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");
        pdRoot.setAttribute("lastModified", df.format(lastModified));
        
        session.insertNode(pdDocument, pdRoot, null);
		
        PackageDescriptor pd = new PackageDescriptorImpl(pdDocument, session);
        pd.setPackageContents(packageContents);
        pd.setPackageIdentity(identity);

        log.exiting(getClass().getName(), "createPackageDescriptor");
        
        return pd;
    }

    public PackageContent createPackageContent(String contentID, String pathName) {
        log.entering(getClass().getName(), "createPackageContent");
        
        session.testParameter(contentID, 1, null);
        session.testParameter(pathName, 2, null);
        
        Element pkgContentElement = session.createPackageElement("Content");
        PackageContent pkgContent = new PackageContentImpl(pkgContentElement, session);
        pkgContent.setContentID(contentID);
        pkgContent.setPathname(pathName);
        
        log.exiting(getClass().getName(), "createPackageContent");

        return pkgContent;
    }

	public PackageIdentity createPackageIdentity() {
		log.entering(getClass().getName(), "createPackageIdentity");
		
		Element pkgIdentityElement = session.createPackageElement("PackageIdentity");
		PackageIdentity pkgIdentity = new PackageIdentityImpl(pkgIdentityElement, session);
		
		log.exiting(getClass().getName(), "createPackageIdentity");

		return pkgIdentity;
	}
	
	public Resource createHostedResource(String resourceID, String resourceType) {
		log.entering(getClass().getName(), "createHostedResource");
		
		Resource resource = createResource(resourceID, resourceType, true);
		
		log.exiting(getClass().getName(), "createHostedResource");
		
		return resource;
	}
	
	public Resource createResource(String resourceID, String resourceType) {
		log.entering(getClass().getName(), "createResource");
		
		Resource resource = createResource(resourceID, resourceType, false);
		
		log.exiting(getClass().getName(), "createResource");
		
		return resource;
	}

	private Resource createResource(String resourceID, String resourceType, boolean isHostedResource) {		
		session.testParameter(resourceID, 1, null);
		session.testParameter(resourceType, 2, null);
		
		Element resourceElement = null;
		if (isHostedResource) {
			resourceElement = session.createDeploymentElement("HostedResource");
		}
		else {
			resourceElement = session.createDeploymentElement("Resource");
		}
		
		Resource resource = new ResourceImpl(resourceElement, session);
		resource.setResourceID(resourceID);
		resource.setResourceType(resourceType);

		return resource;
	}
	
	public ReferencedPackage createReferencedPackage(String contentRef, String id) {
		log.entering(getClass().getName(), "createReferencedPackage");
		
		ReferencedPackage referencedPackage = createReferencedPackage(contentRef, id, "ReferencedPackage");
		
		log.exiting(getClass().getName(), "createReferencedPackage");
		
		return referencedPackage;
	}
	
	public ReferencedPackage createContainedPackage(String contentRef, String id) {
		log.entering(getClass().getName(), "createContainedPackage");
		
		ReferencedPackage referencedPackage = createReferencedPackage(contentRef, id, "ContainedPackage");
		
		log.exiting(getClass().getName(), "createContainedPackage");
		
		return referencedPackage;
	}
	
	public ReferencedPackage createContainedLocalizationPackage(String contentRef, String id) {
		log.entering(getClass().getName(), "createContainedLocalizationPackage");
		
		ReferencedPackage referencedPackage = createReferencedPackage(contentRef, id, "ContainedLocalizationPackage");
		
		log.exiting(getClass().getName(), "createContainedLocalizationPackage");
		
		return referencedPackage;
	}
	
    private ReferencedPackage createReferencedPackage(String contentRef, String id, String tagName) {
        session.testParameter(contentRef, 1, null);
        session.testParameter(id, 2, null);
        
        Element refedPkgElement = session.createDeploymentElement(tagName);
        ReferencedPackage referencedPkg = new ReferencedPackageImpl(refedPkgElement, session);
        referencedPkg.setReferencedPackageID(id);
        referencedPkg.setContentReference(contentRef);
        
        return referencedPkg;
    }

    public ResultingChangeMap createResultingChangeMap(String resourceRef) {
        log.entering(getClass().getName(), "createResultingChangeMap");
        
        session.testParameter(resourceRef, 1, null);
        
        Element resultingChangeMapElement = session.createDeploymentElement("ResultingChangeMap");
        ResultingChangeMap resultingChangeMap = new ResultingChangeMapImpl(resultingChangeMapElement, session);
        resultingChangeMap.setResourceReference(resourceRef);

        log.exiting(getClass().getName(), "createResultingChangeMap");
        
        return resultingChangeMap;
    }
    
    public ResultingResourceMap createResultingResourceMap(String resourceRef) {
        log.entering(getClass().getName(), "createResultingResourceMap");
        
        session.testParameter(resourceRef, 1, null);
        
        Element resultingResMapElement = session.createDeploymentElement("ResultingResourceMap");
        ResultingResourceMap resultingResourceMap = new ResultingResourceMapImpl(resultingResMapElement, session);
        resultingResourceMap.setResourceReference(resourceRef);

        log.exiting(getClass().getName(), "createResultingResourceMap");
        
        return resultingResourceMap;
    }

    public ResourceMap createRequiredResourceMap(String resourceRef) {
        log.entering(getClass().getName(), "createRequiredResourceMap");
        
        session.testParameter(resourceRef, 1, null);
        
        Element resourceMapElement = session.createDeploymentElement("RequiredResourceMap");
        ResourceMap resourceMap = new ResourceMapImpl(resourceMapElement, session);
        resourceMap.setResourceReference(resourceRef);

        log.exiting(getClass().getName(), "createRequiredResourceMap");
        
        return resourceMap;
    }
}
