/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.xmldsig.Signature;

/**
 * The root element of a package descriptor XML document is PackageDescriptor. PackageDescriptor includes elements that describe the 
 * package identity and the contents that make up the package. The PackageDescriptor includes the associated deployment descriptor 
 * XML document by defining a Content element with a purpose attribute set to deploymentDescriptor.
 */
public interface PackageDescriptor extends Descriptor {

	/**
	 * Gets the Package Contents.
	 * 
	 * The Contents element defines a list of one or more Content elements describing all the files that are part of the package. 
	 * All files in the package MUST be defined in Contents.
	 *  
	 * @return A collection of package contents.
	 */
    Collection<PackageContent> getPackageContents();
	
	/**
	 * Gets the PackageIdentity.
	 * 
	 * The PackageIdentity element provides identity information about the software package that can be used by the consumer of the 
	 * package for deployment planning or aggregation of the package into a larger solution.
	 *  
	 * @return A PackageIdentity object that contains identity information for the software package.
	 */
	PackageIdentity getPackageIdentity();
	
	/**
	 * Sets the PackageIdentity.
	 * 
	 * The PackageIdentity element provides identity information about the software package that can be used by the consumer of the 
	 * package for deployment planning or aggregation of the package into a larger solution.
	 * 
	 * @param packageIdentity A PackageIdentity object that contains identity information for the software package.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setPackageIdentity(PackageIdentity packageIdentity);

	/**
	 * Sets the Package Contents.
	 * 
	 * The Contents element defines a list of one or more Content elements describing all the files that are part of the package. 
	 * All files in the package MUST be defined in Contents.
	 *  
	 * @param packageContents A collection of package contents.
	 * @throws InvalidInputException if any of the parameters is null
	 */
	void setPackageContents(Collection<PackageContent> packageContents);
	
	/**
	 * Gets the Signature.
	 * 
	 * The package descriptor and each file in the package MAY be digitally signed. It is RECOMMENDED that they be digitally signed by 
	 * using an XML-Signature [XMLDSIG-CORE].<br><br>
	 * The signature element is an enveloped signature over the SDD package. Note that each Content element included in the package is 
	 * digitally signed indirectly via this digest. Files can also be individually signed in the Content element.
	 * 
	 * @return A Signature object for the package descriptor.
	 */
	Signature getSignature();

	/**
	 * Sets the Signature.
	 * 
	 * The package descriptor and each file in the package MAY be digitally signed. It is RECOMMENDED that they be digitally signed by 
	 * using an XML-Signature [XMLDSIG-CORE].<br><br>
	 * The signature element is an enveloped signature over the SDD package. Note that each Content element included in the package is 
	 * digitally signed indirectly via this digest. Files can also be individually signed in the Content element.
	 * 
	 * @param signature A Signature object for the package descriptor.
	 */
	void setSignature(Signature signature);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
