/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.smlvalidators;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;

import org.eclipse.cosmos.rm.internal.validation.artifacts.ReferenceDescriptor;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.ReferenceExtractor;
import org.eclipse.cosmos.rm.internal.validation.reference.ReferenceContext;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.eclipse.cosmos.rm.internal.validation.reference.XPointer;
import org.eclipse.cosmos.rm.internal.validation.reference.XPointerExpression;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Node;

/**
 * A Reference validator is used to display warning messages for
 * any dangling references that are found. 
 * 
 * @author Ali Mehregani
 * @author John Arwe
 */
public class ReferenceValidator extends AbstractSMLValidator
{
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.launcher.IValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);
		
		// Register the required data builders
		DataBuilderRegistry databuilderRegistry = DataBuilderRegistry.getInstanceLevelRegistry();
		databuilderRegistry.registerDataStructureBuilder(ReferenceExtractor.ID, new ReferenceExtractor());
	}
	
	public boolean validate()
	{
		setTaskName(SMLValidationMessages.validationReferences);
		
		// For each reference
		ReferenceExtractor referenceExtractor = (ReferenceExtractor)DataBuilderRegistry.getInstanceLevelRegistry().getDataStructureBuilder(ReferenceExtractor.ID);
		List<ReferenceContext> referenceContextList = referenceExtractor.getContextReferences();
		
		Node lastNode = null;
		ReferenceDescriptor lastReference = null;
		
		
		// For each reference context
		for (int i = 0, refContextCount = referenceContextList.size(); i < refContextCount; i++)
		{
			ReferenceContext referenceContext = referenceContextList.get(i);
			
			// Don't attempt to resolve the reference if this is a nilref
			if (referenceContext.isNilref())
			{
				continue;
			}
			
			// Attempt to resolve the node of the containing document
			Node contextNode = SMLValidatorUtil.locateDocumentNode(referenceContext.getAliases(), referenceContext.getDocumentPosition());			
			List<ReferenceDescriptor> referenceDescriptors = referenceContext.getReferenceDescriptors();
			for (int j = 0, referenceDescCount = referenceDescriptors.size(); j < referenceDescCount; j++)
			{
				ReferenceDescriptor currentReference = null;
				try
				{
					currentReference = referenceDescriptors.get(j);			
					URIReference uriReference = new URIReference(contextNode, currentReference.getReference(), currentReference.getBaseURI());
					Node context = uriReference.retrieveDocumentDOM();
					Node node = null;
					if (context == null)
					{
						showWarning(currentReference , SMLValidationMessages.warningUnresolvedReferenceDocument, 
								new String[] {currentReference.getReference(), uriReference.getDocumentReference() } );						
					}
					else
					{
						// Evaluate the XPointer expression of the reference (if one exists)
						String fragment = uriReference.getFragment();

						if (fragment != null && fragment.length() > 0)
						{
							XPointerExpression xpointerExpression = XPointer.compile(uriReference.getFragment());					
							node = (Node)xpointerExpression.evaluate(referenceContext.getNamespaceContext(), context);
							if (node == null)
							{
								showWarning(currentReference , SMLValidationMessages.warningUnresolvedReferenceFragment, 
										new String[] {currentReference.getReference()} );
							}
						}
					}

					if (j == 0)
					{
						lastNode = node;
						lastReference = currentReference;
					}
					else if ((lastNode == null && node != null) || (lastNode != null && node == null)
							|| (lastNode != null && node != null && lastNode.equals(node) == false))
					{
						showError(lastReference, currentReference);
						if (shouldAbortOnError()) 
						{
							return false;
						}
					} 
				} 
				catch (URISyntaxException e)
				{
					String message = e.getMessage().trim();
					if (message.endsWith(":"))
					{
						message = message.substring(0, message.length()-1);
					}	
					message = NLS.bind(SMLValidationMessages.errorReferenceResolving, 
							new String [] { currentReference.getReference() }) + " " + message;
					showError (message, currentReference);
				} 			
				catch (Exception e)
				{	//	Unexpected exception, include stack trace
					showError (SMLValidationMessages.errorReferenceResolving, currentReference, e);
				} 			
			}
		}

		return true;
	}

	private void showError(ReferenceDescriptor lastReference, ReferenceDescriptor currentReference)
	{
		showError (NLS.bind(SMLValidationMessages.errorInconsistentReference, 
				 new String[]{lastReference.getReference(), currentReference.getReference()}), lastReference);
	}
	
	private void showError(String message, ReferenceDescriptor uriReference, Exception e)
	{
		StringWriter sw = new StringWriter();
		e.printStackTrace(new PrintWriter(sw));			
		showMessage(ValidationMessageFactory.createErrorMessage(
				uriReference.getFilePath(), 
				uriReference.getLineNumber(), 
				NLS.bind(message, uriReference.getReference()) + " " + sw.getBuffer().toString()));
		
	}
	
	private void showError (String message, ReferenceDescriptor reference)
	{
		showMessage(ValidationMessageFactory.createErrorMessage(
				reference.getFilePath(), 
				reference.getLineNumber(), 
				message));
	}

	private void showWarning(ReferenceDescriptor uriReference, String message, String[] fillIns)
	{
		showMessage(ValidationMessageFactory.createWarningMessage(
				uriReference.getFilePath(), 
				uriReference.getLineNumber(), 
				NLS.bind(message, fillIns)));	
	}
	
	private void showMessage(IValidationMessage validationMessage)
	{
		IValidationOutput<String, Object> output = getValidationOutput();
		output.reportMessage(validationMessage);
	}
}
