/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 * CA - added string encoding for XML special characters
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.editor;


import java.util.Arrays;
import java.util.List;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.cosmos.rm.internal.smlif.SMLImages;
import org.eclipse.cosmos.rm.internal.smlif.SMLPlugin;
import org.eclipse.cosmos.rm.internal.smlif.actions.ValidateResourceAction;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLCommonUtil;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLMessages;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLPage;
import org.eclipse.cosmos.rm.internal.smlif.importing.ui.ImportFromSMLIFWizard;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.XMLInternalUtility;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPathEditorInput;
import org.eclipse.ui.IURIEditorInput;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Hyperlink;
import org.eclipse.ui.forms.widgets.Section;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * The overview page of the SML-IF editor
 * 
 * @author Ali Mehregani
 * @modified David Whiteman
 */
public class OverviewPage extends AbstractFormPage
{
	/**
	 * The id of this page
	 */
	public static final String ID = SMLPlugin.PLUGIN_ID + ".OverviewPage"; //$NON-NLS-1$

	/**
	 * The description text field
	 */
	private Text descriptionText;
	
	/**
	 * The name text field
	 */
	private Text nameText;
	
	/**
	 * The version text field
	 */
	private Text versionText;
	
	/**
	 * The display name text field
	 */
	private Text displayNameText;
	
	/**
	 * The base uri text field
	 */
	private Text baseURIText;
	
	/**
	 * Used to disable the modify listeners of the text fields
	 */
	private boolean disableListener;
	
	/**
	 * show editor errors only once
	 */
	private boolean showError = true;
	
	
	/**
	 * Constructor
	 * 
	 * @param editor The form editor
	 */
	public OverviewPage(FormEditor editor)
	{
		super(editor, ID, SMLMessages.editorOverviewTitle);
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.smlif.editor.AbstractFormPage#createLeftColumn(org.eclipse.swt.widgets.Composite)
	 */
	protected void createLeftColumn(Composite parent)
	{
		getManagedForm().addPart(((SMLIFEditor)getEditor()).getSmlifFormPart());
		FormToolkit toolkit = getManagedForm().getToolkit();
		Section section = createSection(parent);
		
		Composite sectionClient = toolkit.createComposite(section);
		GridLayout sectionClientGL = new GridLayout(2, false);
		sectionClientGL.marginTop = 10;
		sectionClient.setLayout(sectionClientGL);
		sectionClient.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		nameText = createTextField(sectionClient, SMLMessages.editorName, null, this);
		versionText = createTextField(sectionClient, SMLMessages.editorVersion, null, this);
		displayNameText = createTextField(sectionClient, SMLMessages.editorDisplayName, null, this);
		baseURIText = createTextField(sectionClient, SMLMessages.editorBaseURI, null, this);
		SMLCommonUtil.addURIValidator(baseURIText, new SMLPage(this));
		descriptionText = createTextField(sectionClient, SMLMessages.editorDescription, true, null, this);
		
		makeSectionReady (section, sectionClient, SMLMessages.editorGeneralInformation);
	}



	/**
	 * @see org.eclipse.cosmos.rm.internal.smlif.editor.AbstractFormPage#createRightColumn(org.eclipse.swt.widgets.Composite)
	 */
	protected void createRightColumn(Composite parent)
	{
		FormToolkit toolkit = getManagedForm().getToolkit();
		Section section = createSection(parent);
		
		Composite sectionClient = toolkit.createComposite(section);
		GridLayout gl = new GridLayout(2, false);
		gl.marginTop = 10;
		sectionClient.setLayout(gl);
		sectionClient.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		Label linkImage = toolkit.createLabel(sectionClient, IValidationConstants.EMPTY_STRING);
		linkImage.setImage(SMLImages.INSTANCE.getImage(SMLImages.VALIDATE));
		
		Hyperlink validateLink = toolkit.createHyperlink(sectionClient, SMLMessages.editorValidateDocument, SWT.WRAP);
		FontData[] fontData = validateLink.getFont().getFontData();
		for (int i = 0; i < fontData.length; i++)
		{
			fontData[i].setStyle(fontData[i].getStyle() | SWT.BOLD);
		}
		validateLink.setFont(new Font(null,fontData));	
		
		validateLink.addHyperlinkListener(new HyperlinkAdapter()
		{
			public void linkActivated(HyperlinkEvent e)
			{
				IFileStore fileSelection = getInputFile();
				if (fileSelection != null) {
					ValidateResourceAction validateAction = new ValidateResourceAction();
					validateAction.selectionChanged(null, new StructuredSelection(fileSelection));
					validateAction.run(null);
				}
			}
		});
		
		linkImage = toolkit.createLabel(sectionClient, IValidationConstants.EMPTY_STRING);
		linkImage.setImage(SMLImages.INSTANCE.getImage(SMLImages.INSTANCES));

		Hyperlink importLink = toolkit.createHyperlink(sectionClient, SMLMessages.editorImportModel, SWT.WRAP);
		importLink.setFont(new Font(null,fontData));
		importLink.addHyperlinkListener(new HyperlinkAdapter()
		{
			public void linkActivated(HyperlinkEvent e)
			{
				IFileStore fileSelection = getInputFile();
				if (fileSelection != null) {
					IWorkbenchWindow window = getSite().getWorkbenchWindow();
					ImportFromSMLIFWizard wizard = new ImportFromSMLIFWizard(nameText.getText());
					wizard.init(window.getWorkbench(), new StructuredSelection(fileSelection));
					WizardDialog dialog = new WizardDialog(window.getShell(), wizard);
					dialog.open();
				}
			}
		});

		makeSectionReady (section, sectionClient, SMLMessages.editorActionsTitle);
	}
	
	public void setActive(boolean active)
	{	
		super.setActive(active);
		
		if (!active)
			return;
		
		disableListener = true;
		
		/* Populate the fields */
		SMLIFEditor editor = (SMLIFEditor)getEditor();	
		
		if(showError)
		{
			showError = false;
			Node rootDocumentNode = editor.getDocumentNode();
			if (rootDocumentNode == null) {
				
				MessageDialog.openError(Display.getCurrent().getActiveShell(), 
						SMLMessages.errorEditorTitle, SMLMessages.errorWrongRootNode);
							
			}
		}
		
		
		String field = editor.retrieveField(SMLIFEditor.TYPE_IDENTITY, ISMLConstants.NAME_ELEMENT);
		nameText.setText(field == null ? IValidationConstants.EMPTY_STRING : field);
		
		field = editor.retrieveField(SMLIFEditor.TYPE_IDENTITY, ISMLConstants.VERSION_ELEMENT);
		versionText.setText(field == null ? IValidationConstants.EMPTY_STRING : field);
		
		field = editor.retrieveField(SMLIFEditor.TYPE_IDENTITY, ISMLConstants.DISPLAY_NAME_ELEMENT);
		displayNameText.setText(field == null ? IValidationConstants.EMPTY_STRING : field);
		
		field = editor.retrieveField(SMLIFEditor.TYPE_IDENTITY, ISMLConstants.BASE_URI_ELEMENT);
		baseURIText.setText(field == null ? IValidationConstants.EMPTY_STRING : field);
		
		field = editor.retrieveField(SMLIFEditor.TYPE_IDENTITY, ISMLConstants.DESCRIPTION_ELEMENT);
		descriptionText.setText(field == null ? IValidationConstants.EMPTY_STRING : field);
		
		disableListener = false;
	}
	
	public void modifyText(ModifyEvent e)
	{		
		if (disableListener)
			return;
		
		SMLIFEditor editor = (SMLIFEditor)getEditor();		
		IDocument document = editor.getSmlifSourcePage().getDocument();
		Node identityNode = SMLCommonUtil.nestedNodeRetrieval(editor.getDocumentNode(), new String[][]{new String[] {ISMLConstants.SMLIF_URI, ISMLConstants.IDENTITY_ELEMENT}}, true, false, null);				
		modifyTextNode(editor, identityNode, 
					e.widget == nameText ? 			ISMLConstants.NAME_ELEMENT :			/* The name field */
					e.widget == versionText ?		ISMLConstants.VERSION_ELEMENT:			/* The version field */
					e.widget == displayNameText ? 	ISMLConstants.DISPLAY_NAME_ELEMENT : 	/* Display name */
					e.widget == baseURIText ? 		ISMLConstants.BASE_URI_ELEMENT : 		/* The base URI */
													ISMLConstants.DESCRIPTION_ELEMENT, 		/* Description */
					((Text)e.widget).getText());
		
		try
		{
			StringBuffer buffer = new StringBuffer();
			XMLInternalUtility.serializeNode(buffer, editor.getDocumentNode().getOwnerDocument());
			document.set(buffer.toString());
		} 
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}


	private void modifyTextNode(SMLIFEditor editor, Node node, String elementName, String newValue)
	{
		// TODO this probably needs to be optimized in multiple ways.  It's already slow that
		// we modify text nodes with every character typed, but the recursive parent node lookup
		// could be improved by caching previous data		
		Node siblingToInsertBefore = getSiblingToInsertBefore(node, elementName);
		boolean append = (siblingToInsertBefore == null);
		Node childNode = SMLCommonUtil.nestedNodeRetrieval(node, new String[][]{new String[] {ISMLConstants.SMLIF_URI, elementName}}, true, append, siblingToInsertBefore);
		if (childNode == null)
			return;
		
		/* Remove all children of the node */
		NodeList children = childNode.getChildNodes();
		for (int i = 0, childCount = children.getLength(); i < childCount; i++)
		{
			childNode.removeChild(children.item(i));
		}
		
		// encodes XML special characters, if any
		String encodedString = XMLInternalUtility.encodeSpecialCharacters(newValue);

		childNode.appendChild(childNode.getOwnerDocument().createTextNode(newValue == null ? IValidationConstants.EMPTY_STRING : encodedString));
	}

	/**
	 * Recursively find the sibling that should come before the element named
	 * elementName following the order for the <identity> element, creating nodes
	 * when they do not exist.
	 */
	private Node getSiblingToInsertBefore(Node parentNode, String elementName) {
		java.util.List<String> identityNodeList = getIdentityNodeList();
		int elementIndex = identityNodeList.indexOf(elementName);
		if ((elementIndex < 0) || (elementIndex >= identityNodeList.size() - 1)) return null;
		String siblingNodeName = identityNodeList.get(elementIndex + 1);
		Node siblingToInsertSiblingBefore = getSiblingToInsertBefore(parentNode, siblingNodeName);
		boolean append = (siblingToInsertSiblingBefore == null);
		return SMLCommonUtil.nestedNodeRetrieval(parentNode, new String[][]{new String[] {ISMLConstants.SMLIF_URI, siblingNodeName}}, true, append, siblingToInsertSiblingBefore);
	}

	/**
	 * Answer the list of Identity element node names
	 * in the order prescribed by the SML-IF spec
	 * @return list of element names
	 */
	private List<String> getIdentityNodeList() {
		String[] array = { ISMLConstants.NAME_ELEMENT,
				ISMLConstants.VERSION_ELEMENT,
				ISMLConstants.DISPLAY_NAME_ELEMENT,
//				ISMLConstants.BASE_URI_ATTRIBUTE,
				ISMLConstants.DESCRIPTION_ELEMENT };
		return Arrays.asList(array);
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.smlif.editor.AbstractFormPage#updateContent()
	 */
	protected void updateContent()
	{
		if (isActive())
			setActive(true);
	}


	private IFileStore getInputFile() {
		IEditorInput input = getEditorInput();
		if (input == null)
		{
			MessageDialog.openError(SMLPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), 
					SMLMessages.errorNoInputTitle, SMLMessages.errorNoInput);
			return null;
		}
		IFileStore fileSelection = null;
		// 186396: Validating external SML-IF file causes class cast exception
		// https://bugs.eclipse.org/bugs/show_bug.cgi?id=186396
		try {
			if (input instanceof IURIEditorInput) {
				IFileStore file = EFS.getStore(((IURIEditorInput)input).getURI());
				fileSelection = file;
			} else {
				IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(((IPathEditorInput)input).getPath());
				fileSelection = EFS.getStore(file.getRawLocationURI());
			}
		} catch (CoreException e) {
			
		}
		return fileSelection;
	}
}
