/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.dc.provisional.cmdbf.services.common;

import java.util.Map;

import org.eclipse.cosmos.dc.provisional.cmdbf.services.common.CMDBfServiceException;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.output.artifacts.IAccepted;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.output.artifacts.IDeclined;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.output.artifacts.IInstanceResponse;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.registration.transform.output.artifacts.RegisterOutputArtifactFactory;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IGraphElement;
import org.eclipse.cosmos.dc.provisional.cmdbf.services.transform.artifacts.IInstanceId;

/**
 * An abstract class that implements common functionality needed
 * by most INotificationHandler types.  Adopters can either extend
 * this class or provide a direct implementation
 * 
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractNotificationHandler implements INotificationHandler
{
	/**
	 * The initialization data
	 */
	private Map<String, Object> init;
	
	/**
	 * The runtime options
	 */
	private Map<String, Object> runtimeOptions;
	
	/**
	 * The ID of an MDR that this handler is associated with
	 */
	public String mdrId;
	
	/**
	 * The local id of an item/relationship that is being processed
	 */
	public String localId;
	
	
	/**
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.cmdbfservice.IQueryHandler#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init) throws CMDBfServiceException
	{
		this.init = init;
	}

	
	/**
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.cmdbfservice.IQueryHandler#isInitialized()
	 */
	public boolean isInitialized()
	{
		return init != null;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.cmdbfservice.IQueryHandler#getRuntimeOptions()
	 */
	public Map<String, Object> getRuntimeOptions()
	{
		return runtimeOptions;
	}

	
	/**
	 * @see org.eclipse.cosmos.dc.provisional.cmdbf.services.cmdbfservice.IQueryHandler#setRuntimeOptions(java.util.Map)
	 */
	public void setRuntimeOptions(Map<String, Object> options)
	{
		this.runtimeOptions = options;
	}
	
	/**
	 * Returns the value stored in the init map
	 * 
	 * @param key The key of the value
	 * @return The value stored
	 */
	public Object getValue(Object key)
	{
		return init.get(key);
	}
	
	/**
	 * Equivalent to {@link #createDeclinedResponse(String[0])}
	 * 
	 * @return A declined response without any 
	 * associated reason
	 */
	protected IInstanceResponse createDeclinedResponse()
	{
		return createDeclinedResponse(new String[0]);
	
	}
		
	/**
	 * Equivalent to {@link #createDeclinedResponse(String[]{reason})}
	 * 
	 * @param reason An associated reason to the declined response
	 * @return A declined response with an 
	 * associated reason
	 */
	protected IInstanceResponse createDeclinedResponse(String reason)
	{
		return createDeclinedResponse(new String[]{reason});	
	}
	
	
	/**
	 * Creates a declined response with associated set 
	 * of reasons
	 * 
	 * @param reason Associated sets of reasons to the declined response
	 * @return A declined response with associated set 
	 * of reasons
	 */
	protected IInstanceResponse createDeclinedResponse(String[] reasons)
	{
		IInstanceResponse response = createResponse();
		IDeclined declined = RegisterOutputArtifactFactory.getInstance().createDeclined();
		
		if (reasons != null)
		{
			for (int i = 0; i < reasons.length; i++)
			{
				declined.addReason(reasons[i]);
			}
		}
		response.setDeclined(declined);			
		return response;	
	}
	
	
	/**
	 * Equivalent to {@link #createAcceptResponse(IInstanceId[0])}
	 * 
	 * @return An accepted response without any alternative
	 * instance ids
	 */
	protected IInstanceResponse createAcceptResponse()
	{
		return createAcceptResponse (new IInstanceId[0]);
	}
	
	
	/**
	 * Equivalent to {@link #createAcceptResponse(IInstanceId[]{alternativeInstanceId})}
	 * 
	 * @param alternativeInstanceId An alternative instance ID
	 * @return An accepted response with an associated
	 * alternative instance id
	 */
	protected IInstanceResponse createAcceptResponse(IInstanceId alternativeInstanceId)
	{
		return createAcceptResponse (new IInstanceId[]{alternativeInstanceId});
	}
	
	
	/**
	 * Creates an accepted response with associated set of alternative
	 * instance ids.
	 * 
	 * @param alternativeInstanceIds A set of alternative instance IDs
	 * @return An accepted response with associated set of alternative
	 * instance ids.
	 */
	protected IInstanceResponse createAcceptResponse(IInstanceId[] alternativeInstanceIds)
	{
		IInstanceResponse response = createResponse();
		IAccepted accepted = RegisterOutputArtifactFactory.getInstance().createAccepted();
		
		if (alternativeInstanceIds != null)
		{
			for (int i = 0; i < alternativeInstanceIds.length; i++)
			{
				accepted.addAlternateInstanceId(alternativeInstanceIds[i]);
			}
		}
		response.setAccepted(accepted);			
		return response;
	}

	
	private IInstanceResponse createResponse()
	{
		IInstanceResponse response = RegisterOutputArtifactFactory.getInstance().createInstanceResponse();
		IInstanceId instanceId = RegisterOutputArtifactFactory.getInstance().createInstanceId(mdrId, localId);
		response.setInstanceId(instanceId);
		
		return response;
	}


	/**
	 * Returns the MDR ID that this handler is associated with
	 * 
	 * @return The MDR ID
	 */
	protected String getMdrId()
	{
		return mdrId;
	}


	/**
	 * Sets the MDR ID that this handler is associated with
	 * 
	 * @param mdrId The MDR ID
	 */
	protected void setMdrId(String mdrId)
	{
		this.mdrId = mdrId;
	}


	/**
	 * Returns the local ID of an item/relationship 
	 * being processed
	 * 
	 * @return The local ID
	 */
	protected String getLocalId()
	{
		return localId;
	}


	/**
	 * Sets the local ID of an item/relationship 
	 * being processed
	 * 
	 * @param localId The local ID
	 */
	protected void setLocalId(String localId)
	{
		this.localId = localId;
	}
	
	
	/**
	 * Retrieves and returns the first instance id associated
	 * with element
	 * 
	 * @param element The element
	 * @return the first associated instance id or null if none
	 * can be found
	 */
	protected IInstanceId retrieveInstanceId(IGraphElement element)
	{
		IInstanceId[] instanceIds = element.getInstanceIds();
		return instanceIds.length > 0 ? instanceIds[0] : null;		
	}
	
	
	protected void setIds (IGraphElement graphElement)
	{
		IInstanceId[] instanceIds = graphElement.getInstanceIds();
		if (instanceIds != null && instanceIds.length > 0)
		{
			setMdrId(instanceIds[0].getMdrId().toString());
			setLocalId(instanceIds[0].getLocalId().toString());
		}
		else
		{
			setMdrId("");
			setLocalId("");
		}
	}
}
