/**********************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

/**
 * This abstract data builder is used to keep track of 
 * nodes using indices which indicate their path from
 * the document node.
 * 
 * @author Ali Mehregani
 * @author John Arwe
 */
public abstract class AbstractTrackerDataBuilder<T> extends AbstractDataBuilder<T>
{
	/**
	 * Constants representing the begin,
	 * startElement, and closeElement event
	 */
	private static int START = 0x00;
	private static int OPEN = 0x01;
	private static int CLOSE = 0x02;
	
	/**
	 * The current index of the document node
	 */
	private List<Integer> documentNodeInx;
	
	/**
	 * Stores the last event detected
	 */
	private int lastEvent;
	
	/**
	 * The current orphan document index
	 */
	private int orphanInx;
	
	
	public AbstractTrackerDataBuilder()
	{
		documentNodeInx = new ArrayList<Integer>();
		orphanInx = -1;	
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init)
	{
		super.initialize(init);
		documentNodeInx.clear();
		lastEvent = START;
		
		String[] aliases = getAliases();
		orphanInx += (aliases == null || aliases.length <= 0) ? 1 : 0;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String name, Attributes attributes) throws SAXException
	{		
		super.startElement(uri, localName, name, attributes);
		
		if (lastEvent == OPEN || lastEvent == START)
		{
			documentNodeInx.add(0);
		}
		else if (lastEvent == CLOSE)
		{
			int lastInx = documentNodeInx.size() - 1;
			documentNodeInx.set(lastInx, documentNodeInx.get(lastInx) + 1);
		}
		
		lastEvent = OPEN;
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String name) throws SAXException
	{		
		super.endElement(uri, localName, name);
		if (lastEvent == CLOSE)
		{
			documentNodeInx.remove(documentNodeInx.size() - 1);
		}	
		
		lastEvent = CLOSE;
	}
	
	
	protected int[] getNodePath()
	{
		// The indices leading to the context node
		int[] nodePath = new int[documentNodeInx.size()]; 
		for (int j = 0; j < nodePath.length; j++)
		{
			nodePath[j] = ((Integer)documentNodeInx.get(j)).intValue();
		}
		
		return nodePath;
	}
	
	
	/**
	 * A static method used to retrieve the node based on the
	 * indices and the starting node passed in.
	 * 
	 * @param startingNode The starting node
	 * @param indices The indices leading to the node to be returned
	 * @return A node based on the indices and the starting node passed in
	 */
	public static Node findNode(Node startingNode, int[] indices)
	{
		//TODO validateIdentityConstraints uses this.  This code does NOT skip non-element nodes, as SMLValidatorUtil.findNode does, and it probably should.
		Node currentNode = startingNode;
		NodeList nodeList = null;
		for (int i = 1; i < indices.length; i++)
		{
			nodeList = currentNode.getChildNodes();
			if (indices[i] < 0 || indices[i] >= nodeList.getLength())
				return null;
			currentNode = nodeList.item(indices[i]);
		}
		
		return currentNode;
	}


	/**
	 * @return the orphanInx
	 */
	protected int getOrphanInx()
	{
		return orphanInx;
	}
}
