/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.util.Arrays;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.TranslatableString;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.InternalDependency;
import org.w3c.dom.Element;

public class InternalDependency_Test extends TestCase
{
    InternalDependency dep;
    InternalDependency clonedDep;
    Element element;
    SPISessionImpl session;

    public InternalDependency_Test(String name)
    {
        super(name);
    }

    protected void setUp() throws Exception
    {
        super.setUp();
        
        dep = SPISession.DEFAULT_INSTANCE.createRequirementFactory().createInternalDependency("contentRef");

        dep.setContentReferenceOperations(Arrays.asList("op1", "op2"));
        dep.setDescription(new TranslatableString("desc", "descKey"));
        dep.setShortDescription(new TranslatableString("shortDesc", "shortDescKey"));
        dep.setType("depType");
        
        clonedDep = (InternalDependency) dep.clone();
        element = ((SPIDataObject) dep).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetTranslatableStrings()
    {
        Properties props = dep.getTranslatableStrings();
        Properties clonedProps = clonedDep.getTranslatableStrings();
        
        assertEquals(2, props.keySet().size());
        assertEquals(2, clonedProps.keySet().size());
        
        assertEquals("desc", props.get("descKey"));
        assertEquals("shortDesc", props.get("shortDescKey"));
        
        assertEquals("desc", clonedProps.get("descKey"));
        assertEquals("shortDesc", clonedProps.get("shortDescKey"));
    }

    public void testGetContentReference()
    {
        assertEquals("contentRef", dep.getContentReference());
        assertEquals("contentRef", clonedDep.getContentReference());
        assertEquals("contentRef", element.getAttribute("contentElementRef"));
    }

    public void testGetContentReferenceOperations()
    {
        assertEquals("op1", dep.getContentReferenceOperations().toArray(new String[0])[0]);
        assertEquals("op2", dep.getContentReferenceOperations().toArray(new String[0])[1]);

        assertEquals("op1", clonedDep.getContentReferenceOperations().toArray(new String[0])[0]);
        assertEquals("op2", clonedDep.getContentReferenceOperations().toArray(new String[0])[1]);
        
        assertEquals("op1", element.getAttribute("contentElementRefOperation").split(" ")[0]);
        assertEquals("op2", element.getAttribute("contentElementRefOperation").split(" ")[1]);
    }

    public void testGetType()
    {
        assertEquals("depType", dep.getType());
        assertEquals("depType", clonedDep.getType());
        assertEquals("depType", element.getAttribute("type"));
    }

    public void testGetDescription()
    {
        assertEquals("desc", dep.getDescription().getValue());
        assertEquals("descKey", dep.getDescription().getTranslationKey());

        assertEquals("desc", clonedDep.getDescription().getValue());
        assertEquals("descKey", clonedDep.getDescription().getTranslationKey());
        
        assertEquals("desc", ( session.getChildren(element, "Description").toArray(new Element[0])[0]).getTextContent());
        assertEquals("descKey", ( session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testGetShortDescription()
    {
        assertEquals("shortDesc", dep.getShortDescription().getValue());
        assertEquals("shortDescKey", dep.getShortDescription().getTranslationKey());

        assertEquals("shortDesc", clonedDep.getShortDescription().getValue());
        assertEquals("shortDescKey", clonedDep.getShortDescription().getTranslationKey());
        
        assertEquals("shortDesc", ( session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getTextContent());
        assertEquals("shortDescKey", ( session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetContentReference()
    {
        dep.setContentReference("differentRef");
        
        assertEquals("differentRef", dep.getContentReference());
        assertEquals("differentRef", element.getAttribute("contentElementRef"));
        
        try
        {
            dep.setContentReference(null);
            fail("ContentReference set to null");
        }
        catch (Exception e)
        {
        }
    }

    public void testSetContentReferenceOperations()
    {
        dep.setContentReferenceOperations(Arrays.asList("op3"));
        
        assertEquals("op3", dep.getContentReferenceOperations().toArray(new String[0])[0]);
        assertEquals("op3", element.getAttribute("contentElementRefOperation").split(",")[0]);
        		
    }

    public void testSetType()
    {
        dep.setType("differentType");
        
        assertEquals("differentType", dep.getType());
        assertEquals("differentType", element.getAttribute("type"));
    }

    public void testSetDescription()
    {
        dep.setDescription(new TranslatableString("differentDesc", "differentDescKey"));
        
        assertEquals("differentDesc", dep.getDescription().getValue());
        assertEquals("differentDescKey", dep.getDescription().getTranslationKey());

        assertEquals("differentDesc", ( session.getChildren(element, "Description").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentDescKey", ( session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetShortDescription()
    {
        dep.setShortDescription(new TranslatableString("differentShortDesc", "differentShortDescKey"));

        assertEquals("differentShortDesc", dep.getShortDescription().getValue());
        assertEquals("differentShortDescKey", dep.getShortDescription().getTranslationKey());

        assertEquals("differentShortDesc", ( session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentShortDescKey", ( session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }
}
