/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requiredbase.RequiredBase;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;


/**
 * CompositeLocalizationUnitType provides the type definition for all CompositeLocalizationUnit elements in the LocalizationContent 
 * hierarchy. CompositeLocalizationUnit elements define nested localization content elements and metadata that applies to all of 
 * the nested elements.
 */
public interface CompositeLocalizationUnit extends ContentUnit {
    /**
     * Gets the condition.
     *  
     * If the composite and the elements nested beneath it are applicable only in certain environments, a Condition can be defined. 
     * When the Condition is not met, the composite and its nested elements are not in scope.
     * 
     * @return A condition that determines if the CompositeLocalizationUnit is relevant to a particular deployment.
     */
    Condition getCondition();
    /**
     * Gets the required base.
     * 
     * If the processing of all the update artifacts in the nested content elements results in a single resource being updated, 
     * that resource can be defined in the CompositeLocalizationUnits RequiredBase element.
     * 
     * @return A resource that will be updated when the nested elements are processed.
     */
    RequiredBase getRequiredBase();
    /**
     * Gets the requirements.
     * 
     * When a CompositeLocalizationUnit is in scope for a particular deploymentas determined by evaluation of its LocalizationBase 
     * and Languages propertiesthen its requirements MUST be met.
     * 
     * @return A collection of Requirements that must be met prior to successful processing of the nested content elements. 
     */
    Collection<Requirement> getRequirements();
    /**
     * Gets the languages.
     * 
     * The Languages element in the CompositeLocalizationUnit MUST NOT be defined or MUST define the union of all languages supported 
     * by the nested content elements. For nested content elements to be evaluated to determine if they are in scope, the 
     * CompositeLocalizationUnit must be in scope. When Languages is present in the CompositeLocalizationUnit, it must define 
     * one of the languages in scope for the particular deployment if any of the nested elements are to be evaluated. If Languages 
     * is not present in a CompositeLocalizationUnit, evaluation of all the child elements still is required, as long as the other 
     * elements of CompositeLocalizationUnit have evaluated to true. When the Languages and/or the LocalizationBase element in a 
     * CompositeLocalizationUnit is not defined, the nested content elements must be evaluated to determine if they are in scope.
     * 
     * @return A collection of Languages. Localization elements defined within CompositeLocalizationUnit contain materials translated into these languages.
     */
    Collection<Language> getLanguages();
    /**
     * Gets the completion elements.
     * 
     * When a particular completion action applies to all nested elements and should be performed only once for the entire group, 
     * it can be defined in the CompositeLocalizationUnit rather than in each individual element.
     * 
     * @return A collection of completion actions such as restart and the conditions under which the action is applied.
     */
    Collection<Completion> getCompletions();
    /**
     * Gets the localization base.
     * 
     * A LocalizationBase element evaluates to true when the resource identified in the base is created by a content element that 
     * is in scope for the deployment or it already exists in the deployment environment.<br><br>
     * When the LocalizationBase is defined it must evaluate to true for any of the nested content elements to be evaluated. If it 
     * evaluates to false, none of the nested content elements are in scope. If it evaluates to true, the nested content elements 
     * may be in scope.<br><br>
     * When the LocalizationBase and/or the Languages element in a CompositeLocalizationUnit is not defined, the nested content 
     * elements must be evaluated to determine if they are in scope.
     * 
     * @return A resource whose translatable characteristics will be localized by processing the nested content elements.
     */
    RequiredBase getLocalizationBase();
    /**
     * Gets the resulting resources.
     * 
     * If there are one or more resources that will be created when the nested content elements are processed, they can be defined here.
     * 
     * @return A collection of localization resources that will be installed or updated by processing the nested content elements.
     */
    Collection<ResultingResource> getResultingResources();
    /**
     * Gets the localization units.
     * 
     * LocalizationUnits defined within the composite typically have common metadata. Metadata defined in the composite does 
     * not need to be repeated in the nested element. Definitions in the nested LocalizationUnit are additions to those defined 
     * in the composite.
     * 
     * @return A collection of LocalizationUnits that contain artifacts that will create, modify or delete language support.
     */
    Collection<LocalizationUnit> getLocalizationUnits();
    /**
     * Gets the contained localization packages.
     * 
     * A ContainedLocalizationPackage is defined in a CompositeLocalizationUnit for the same reasons that a LocalizationUnit 
     * is, because it has metadata in common with other elements defined in the composite.
     * 
     * @return A collection of ReferencedPackages that identify an SDD whose contents are aggregated to create, modify or delete language support.
     */
    Collection<ReferencedPackage> getContainedLocalizationPackages();
    /**
     * Gets the composite localization units.
     * 
     * A CompositeLocalizationUnit can be nested inside another CompositeLocalizationUnit when some of the metadata is shared only 
     * by a subset of the elements nested in the higher level composite.<br><br>
     * <ul>For example, the higher level composite might contain operating system requirements that apply to all localization content 
     * and nested composites might group localization content by localization base.</ul>
     * 
     * @return A collection of CompositeLocalizationUnits that group localization content and defines metadata common to all the grouped content. 
     */
    Collection<CompositeLocalizationUnit> getCompositeLocalizationUnits();
    
    /**
     * Sets the condition.
     * 
     * If the composite and the elements nested beneath it are applicable only in certain environments, a Condition can be defined. 
     * When the Condition is not met, the composite and its nested elements are not in scope.
     * 
     * @param condition A condition that determines if the CompositeLocalizationUnit is relevant to a particular deployment.
     */
    void setCondition(Condition condition);
    /**
     * Sets the required base.
     * 
     * If the processing of all the update artifacts in the nested content elements results in a single resource being updated, 
     * that resource can be defined in the CompositeLocalizationUnits RequiredBase element.
     * 
     * @param requiredBase A resource that will be updated when the nested elements are processed.
     */
    void setRequiredBase(RequiredBase requiredBase);
    /**
     * Sets the requirements.
     * 
     * When a CompositeLocalizationUnit is in scope for a particular deploymentas determined by evaluation of its LocalizationBase 
     * and Languages propertiesthen its requirements MUST be met.
     *  
     * @param requirements Requirements that must be met prior to successful processing of the nested content elements.
     */
    void setRequirements(Collection<Requirement> requirements);
    /**
     * Sets the languages.
     * 
     * The Languages element in the CompositeLocalizationUnit MUST NOT be defined or MUST define the union of all languages supported 
     * by the nested content elements. For nested content elements to be evaluated to determine if they are in scope, the 
     * CompositeLocalizationUnit must be in scope. When Languages is present in the CompositeLocalizationUnit, it must define 
     * one of the languages in scope for the particular deployment if any of the nested elements are to be evaluated. If Languages 
     * is not present in a CompositeLocalizationUnit, evaluation of all the child elements still is required, as long as the other 
     * elements of CompositeLocalizationUnit have evaluated to true. When the Languages and/or the LocalizationBase element in a 
     * CompositeLocalizationUnit is not defined, the nested content elements must be evaluated to determine if they are in scope.
     * 
     * @param languages A collection of Languages. Localization elements defined within CompositeLocalizationUnit contain materials translated into these languages.
     */
    void setLanguages(Collection<Language> languages);
    /**
     * Sets the completion elements.
     * 
     * When a particular completion action applies to all nested elements and should be performed only once for the entire group, 
     * it can be defined in the CompositeLocalizationUnit rather than in each individual element.
     * 
     * @param completions A collection of completion actions such as restart and the conditions under which the action is applied.
     */
    void setCompletions(Collection<Completion> completions);
    /**
     * Sets the localization base.
     * 
     * A LocalizationBase element evaluates to true when the resource identified in the base is created by a content element that 
     * is in scope for the deployment or it already exists in the deployment environment.<br><br>
     * When the LocalizationBase is defined it must evaluate to true for any of the nested content elements to be evaluated. If it 
     * evaluates to false, none of the nested content elements are in scope. If it evaluates to true, the nested content elements 
     * may be in scope.<br><br>
     * When the LocalizationBase and/or the Languages element in a CompositeLocalizationUnit is not defined, the nested content 
     * elements must be evaluated to determine if they are in scope.
     * 
     * @param localizationBase A resource whose translatable characteristics will be localized by processing the nested content elements.
     */
    void setLocalizationBase(RequiredBase localizationBase);
    /**
     * Sets the resulting resources.
     * 
     * If there are one or more resources that will be created when the nested content elements are processed, they can be defined here.
     * 
     * @param resources A collection of localization resources that will be installed or updated by processing the nested content elements.
     */
    void setResultingResources(Collection<ResultingResource> resources);
    /**
     * Sets the localization units.
     * 
     * LocalizationUnits defined within the composite typically have common metadata. Metadata defined in the composite does 
     * not need to be repeated in the nested element. Definitions in the nested LocalizationUnit are additions to those defined 
     * in the composite.
     * 
     * @param localizationUnits A collection of LocalizationUnits that contain artifacts that will create, modify or delete language support.
     * @throws InvalidInputException if any of the parameters is null
     */
    void setLocalizationUnits(Collection<LocalizationUnit> localizationUnits);
    /**
     * Sets the contained localization packages.
     * 
     * A ContainedLocalizationPackage is defined in a CompositeLocalizationUnit for the same reasons that a LocalizationUnit 
     * is, because it has metadata in common with other elements defined in the composite.
     * 
     * @param packages A collection of ReferencedPackages that identify an SDD whose contents are aggregated to create, modify or delete language support.
     * @throws InvalidInputException if any of the parameters is null
     */
    void setContainedLocalizationPackages(Collection<ReferencedPackage> packages);
    /**
     * Sets the composite localization units.
     * 
     * A CompositeLocalizationUnit can be nested inside another CompositeLocalizationUnit when some of the metadata is shared only 
     * by a subset of the elements nested in the higher level composite.<br><br>
     * <ul>For example, the higher level composite might contain operating system requirements that apply to all localization content 
     * and nested composites might group localization content by localization base.</ul>
     *  
     * @param localizationUnits A collection of CompositeLocalizationUnits that group localization content and defines metadata common to all the grouped 
     * content.
     * @throws InvalidInputException if any of the parameters is null
     */
    void setCompositeLocalizationUnits(Collection<CompositeLocalizationUnit> localizationUnits);

	/**
	 * Creates copy of this object. 
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
