/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DeploymentDescriptor;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.Artifact;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.DerivedVariable;

/**
 * An ConditionFactory is used to create objects representing or directly
 * related to content units.
 *
 */
public interface ContentUnitFactory {
	static final int INSTALL_ARTIFACT = 0;
	static final int UNDO_ARTIFACT = 1;
	static final int UNINSTALL_ARTIFACT = 2;
	static final int UPDATE_ARTIFACT = 3;
	
	/**
	 * Creates a new InstallableUnit.
	 * 
	 * A install unit object requires at least one {@link Artifact}, which may be
	 * of any artifact type, at creation time.  Others may be added or removed
	 * post-creation. 
	 * 
	 * @param id an identifier that can be used to reference the Installable
	 * Unit within a {@link DeploymentDescriptor}
	 * @param targetResourceReference Reference to the resource that can
	 * process the InstallableUnit's artifacts
	 * @param initialArtifact An artifact associated with the InstallableUnit
	 * @param artifactType One of the constants defined by this interface for
	 * artifact types
	 * @return a new InstallableUnit object
	 * @throws InvalidInputException if any of the parameters is null
	 */
	InstallableUnit createInstallableUnit(
			String id, String targetResourceReference, Artifact initialArtifact,
			int artifactType);

	
	/**
	 * Creates a new InstallabelUnit.
	 * 
	 * This method is deprecated.
	 * 
	 * @param id an identifier that can be used to reference the Installable
	 * Unit within a {@link DeploymentDescriptor}
	 * @param targetResourceReference Reference to the resource that can
	 * process the InstallableUnit's artifacts
	 * @param installArtifact An artifact associated with the InstallableUnit
	 * @return a new InstallableUnit
	 * @throws InvalidInputException if any of the parameters is null
	 * @deprecated
	 */
	InstallableUnit createInstallableUnit(
			String id, String targetResourceReference, Artifact installArtifact);
	
	/**
	 * Creates a new LocalizationUnit.
	 * 
	 * This method is deprecated.
	 * 
	 * @param id an identifier that can be used to reference the Localization
	 * Unit within a {@link DeploymentDescriptor}
	 * @param targetResourceReference Reference to the resource that can process
	 * the LocalizationUnit's artifacts
	 * @param installArtifact An artifact associated with the InstallableUnit
	 * @return a new LocalizationUnit
	 * @throws InvalidInputException if any of the parameters is null
	 * @deprecated
	 */
	LocalizationUnit createLocalizationUnit(
			String id, String targetResourceReference, Artifact installArtifact);
	
	
	/**
	 * Creates a new LocalizationUnit.
	 *
	 * 
	 * @param id an identifier that can be used to reference the Localization
	 * Unit within a {@link DeploymentDescriptor}
	 * @param targetResourceReference Reference to the resource that can process
	 * the LocalizationUnit's artifacts
	 * @param installArtifact An artifact associated with the InstallableUnit
	 * @param artifactType One of the constants defined by this interface for
	 * artifact types.
	 * @return a new LocalizationUnit
	 * @throws InvalidInputException if any of the parameters is null
	 */
	LocalizationUnit createLocalizationUnit(
			String id, String targetResourceReference, Artifact installArtifact,
			int artifactType);
	
	/**
	 * Creates a new ConfigurationUnit.
	 * 
	 * @param id an identifier that can be used to reference the Configuration
	 * Unit within a {@link DeploymentDescriptor}
	 * @param targetResourceReference Reference to the resource that can process
	 * the ConfigurationUnit's artifacts
	 * @param installArtifact An artifact associated with the InstallableUnit
	 * @param artifactType One of the constants defined by this interface for
	 * artifact types.
	 * @return a new ConfigurationUnit
	 * @throws InvalidInputException if any of the parameters is null
	 */
	ConfigurationUnit createConfigurationUnit(
			String id, String targetResourceReference, Artifact configArtifact);
	
	/**
	 * Creates a new CompositeUnit.
	 * 
	 * @param id an identifier that can be used to reference the Composite
	 * Unit within a {@link DeploymentDescriptor}
	 * @return a new CompositeUnit
	 * @throws InvalidInputException if any of the parameters is null
	 */
	CompositeUnit createCompositeUnit(String id) throws InvalidInputException;
	
	/**
	 * Creates a new CompositeInstallable.
	 * 
	 * @param id an identifier that can be used to reference the CompositeInstallable
	 * within a {@link DeploymentDescriptor}
	 * @param operation The deployment lifecycle operation described by the CompositeInstallable definition
	 * @return a new CompositeInstallable
	 * @throws InvalidInputException if any of the parameters is null
	 */
	CompositeInstallable createCompositeInstallable(String id, String operation) throws InvalidInputException;
	
	/**
	 * Creates a new CompositeLocalizationUnit.
	 * 
	 * @param id an identifier that can be used to reference the CompositeLocalizationUnit
	 * within a {@link DeploymentDescriptor}
	 * @param localizationUnit A collection of LocalizationUnit objects that contain artifacts that will create, modify or delete language support
	 * @return a new CompositeLocalizationUnit
	 * @throws InvalidInputException if any of the parameters is null
	 */
	CompositeLocalizationUnit createCompositeLocalizationUnitWithLocalizationUnits(String id, Collection<LocalizationUnit> localizationUnit) throws InvalidInputException;
	
	/**
	 * Creates a new CompositeLocalizationUnit.
	 * 
	 * @param id an identifier that can be used to reference the CompositeLocalizationUnit
	 * within a {@link DeploymentDescriptor}
	 * @param compositeLocalizationUnit A collection of compositeLocalizationUnit objects that groups localization content and defines metadata common to all the grouped content
	 * @return a new CompositeLocalizationUnit
	 * @throws InvalidInputException if any of the parameters is null
	 */
	CompositeLocalizationUnit createCompositeLocalizationUnitWithCompositeLocalizationUnits(String id, Collection<CompositeLocalizationUnit> compositeLocalizationUnit) throws InvalidInputException;
	
	/**
	 * Creates a new CompositeLocalizationUnit.
	 * 
	 * @param id an identifier that can be used to reference the CompositeLocalizationUnit
	 * within a {@link DeploymentDescriptor}
	 * @param containedLocalizationPackage A collection of ReferencedPackage objects that identify an SDD whose contents are aggregated to create, modify or delete language support
	 * @return a new CompositeLocalizationUnit
	 * @throws InvalidInputException if any of the parameters is null
	 */
	CompositeLocalizationUnit createCompositeLocalizationUnitWithContainedLocalizationPackages(String id, Collection<ReferencedPackage> containedLocalizationPackage) throws InvalidInputException;
	
	/**
	 * Creates a new Group.
	 * 
	 * @param id an identifier that can be used to reference the Group
	 * within a {@link DeploymentDescriptor}
	 * @param selectedFeatures A collection of features that are part of the group
	 * @return a new Group
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Group createGroup(String id, Collection<String> selectedFeatures) throws InvalidInputException;
	
	/**
	 * Creates a new LanguageSet.
	 * 
	 * @param languages A collection of languages that are part of the LanguageSet
	 * @return a new LanguageSet
	 * @throws InvalidInputException if any of the parameters is null
	 */
	LanguageSet createLanguageSet(Collection<Language> languages) throws InvalidInputException;
	
	/**
	 * Creates a new Language.
	 * 
	 * @param type The locale code for the language
	 * @return a new Language
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Language createLanguage(String type) throws InvalidInputException;
	
	/**
	 * Creates a new Relationship.
	 * 
	 * @param relatedResourceRef The second resource in the relationship
	 * @param type The type of the relationship
	 * @return a new Relationship
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Relationship createRelationship(String relatedResourceRef, String type) throws InvalidInputException;
	
	/**
	 * Creates a new PackageFeatureReference.
	 * 
	 * @param contentElementRef Reference to a content element
	 * within a {@link DeploymentDescriptor}
	 * @param packageFeatureRef The features id as defined in the referenced packages deployment descriptor.
	 * @return a new PackageFeatureReference
	 * @throws InvalidInputException if any of the parameters is null
	 */
	PackageFeatureReference createPackageFeatureReference(String contentElementRef, String packageFeatureRef) throws InvalidInputException;
	
	/**
	 * Creates a new Feature.
	 * 
	 * @param id an identifier that can be used to reference the Feature
	 * within a {@link DeploymentDescriptor}
	 * @param features A collection of NestedFeature objects
	 * @return a new Feature
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Feature createFeatureWithNestedFeatures(String id, Collection<NestedFeature> features) throws InvalidInputException;
	
	/**
	 * Creates a new Feature.
	 * 
	 * @param id an identifier that can be used to reference the Feature
	 * within a {@link DeploymentDescriptor}
	 * @param contentReferences A collection of references to content elements to be deployed when the feature is selected.
	 * @return a new Feature
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Feature createFeatureWithContentElements(String id, Collection<String> contentReferences) throws InvalidInputException;
	
	/**
	 * Creates a new Feature.
	 * 
	 * @param id an identifier that can be used to reference the Feature
	 * within a {@link DeploymentDescriptor}
	 * @param requiredFeatureReference A collection of references to a feature that is required when the defining feature 
	 * is selected and so is selected automatically
	 * @return a new Feature
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Feature createFeatureWithRequiredFeatures(String id, Collection<String> requiredFeatureReference) throws InvalidInputException;
	
	/**
	 * Creates a new Feature.
	 * 
	 * @param id an identifier that can be used to reference the Feature
	 * within a {@link DeploymentDescriptor}
	 * @param derivedVariables A collection of variables that can be used anywhere in any variable 
	 * expression in the SDD
	 * @return a new Feature
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Feature createFeatureWithDerivedVariables(String id, Collection<DerivedVariable> derivedVariables) throws InvalidInputException;

	/**
	 * Creates a new Feature.
	 * 
	 * @param id an identifier that can be used to reference the Feature
	 * within a {@link DeploymentDescriptor}
	 * @param packageFeatureReferences A collection of references to a feature to be selected in a ContainedPackage 
	 * defined in either the BaseContent or SelectableContent hierarchies.
	 * @return a new Feature
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Feature createFeatureWithPackageFeatureReferences(String id, Collection<PackageFeatureReference> packageFeatureReferences) throws InvalidInputException;
	
	/**
	 * Creates a new NestedFeature.
	 * 
	 * @param id an identifier that can be used to reference the NestedFeature
	 * within a {@link DeploymentDescriptor}
	 * @param features A collection of NestedFeature objects
	 * @return a new NestedFeature
	 * @throws InvalidInputException if any of the parameters is null
	 */
    NestedFeature createNestedFeatureWithNestedFeatures(String id, Collection<NestedFeature> features) throws InvalidInputException;
    
    /**
	 * Creates a new NestedFeature.
	 * 
	 * @param id an identifier that can be used to reference the NestedFeature
	 * within a {@link DeploymentDescriptor}
	 * @param contentReferences A collection of references to content elements to be deployed when the feature is selected.
	 * @return a new NestedFeature
	 * @throws InvalidInputException if any of the parameters is null
	 */
    NestedFeature createNestedFeatureWithContentElements(String id, Collection<String> contentReferences) throws InvalidInputException;
    
    /**
	 * Creates a new NestedFeature.
	 * 
	 * @param id an identifier that can be used to reference the NestedFeature
	 * within a {@link DeploymentDescriptor}
	 * @param requiredFeatureReference A collection of references to a feature that is required when the defining feature 
	 * is selected and so is selected automatically
	 * @return a new NestedFeature
	 * @throws InvalidInputException if any of the parameters is null
	 */
    NestedFeature createNestedFeatureWithRequiredFeatures(String id, Collection<String> requiredFeatureReference) throws InvalidInputException;
    
    /**
	 * Creates a new NestedFeature.
	 * 
	 * @param id an identifier that can be used to reference the NestedFeature
	 * within a {@link DeploymentDescriptor}
	 * @param derivedVariables A collection of variables that can be used anywhere in any variable 
	 * expression in the SDD
	 * @return a new NestedFeature
	 * @throws InvalidInputException if any of the parameters is null
	 */
    NestedFeature createNestedFeatureWithDerivedVariables(String id, Collection<DerivedVariable> derivedVariables) throws InvalidInputException;
    
    /**
	 * Creates a new NestedFeature.
	 * 
	 * @param id an identifier that can be used to reference the NestedFeature
	 * within a {@link DeploymentDescriptor}
	 * @param packageFeatureReferences A collection of references to a feature to be selected in a ContainedPackage 
	 * defined in either the BaseContent or SelectableContent hierarchies.
	 * @return a new NestedFeature
	 * @throws InvalidInputException if any of the parameters is null
	 */
    NestedFeature createNestedFeatureWithPackageFeatureReferences(String id, Collection<PackageFeatureReference> packageFeatureReferences) throws InvalidInputException;

    /**
	 * Creates a new MultiFeatureSelect.
	 * 
	 * @param features A collection of references to features in the list of features defined in the MultiSelect element
	 * @return a new MultiFeatureSelect
	 * @throws InvalidInputException if any of the parameters is null
	 */
	MultiFeatureSelect createMultiFeatureSelect(Collection<String> features) throws InvalidInputException;
	
	/**
	 * Creates a new Multiplicity.
	 * 
	 * @param multiplesAllowed Indicates that if multiple instances of the feature are allowed
	 * @return a new Multiplicity
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Multiplicity createMultiplicity(Boolean multiplesAllowed) throws InvalidInputException;
	
	/**
	 * Creates a new ConstrainedResource.
	 * 
	 * @param resourceRef A reference to the constrained resource
	 * @return a new ConstrainedResource
	 * @throws InvalidInputException if any of the parameters is null
	 */
	ConstrainedResource createConstrainedResource(String resourceRef) throws InvalidInputException;
	
	/**
	 * Creates a new ContentSelectionFeature.
	 * 
	 * @param feature A definition of user-selectable content
	 * @return a new ContentSelectionFeature
	 * @throws InvalidInputException if any of the parameters is null
	 */
	ContentSelectionFeature createContentSelectionFeature(String feature) throws InvalidInputException;

	/**
	 * Creates a new ResultingChange object.
	 * 
	 * @return a ResultingChange object, which can be added to an
	 * {@link InstallableUnit} or a {@link ConfigurationUnit}
	 */
	ResultingChange createResultingChange();
	
	/**
	 * Creates a new ResultingResource object.
	 * 
	 * @param resultingResourceReference A reference to a resource in topology
	 * @return a ResultingResource object, which can be added to an
	 * {@link InstallableUnit} or a {@link LocalizationUnit}
	 * @throws InvalidInputException if any of the parameters is null
	 */
	ResultingResource createResultingResource(String resultingResourceReference);
	
	/**
	 * Creates a completion object.
	 * 
	 * Completion type is typically one of the following:
	 * <ul>
	 * <li>restartRequired</li>
	 * <li>restartOccurs</li>
	 * <li>logoffRequired</li>
	 * </ul>
	 * 
	 * @param completionType The type of the completion action
	 * @param resourceReference The resource where the completion action will be executed 
	 * @return A Completion object, which can be added to an
	 * {@link InstallableUnit} or a {@link ConfigurationUnit}
	 * @throws InvalidInputException if any of the parameters is null
	 */
	Completion createCompletion(String completionType, String resourceReference);
}
