/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.databuilders;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.net.URI;
import org.apache.xerces.xs.ElementPSVI;
import org.apache.xerces.xs.PSVIProvider;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLIFIdentity;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.core.IFoundationBuilder;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.w3c.dom.Node;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Abstract class that provides a default implementation of the IDataBuilder interface.
 *
 * @author Sheldon Lee-Loy
 * @author Ali Mehregani
 * @author John Arwe
 */
public abstract class AbstractDataBuilder<T> extends DefaultHandler implements IDataBuilder<T>
{
	/**
	 * The initialization data
	 */
	private Map<String, Object> init;

	/**
	 * Stores line and column information
	 */
	private Locator locator;

	/**
	 * The target namespace from the current XML Schema (xs:schema) element
	 */
	private String targetNamespace;

	/**
	 * Stores the default namespace from the current XML Schema (xs:schema) element
	 */
	private String defaultNamespace;

	/**
	 * The prefix-namespace bindings from the current XML Schema (xs:schema) element
	 * KEY = A {@link String} indicating a prefix
	 * VALUE = A {@link String} indicating a URL
	 */
	private Map<String, String> prefixMap;

	/**
	 * The validity of the structure generated.  Set to true by
	 * default.
	 */
	private boolean validity;

	/**
	 * The error message
	 */
	private IValidationMessage errorMessage;

	/**
	 * The file path currently being processed
	 */
	private String filePath;

	/**
	 * The supported events of this data builder
	 */
	private int supportedEvents;

	/**
	 * The outputter associated with this data builder, or null
	 */
	private IValidationOutput<?,?> messageOutputter;

	/**
	 * PSVI provider of this data builder
	 */
	private PSVIProvider psvi;

	/**
	 * The base line number
	 */
	private int baseLineNumber;

	/**
	 * The aliases of the current document being parsed
	 */
	private String[] aliases;

	/**
	 * Indicates if the current element belongs to the SML namespace
	 */
	private boolean smlElement;

	/**
	 * Indicates if the current element belongs to the SML-IF namespace
	 */
	private boolean smlifElement;

	/**
	 * Indicates if smlif:identity processing has completely finished,
	 * so the effective model base URI can be set regardless of whether it
	 * was set via xml:base or smlif:identity/smlif:baseURI, and then
	 * whether or not the calculation has been performed already. 
	 */
	private boolean modelBaseURICanBeCalculated;
	private boolean modelBaseURIHasBeenCalculated;

	/**
	 * Simulates the [base URI] infoset property value for the current element,
	 * calculated according to XML Base {@link http://www.w3.org/TR/xmlbase/} 
	 * augmented by SML-IF (XML Base
	 * takes precedence when both are present).
	 * <p> Each stack entry MAY be absolute or a relative reference.
	 * Relative references are not problematic as base URIs unless
	 * (in the context of SML-IF) they are used without some absolute base URI having
	 * been established.  E.g. relative references as xml:base values are legal
	 * if the SML-IF contains only absolute URIs in SML references, aliases, etc.
	 *
	 * <p> The oldest stack entry will logically contain:
	 * <ul>
	 * <li> for an SML-IF document,  null as the base URI since SML-IF does not allow
	 *   appeal to base URI values outside of the SML-IF document.
	 * <li> for a contained SML-IF document (data or base64data), the SML-IF document base URI 
	 * <p> The document base URI in turn may be computed from
	 * <ul>
	 * <li>An xml:base specification from the ancestor-of-self axis (XPath 1.0)
	 * <li>smlif:document/ * /smlif:baseURI
	 * <li>smlif:identity/ smlif:baseURI
	 *  </ul>
	 * <li> for a referred-to SML-IF document (locator), the base URI of the external
	 *   document (which is never null)
	 *  </ul>
	 *
	 * <p> The current stack entry will logically contain:
	 * <ul>
	 * <li>An xml:base specification from the ancestor-of-self axis (XPath 1.0), or null
	 * <li>The SML-IF document base URI, or null
	 * <li>The SML-IF model base URI, or null
	 *  </ul>
	 */
	private Stack<URI> baseURIInfoset = new Stack<URI>(); 

	/**
	 * Constructor
	 */
	public AbstractDataBuilder()
	{
		setStructureValidity(true);
		targetNamespace = "";
		prefixMap = new Hashtable<String, String>();
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#reset()
	 */
	public void reset()
	{
		baseURIInfoset.clear();
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init)
	{
		this.init = init;
		this.psvi = (PSVIProvider)init.get(IValidator.ATTRIBUTE_PSVI_INSTANCE);
		this.messageOutputter = (IValidationOutput<?,?>) init.get(IValidator.ATTRIBUTE_OUTPUT);
		Object tmp = init.get(IValidator.ATTRIBUTE_BASE_LINE_NUMBER);
		this.baseLineNumber = tmp == null ? 0 : (Integer) tmp;
		tmp = init.get(IValidator.ATTRIBUTE_DOCUMENT_ALIASES);
		this.aliases = tmp == null ? new String[0] : (String[])tmp;
		baseURIInfoset.push((URI)init.get(IValidator.ATTRIBUTE_DOCUMENT_BASEURI));
		this.modelBaseURICanBeCalculated = false;
		this.modelBaseURIHasBeenCalculated = false;
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#endDefinitions()
	 */
	public void endDefinitions()
	{
		// no-op
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#endInstances()
	 */
	public void endInstances()
	{
		// no-op
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#startDefinitions()
	 */
	public void startDefinitions()
	{
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#startInstances()
	 */
	public void startInstances()
	{
	}


	/**
	 * This is used to determine the target name space of the document currently
	 * being parsed
	 *
	 * @see org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
	 */
	public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException
	{

		// Overriding methods must take care to save any state they need which is changed by this method, e.g. the base URI value.

		smlifElement = ISMLConstants.SMLIF_URI.equals(uri);
		smlElement = ISMLConstants.SML_URI.equals(uri);

		if (ISMLConstants.SCHEMA_URI.equals(uri) && ISMLConstants.SCHEMA_ELEMENT.equals(localName))
		{
			targetNamespace = attributes.getValue(ISMLConstants.TARGET_NAMESPACE_ATTRIBUTE);

			for (int i = 0, attributeCount = attributes.getLength(); i < attributeCount; i++)
			{
				String[] attributeName = SMLValidatorUtil.tokenizeName(attributes.getQName(i));
				if (ISMLConstants.XML_NS_ATTRIBUTE.equals(attributeName[0]))
				{
					prefixMap.put(attributeName[1], attributes.getValue(i));
				}
				else if (ISMLConstants.XML_NS_ATTRIBUTE.equals(attributeName[1]))
				{
					defaultNamespace = attributes.getValue(i);
				}
			}
		}

		if (modelBaseURICanBeCalculated && !modelBaseURIHasBeenCalculated )
		{
			//	At this point, any smlif:baseURI inside the model identity has been processed.
			//	If no superseding xml:base value was present on the model element, then the
			//	smlif:model/smlif:identity/smlif:baseURI value already stored should be used
			//	as the base URI for later relative references (e.g. document aliases).
			// This value itself might later be superseded by an xml:base specification on a descendant,
			//	or a document-level smlif:baseURI.
			
			modelBaseURIHasBeenCalculated = true;	 
			//	Only do this once.  If no model base URI is present, that is legal.
			//	If a relative reference is used later, and no base URI has been specified by that time,
			//	-then- it becomes an invalid SML-IF document.
			
			if(!baseURIInfoset.empty() && baseURIInfoset.peek() == null) 
			{
				SMLIFIdentity identity = (SMLIFIdentity)SMLValidatorUtil.retrieveDataStructure(DataBuilderRegistry.TOP_LEVEL_MODE, IdentityDataBuilder.ID);
				if (identity != null )		//	should always be true if flags were set correctly
				{
					String baseURI = identity.getBaseURI();
					if (baseURI != null) 
					{
						int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();	// null if running SMLModelUnits instead of SML-IF doc input
						IValidationOutput<?, ?> logger = getMessageOutputter(); 
						URI candidateModelBaseURI = URIReference.validModelBaseURIAsURI(baseURI, null, logger, lineNum );
						if (candidateModelBaseURI != null)
						{
							baseURIInfoset.pop(); 						// replace the current (null) value with the model level smlif:baseURI value 
							baseURIInfoset.push(  candidateModelBaseURI   );
						}
					}
				}
			}
		}

		// A conscious choice was made to NOT issue any diagnostic messages for an incorrect xml:base,
		//	not even on the smlif:model element.  As long as it passes schema validity assessment (which
		//	issues its own errors if the content is not a valid URI etc), it is acceptable.  Even if it contains a
		//	relative reference, until its value is appealed to as a base URI no error messages will be issued.
		//	Thus a model with a relative reference as a model base URI, but which is never used because
		//	the model contains all absolute or same-document URIs, would result in no error messages.
		//	This seems to capture the spirit of the XML Base specification.
		
		//	The next element after </smlif:identity>, e.g. smlif:ruleBindings, might specify its own xml:base,
		//	and if it does then the new value should logically override the value just set above.
		String xmlBaseAttrValue = attributes.getValue(ISMLConstants.XML_BASE_ATTRIBUTE);
		URI xmlBaseURI = null;
		if (xmlBaseAttrValue == null) { // Not specified here, logically inherit parent element's value
			xmlBaseURI =  baseURIInfoset.empty() ? null : baseURIInfoset.peek();
		}
		else { // A new specification exists, use it if the value appears to be valid
			xmlBaseURI = validateBaseURIADB( xmlBaseAttrValue );
			if(!baseURIInfoset.empty() && baseURIInfoset.peek() != null) {
				xmlBaseURI = xmlBaseURI.resolve(baseURIInfoset.peek()); // incorporate the existing base URI (which may also be relative)
			}
		}
		baseURIInfoset.push(  xmlBaseURI   );

	}


	/**
	 * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
	 */
	public void endElement(String uri, String localName, String qName) throws SAXException
	{

		// Overriding methods must take care to save any state they need which is changed by this method.
		// If base URIs will be queried, the value on entry should be saved and used.
		// TODO Save base URI value here on behalf of overriding methods

		if (ISMLConstants.SCHEMA_URI.equals(uri) && ISMLConstants.SCHEMA_ELEMENT.equals(localName))
		{
			targetNamespace = "";
			prefixMap.clear();
		}

		baseURIInfoset.pop(); 	// If an overriding implementation needs access to xml:base value in its own endElement
       									//	code, it should save the value prior to calling.
		
		if (ISMLConstants.SMLIF_URI.equals(uri))
		{
			if (ISMLConstants.IDENTITY_ELEMENT.equals(localName))
			{
				modelBaseURICanBeCalculated = true;
			}
		}

	}


	/**
	 * @see org.xml.sax.helpers.DefaultHandler#setDocumentLocator(org.xml.sax.Locator)
	 */
	public void setDocumentLocator(Locator locator)
	{
		this.locator = locator;
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#isStructureValid()
	 */
	public boolean isStructureValid()
	{
		return validity;
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#setStructureValidity(boolean)
	 */
	public void setStructureValidity(boolean validity)
	{
		this.validity = validity;
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getErrorMessage()
	 */
	public IValidationMessage getErrorMessage()
	{
		return errorMessage;
	}

	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#setErrorMessage(java.lang.String)
	 */
	public void setErrorMessage(IValidationMessage error)
	{
		this.errorMessage = error;
	}

	/**
	 * Appends to the error message
	 *
	 * @param error The error, which is appended as a separate line
	 */
	protected void appendToErrorMessage(String error)
	{
		String errMsg = errorMessage.getAttribute(IValidationMessage.ATTRIBUTE_MESSAGE, null);
		if (errMsg == null) {
			errMsg = "";
		}

		if (errMsg.length() > 0) {
			errMsg += IValidationConstants.LINE_SEPARATOR;
		}

		errMsg += error;
	}

	/**
	 * @return the prefix
	 */
	public Map<String, String> getPrefixMap()
	{
		return prefixMap;
	}

	/**
	 * @param prefix the prefix to set
	 */
	public void setPrefixMap(Map<String, String> prefix)
	{
		this.prefixMap = prefix;
	}

	/**
	 * @return the targetNamespace
	 */
	public String getTargetNamespace()
	{
		return targetNamespace;
	}

	/**
	 * @param targetNamespace the targetNamespace to set
	 */
	public void setTargetNamespace(String targetNamespace)
	{
		this.targetNamespace = targetNamespace;
	}

	/**
	 * @return the defaultNamespace
	 */
	public String getDefaultNamespace()
	{
		return defaultNamespace;
	}

	/**
	 * @param defaultNamespace the defaultNamespace to set
	 */
	public void setDefaultNamespace(String defaultNamespace)
	{
		this.defaultNamespace = defaultNamespace;
	}



	/**
	 * Sets the path of the current file being processed
	 *
	 * @param filePath The path
	 */
	public void setFilePath(String filePath)
	{
		this.filePath = filePath;
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getFilePath()
	 */
	public String getFilePath()
	{
		return filePath;
	}


	/**
	 * Subclasses are expected to overwrite this method to process
	 * comments if the associated event is supported.
	 *
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#comment(char[], int, int)
	 */
	public void comment(char[] cs, int i, int j)
	{
	}


	/**
	 * Subclasses are expected to add events for processing different
	 * sections of a document.  See a list of event constants under
	 * {@link IFoundationBuilder} for a list of events that are supported.
	 *
	 * @param event The event to be added.
	 */
	protected void addEvent(int event)
	{
		addEvents(new int[]{event});
	}

	/**
	 * Same semantics as {@link #addEvent(int)} but with the
	 * ability to add multiple events.
	 *
	 * @param events The events to be added
	 */
	protected void addEvents(int[] events)
	{
		for (int i = 0; i < events.length; i++)
		{
			supportedEvents = supportedEvents | (1 << events[i]);
		}
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getSupportedEvents()
	 */
	public int getSupportedEvents()
	{
		return supportedEvents;
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#getMessageOutputter()
	 */
	public IValidationOutput<?, ?> getMessageOutputter()
	{
		return messageOutputter;
	}


	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.databuilders.IDataBuilder#setMessageOutputter(org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput)
	 */
	public void setMessageOutputter(IValidationOutput<?, ?> messageOutputter)
	{
		this.messageOutputter = messageOutputter;
	}


	/**
	 * @return the locator
	 */
	public Locator getLocator()
	{
		return locator;
	}


	/**
	 * @param locator the locator to set
	 */
	public void setLocator(Locator locator)
	{
		this.locator = locator;
	}


	/**
	 * @return the init
	 */
	public Map<String, Object> getInit()
	{
		return init;
	}


	/**
	 * @return the psvi
	 */
	public PSVIProvider getPsvi()
	{
		return psvi;
	}


	/**
	 * @param psvi the psvi to set
	 */
	public void setPsvi(PSVIProvider psvi)
	{
		this.psvi = psvi;
	}


	/**
	 * @return the baseLineNumber
	 */
	public int getBaseLineNumber()
	{
		return baseLineNumber;
	}


	/**
	 * @param baseLineNumber the baseLineNumber to set
	 */
	public void setBaseLineNumber(int baseLineNumber)
	{
		this.baseLineNumber = baseLineNumber;
	}


	/**
	 * @return the aliases
	 */
	public String[] getAliases()
	{
		return aliases;
	}


	/**
	 * @param aliases the aliases to set
	 */
	public void setAliases(String[] aliases)
	{
		this.aliases = aliases;
	}


	public AnnotationResult retrieveElementAnnotation(String uri, String localName)
	{
		ElementPSVI elementPSVI = getPsvi().getElementPSVI();
		XSElementDeclaration elementDeclaration = elementPSVI == null ? null : elementPSVI.getElementDeclaration();
		return SMLValidatorUtil.retrieveAnnotation(elementDeclaration, uri, localName);
	}


	/**
	 * @return the smlElement
	 */
	public boolean isSMLElement()
	{
		return smlElement;
	}


	/**
	 * @return the smlifElement
	 */
	public boolean isSMLIFElement()
	{
		return smlifElement;
	}


	/**
	 * Returns the line number of the current context
	 *
	 * @return The line number of current context
	 */
	public int getContextLineNumber()
	{
		return getBaseLineNumber() + getLocator().getLineNumber() - 1;
	}


	/**
	 * Returns the data builder's simulation of the [base URI] infoset property,
	 * after transformation using any ancestor base URI specifications within
	 * the SML-IF document.  
	 * <p>It is possible that a non-null value will be a
	 * relative reference, if no absolute URI was specified as an ancestor
	 * base URI.
	 *
	 * @return The simulated [base URI] infoset property value (relative reference or absolute URI), or null if none has been encountered
	 */
	public URI getCurrentBaseURI()
	{
		return baseURIInfoset.isEmpty() == true ? null : baseURIInfoset.peek();
	}


	public static class AnnotationResult
	{
		private List<Node> nodes;
		private XSTypeDefinition type;

		public AnnotationResult()
		{
			nodes = new ArrayList<Node>();
		}

		/**
		 * @return the node
		 */
		public Node[] getNodes()
		{
			return nodes.toArray(new Node[nodes.size()]);
		}
		/**
		 * @param node the node to set
		 */
		public void addNode(Node node)
		{
			this.nodes.add(node);
		}
		/**
		 * @return the type
		 */
		public XSTypeDefinition getType()
		{
			return type;
		}
		/**
		 * @param type the type to set
		 */
		public void setType(XSTypeDefinition type)
		{
			this.type = type;
		}
	}

	private URI validateBaseURIADB( String URIAsString ) 
	{
		int lineNum = getLocator() == null ? -1 : getLocator().getLineNumber();	// null if running SMLModelUnits instead of SML-IF doc input
		IValidationOutput<?, ?> logger = getMessageOutputter(); 
		URI candidateModelBaseURI = URIReference.validModelBaseURIAsURI(URIAsString, null, logger, lineNum );
		return candidateModelBaseURI;	//	null, if not valid
	}
}
