/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.rm.internal.smlif.export.ui;

import java.util.HashMap;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.cosmos.rm.internal.repository.resource.SMLFileMetadata;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLMessages;
import org.eclipse.cosmos.rm.internal.smlif.common.SMLRepositoryManager;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.IWorkbench;

/**
 * Wizard for exporting SML documents into an SML-Interchange Format file.  If a .smlif_meta
 * file is detected in the base project of this export, the metadata therein will also be used
 * in building the SML-IF file.
 * 
 * @author David Whiteman
 * @modified Ali Mehregani
 */
public class ExportToSMLIFWizard extends Wizard implements IExportWizard, ICheckStateListener {
	
	protected ExportToSMLIFWizardPage mainPage;
	protected ExportToSMLIFWizardPage2 page2;
	protected ExportToSMLIFWizardPage3 page3;
	protected IStructuredSelection selection;
	protected HashMap<IResource, ISMLDocument> selectedFiles = new HashMap<IResource, ISMLDocument>();
	
	public ExportToSMLIFWizard() {
		super();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#performFinish()
	 */
	public boolean performFinish() {
		return mainPage.doExport();
	}
	 
	/* (non-Javadoc)
     * @see org.eclipse.jface.wizard.IWizard#addPages()
     */
    public void addPages() {
        super.addPages(); 
        
		mainPage = new ExportToSMLIFWizardPage(SMLMessages.exportWizardPageTitle, selection);
		mainPage.setDescription(SMLMessages.exportWizardFileInstruction);
        
        addPage(mainPage);        

		page2 = new ExportToSMLIFWizardPage2(SMLMessages.exportWizardPageTitle, selection);
		page2.setDescription(SMLMessages.exportWizardDescriptionPageTwo);
        
        addPage(page2);        

        page3 = new ExportToSMLIFWizardPage3(SMLMessages.exportWizardPageTitle, selection);
		page3.setDescription(SMLMessages.exportWizardDescriptionPage3);
        
        addPage(page3);        

        setWindowTitle(SMLMessages.exportWizardTitle);
    }

	public void init(IWorkbench workbench, IStructuredSelection selection) {
		this.selection = selection;
	}

	public ExportToSMLIFWizardPage getMainPage() {
		return mainPage;
	}

	/**
	 * One file or folder has been checked or unchecked.
	 * Update the model and the rest of the wizard pages.
	 */
	public void checkStateChanged(CheckStateChangedEvent event) {
		//if (!(event.getElement() instanceof IFile)) return;
		IResource file = (IResource) event.getElement();
		if (event.getChecked()) {
			addFileToSelected(file);
		} else {
			removeFileFromSelected(file);
		}
		updateForCheckChange();
	}
	
	/**
	 * Update the model when selectAll or deselectAll
	 * button is pressed on the main wizard page
	 * 
	 * @param checked
	 */
	public void allFilesChecked(boolean checked) {
		if (checked) {
			Runnable runnable = new Runnable() {
				public void run() {
					addFileToSelected(mainPage.getSelectedProject());
				}
			};
			org.eclipse.swt.custom.BusyIndicator.showWhile(null, runnable);
		} else {
			selectedFiles.clear();
		}
		updateForCheckChange();
	}

	/**
	 * The check state has changed for one or more files
	 * in the model - update all interested widgets
	 */
	private void updateForCheckChange() {
		page2.updateDocumentList();
		page2.initDocumentIdentityFields();
		page3.updateComboChoices();
		mainPage.updatePageComplete();
	}

	protected void addFileToSelected(IResource file) {
		if (file instanceof IContainer) {
			Object[] kids = ((ExportResourceTreeContentProvider) mainPage
					.getFileTree().getContentProvider()).getChildren(file);
			for (int i = 0; i < kids.length; i++) {
				addFileToSelected((IResource) kids[i]);
			}
			return;
		}
		primAddFileToSelected(file);
	}

	protected void primAddFileToSelected(IResource file) 
	{		
		if (selectedFiles.get(file) == null) 
		{			
			try
			{
				ISMLDocument[] document = SMLRepositoryManager.getRepository(file.getProject().getLocation().toString(), false).
					fetchDocuments(new SMLFileMetadata(file.getLocation().toString(), ISMLMetadata.DOCUMENT_TYPE_NONE, null, null, null));
				if (document != null && document.length == 1)
					selectedFiles.put(file, document[0]);
			} 
			catch (RepositoryOperationException e)
			{
				e.printStackTrace();
			}			
		}		
	}

	protected void removeFileFromSelected(IResource file) {
		if (file instanceof IContainer) {
			Object[] kids = ((ExportResourceTreeContentProvider) mainPage
					.getFileTree().getContentProvider()).getChildren(file);
			for (int i = 0; i < kids.length; i++) {
				removeFileFromSelected((IResource) kids[i]);
			}
			return;
		}
		primRemoveFileFromSelected(file);
	}

	protected void primRemoveFileFromSelected(IResource file) {
		if (selectedFiles.containsKey(file)) {
			selectedFiles.remove(file);
		}
	}

	public HashMap<IResource, ISMLDocument> getSelectedFiles() {
		return selectedFiles;
	}


	protected static Button createVerticalButtonPanelButton(Composite buttonComposite,
			String buttonText) {
		Button button = new Button(buttonComposite, SWT.NONE);
		button.setText(buttonText);
		GridData buttonData = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		buttonData.verticalAlignment = SWT.BEGINNING;
		button.setLayoutData(buttonData);
		return button;
	}

	protected static Composite createVerticalButtonPanel(Composite parent) {
		Composite buttonComposite = new Composite(parent, SWT.NONE);
		GridLayout buttonLayout = new GridLayout();
		buttonLayout.numColumns = 1;
		buttonLayout.marginHeight = 0;
		buttonLayout.marginWidth = 0;
		buttonComposite.setLayout(buttonLayout);
		GridData buttonCompositeData = new GridData();
		buttonCompositeData.verticalAlignment = GridData.FILL;
		buttonComposite.setLayoutData(buttonCompositeData);
		return buttonComposite;
	}

	public ExportToSMLIFWizardPage2 getPage2()
	{
		return page2;
	}
}
