/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requiredbase.RequiredBase;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;

/**
 * A CompositeInstallable supports the definition of metadata about package content for one deployment 
 * lifecycle operation. One CompositeInstallable can be defined for each operation supported by the 
 * software package. When more than one CompositeInstallable is defined in an SDD, there MUST NOT be 
 * more than one CompositeInstallable in scope for a particular deployment defined for any one operation.
 */
public interface CompositeInstallable extends ContentUnit {
	/**
	 * Gets the operation.
	 * 
	 * This is the operation that may be applied to the SDD package whose metadata is described by the CompositeInstallable. 
	 * 
	 * @return The deployment lifecycle operation described by the CompositeInstallable definition.
	 */
	String getOperation();
	
    /**
     * Gets the condition.
     * 
     * When the condition defined in the CompositeInstallable is not met for a particular deployment, the CompositeUnit and all 
     * the content elements defined below the CompositeUnit are out of scope for that particular deployment.
     * 
     * @return A condition that determines if the content of the CompositeInstallable is relevant to a particular deployment. 
     */
    Condition getCondition();
    /**
     * Gets the required base.
     * 
     * When a resource or resources corresponding to the overall software will be modified during deployment, that resource or 
     * those resources MAY be defined in the RequiredBase element. The RequiredBase definition represents a requirement that the described resource be available for modification to apply the single operation defined by the CompositeInstallable. When RequiredBase is defined, the operation defined by CompositeInstallable MUST be one of the following: update, undo, uninstall, or repair. By specifying the required base separately from other requirements, it is possible for consumers of the SDD to easily determine if the base is available before processing other requirements.
     * 
     * @return Resource that can be updated by the CompositeInstallable.
     */
    RequiredBase getRequiredBase();
    /**
     * Gets the requirements.
     * 
     * These are requirements that must be met regardless of what content is selected for deployment and which conditions within 
     * the content hierarchy evaluates to true.<br><br>
     * Requirements that apply only to a portion of the content SHOULD be defined at the point in the content hierarchy where they 
     * apply.<br><br>
     * All requirements specified on content elements that are in scope for a particular deployment MUST be met. This represents a 
     * logical AND of the requirements. Care should be taken by the SDD author to ensure that conflicting requirements cannot be in scope for the same deployment.  
     * 
     * @return A collection of Requirement objects that must be met before successful application of the CompositeInstallable. 
     */
    Collection<Requirement> getRequirements();
    /**
     * Gets the mandatory languages.
     * 
     * When the SDD contains language support, the Languages element can be defined to describe the languages supported; which 
     * languages are required and which are selectable; and how language selections are grouped.<br><br>
     * Languages defined in the Mandatory element under Languages are always in scope. Languages defined in the Optional element 
     * under Languages are in scope if selected by the deployer.<br><br>
     * The Languages element is used to declare the mandatory and optional language support available in the package. Languages 
     * whose support is deployed by LocalizationUnits in LocalizationContent MUST be defined as either a mandatory language or an 
     * optional language. In addition, languages whose support is deployed along with other content by InstallableUnits in BaseContent or SelectableContent SHOULD be defined as a mandatory language.
     * 
     * @return A collection of required languages. 
     */
    Collection<Language> getMandatoryLanguages();
    /**
     * Gets optional languages.
     * 
     * When the SDD contains language support, the Languages element can be defined to describe the languages supported; which 
     * languages are required and which are selectable; and how language selections are grouped.<br><br>
     * Languages defined in the Mandatory element under Languages are always in scope. Languages defined in the Optional element 
     * under Languages are in scope if selected by the deployer.<br><br>
     * The Languages element is used to declare the mandatory and optional language support available in the package. Languages 
     * whose support is deployed by LocalizationUnits in LocalizationContent MUST be defined as either a mandatory language or an optional language. In addition, languages whose support is deployed along with other content by InstallableUnits in BaseContent or SelectableContent SHOULD be defined as a mandatory language.
     * 
     * @return A collection of selectable languages.
     */
    Collection<Language> getOptionalLanguages();
    /**
     * Gets the optional language sets.
     * 
     * When the SDD contains language support, the Languages element can be defined to describe the languages supported; which 
     * languages are required and which are selectable; and how language selections are grouped.<br><br>
     * Languages defined in the Mandatory element under Languages are always in scope. Languages defined in the Optional element 
     * under Languages are in scope if selected by the deployer.<br><br>
     * The Languages element is used to declare the mandatory and optional language support available in the package. Languages 
     * whose support is deployed by LocalizationUnits in LocalizationContent MUST be defined as either a mandatory language or an optional language. In addition, languages whose support is deployed along with other content by InstallableUnits in BaseContent or SelectableContent SHOULD be defined as a mandatory language.
     * 
     * @return A collection of LanguageSets. 
     */
    Collection<LanguageSet> getOptionalLanguageSets();
    /**
     * Gets resulting resources.
     * 
     * The software whose deployment is described by the SDD can be described in the CompositeInstallables ResultingResource element. 
     * This software may consist of many resources that are described in the ResultingResource elements of the InstallableUnits 
     * and/or LocalizationUnits defined within the CompositeInstallable.
     * 
     * @return A collection of resources that result from applying the CompositeInstallable.
     */
    Collection<ResultingResource> getResultingResources();
    /**
     * Gets resulting changes.
     * 
     * Configuration changes that result from deployment regardless of selected content or condition evaluation can be described 
     * in the CompositeInstallables ResultingChange element.<br><br>
     * Note that a ResultingChange is a change that is made to an existing resource. This is in contrast with ResultingResource, 
     * which describes newly created resources.
     * 
     * @return A collection of configuration changes that result from applying the CompositeInstallable.
     */
    Collection<ResultingChange> getResultingChanges();
    /**
     * Gets the base installable units.
     * 
     * The InstallableUnit element is an atomic content element that defines artifacts that install or update software and defines 
     * requirements for applying those artifacts. It may also define artifacts that undo an update or that uninstall or repair 
     * existing software.
     * 
     * @return A collection of InstallableUnits that define base content.
     */
    Collection<InstallableUnit> getBaseInstallableUnits();
    /**
     * Gets the base configuration units.
     * 
     * The ConfigurationUnit element defines artifacts that configure one or more existing resources. It also defines the requirements 
     * for applying those artifacts. It MUST NOT install, update, or uninstall resources. 
     * 
     * @return ConfigurationUnits that define base configuration content.
     */
    Collection<ConfigurationUnit> getBaseConfigurationUnits();
    /**
     * Gets the base composite units.
     * 
     * The CompositeUnit element is used to organize content elements within the base or selectable content hierarchies. It can 
     * define any number of InstallableUnits, ConfigurationUnits, ContainedPackages and other CompositeUnits. Composite units assist 
     * in organizing the deployment package. A composite unit can provide a convenient way to specify variables, requirements, 
     * conditions and other information that applies to every content element defined below the composite unit. Within the selectable 
     * content hierarchy, composite units can be used to group content elements that are selected by feature sets or groups. When a 
     * feature containing a composite unit is selected, all its child content elements are selected by association. Organization of 
     * content within a composite unit does not imply any relationships among the resources that result from deployment of the 
     * composite content. 
     * 
     * @return CompositeUnits that organize base content.
     */
    Collection<CompositeUnit> getBaseCompositeUnits();
    /**
     * Gets the base contained packages.
     * 
     * A referenced package identifies an aggregated SDD and describes the conditions of its aggregation. ReferencedPackageType 
     * provides the type definition for ContainedPackage and Requisite elements. ContainedPackage elements identify an SDD package 
     * that is treated like a content element of the defining SDD. Requisite elements identify an SDD package that can be deployed, 
     * if necessary, to satisfy resource constraints.
     * 
     * @return A collection of SDD packages whose content is considered to be base content in the context of this aggregation.
     */
    Collection<ReferencedPackage> getBaseContainedPackages();
    /**
     * Gets the groups.
     * 
     * Groups can be used by the SDD author to define a convenient way for deployers to select a group of features.<br><br>
     * Typical and Custom are examples of groups that are commonly presented in installation interfaces.
     * 
     * @return Groups of features that can be selected as a unit.
     */
    Collection<Group> getGroups();
    /**
     * Gets the features.
     * 
     * Features can be used to organize optional functionality into meaningful selections. Features should be meaningful from the 
     * deployers point of view.
     * 
     * @return A collection of user-selectable content.
     */
    Collection<Feature> getFeatures();
    /**
     * Gets the multi-select list of feature references.
     * 
     * A list of feature references whose selection is controlled as a multi-select list with 
     * defined minimum and maximum selections.
     * 
     * @return A collection of MultiFeatureSelects.
     */
    Collection<MultiFeatureSelect> getMultiFeatureSelects();
    /**
     * Gets the selectable installable units.
     * 
     * The InstallableUnit element is an atomic content element that defines artifacts that install or update software and defines 
     * requirements for applying those artifacts. It may also define artifacts that undo an update or that uninstall or repair 
     * existing software.
     * 
     * @return A collection of InstallableUnits that define selectable content.
     */
    Collection<InstallableUnit> getSelectableInstallableUnits();
    /**
     * Gets the selectable configuration units.
     * 
     * The ConfigurationUnit element defines artifacts that configure one or more existing resources. It also defines the requirements 
     * for applying those artifacts. It MUST NOT install, update, or uninstall resources. 
     * 
     * @return A collection of ConfigurationUnits that define selectable configuration.
     */
    Collection<ConfigurationUnit> getSelectableConfigurationUnits();
    /**
     * Gets the selectable composite units.
     * 
     * The CompositeUnit element is used to organize content elements within the base or selectable content hierarchies. It can define 
     * any number of InstallableUnits, ConfigurationUnits, ContainedPackages and other CompositeUnits. Composite units assist in 
     * organizing the deployment package. A composite unit can provide a convenient way to specify variables, requirements, conditions 
     * and other information that applies to every content element defined below the composite unit. Within the selectable content 
     * hierarchy, composite units can be used to group content elements that are selected by feature sets or groups. When a feature 
     * containing a composite unit is selected, all its child content elements are selected by association. Organization of content 
     * within a composite unit does not imply any relationships among the resources that result from deployment of the composite 
     * content. 
     * 
     * @return A collection of CompositeUnits that organize content elements that define selectable content.
     */
    Collection<CompositeUnit> getSelectableCompositeUnits();
    /**
     * Gets the selectable contained packages.
     * 
     * A referenced package identifies an aggregated SDD and describes the conditions of its aggregation. ReferencedPackageType 
     * provides the type definition for ContainedPackage and Requisite elements. ContainedPackage elements identify an SDD package 
     * that is treated like a content element of the defining SDD. Requisite elements identify an SDD package that can be deployed, 
     * if necessary, to satisfy resource constraints.
     * 
     * @return A collection of SDD packages whose content is selectable in the context of the aggregating SDD.
     */
    Collection<ReferencedPackage> getSelectableContainedPackages();
    /**
     * Gets the localization units.
     * 
     * When there is no need to group a LocalizationUnit with other units that have common metadata, the LocalizationUnit is defined 
     * at the top level of the hierarchy. A LocalizationUnit defined at the top level of the LocalizationContent hierarchy is in scope 
     * for a particular deployment when its Condition and LocalizationBase, if any, evaluate to true and its Languages element, if any, 
     * defines a language that is in scope for the deployment. 
     * 
     * @return A collection of LocalizationUnits that contain artifacts that create, modify or delete language support.
     */
    Collection<LocalizationUnit> getLocalizationUnits();
    /**
     * Gets the contained localization packages.
     * 
     * ContainedLocalizationPackage definitions include a list of languages supported by the contained package. The package need not 
     * be processed if none of those languages is in scope for a particular deployment.
     * 
     * @return A collection of SDDs whose contents are aggregated to create, modify or delete language support.
     */
    Collection<ReferencedPackage> getContainedLocalizationPackages();
    /**
     * Gets composite localization units.
     * 
     * CompositeLocalizationUnit is a construct that allows organization of localization content in a way that is meaningful to the 
     * SDD author.<br><br>
     * One example use of a CompositeLocalizationUnit is to group a set of LocalizationUnits that provide support for a variety of 
     * languages for the same resource. This eliminates the need to define identical LocalizationBase elements in every LocalizationUnit. 
     * It can be defined once in the CompositeLocalizationUnit.<br><br>
     * If evaluation of the CompositeLocalizationUnit's Condition, Languages and LocalizationBase determines that it is not selected for 
     * deployment, none of the content elements defined below it in the hierarchy are selected.<br><br>
     * Requirements, Variables, Conditions and Completion elements common to all child content elements MAY be defined once in the 
     * CompositeLocalizationUnit rather than once in each nested element.
     * 
     * @return A collection of CompositeLocalizationUnits that group localization content and defines metadata common to all the grouped content. 
     */
    Collection<CompositeLocalizationUnit> getCompositeLocalizationUnits();
    
    /**
     * Sets the operation.
     * 
     * This is the operation that may be applied to the SDD package whose metadata is described by the CompositeInstallable.
     * 
     * @param operation The deployment lifecycle operation described by the CompositeInstallable definition.
     * @throws InvalidInputException if the parameter is null
     */
    void setOperation(String operation);
    /**
     * Sets the condition.
     * 
     * When the condition defined in the CompositeInstallable is not met for a particular deployment, the CompositeUnit and all the 
     * content elements defined below the CompositeUnit are out of scope for that particular deployment.
     * 
     * @param condition A condition that determines if the content of the CompositeInstallable is relevant to a particular deployment.
     */
    void setCondition(Condition condition);
    /**
     * Sets the required base.
     * 
     * When a resource or resources corresponding to the overall software will be modified during deployment, that resource or those 
     * resources MAY be defined in the RequiredBase element. The RequiredBase definition represents a requirement that the described 
     * resource be available for modification to apply the single operation defined by the CompositeInstallable. When RequiredBase is 
     * defined, the operation defined by CompositeInstallable MUST be one of the following: update, undo, uninstall, or repair. By 
     * specifying the required base separately from other requirements, it is possible for consumers of the SDD to easily determine 
     * if the base is available before processing other requirements.
     * 
     * @param requiredBase Resource or resources that can be updated by the CompositeInstallable.
     */
    void setRequiredBase(RequiredBase requiredBase);
    /**
     * Sets the requirements.
     * 
     * These are requirements that must be met regardless of what content is selected for deployment and which conditions within the 
     * content hierarchy evaluates to true.<br><br>
     * Requirements that apply only to a portion of the content SHOULD be defined at the point in the content hierarchy where they apply.<br><br>
     * All requirements specified on content elements that are in scope for a particular deployment MUST be met. This represents a 
     * logical AND of the requirements. Care should be taken by the SDD author to ensure that conflicting requirements cannot be in 
     * scope for the same deployment.  
     * 
     * @param requirements A collection of Requirements that must be met before successful application of the CompositeInstallable.
     */
    void setRequirements(Collection<Requirement> requirements);
    /**
     * Sets the mandatory languages.
     *  
     * When the SDD contains language support, the Languages element can be defined to describe the languages supported; which languages 
     * are required and which are selectable; and how language selections are grouped.<br><br>
     * Languages defined in the Mandatory element under Languages are always in scope. Languages defined in the Optional element under 
     * Languages are in scope if selected by the deployer.<br><br>
     * The Languages element is used to declare the mandatory and optional language support available in the package. Languages whose 
     * support is deployed by LocalizationUnits in LocalizationContent MUST be defined as either a mandatory language or an optional 
     * language. In addition, languages whose support is deployed along with other content by InstallableUnits in BaseContent or 
     * SelectableContent SHOULD be defined as a mandatory language.
     * 
     * @param languages A collection of required languages.
     */
    void setMandatoryLanguages(Collection<Language> languages);
    /**
     * Sets the optional languages.
     * 
     * When the SDD contains language support, the Languages element can be defined to describe the languages supported; which languages 
     * are required and which are selectable; and how language selections are grouped.<br><br>
     * Languages defined in the Mandatory element under Languages are always in scope. Languages defined in the Optional element under 
     * Languages are in scope if selected by the deployer.<br><br>
     * The Languages element is used to declare the mandatory and optional language support available in the package. Languages whose 
     * support is deployed by LocalizationUnits in LocalizationContent MUST be defined as either a mandatory language or an optional 
     * language. In addition, languages whose support is deployed along with other content by InstallableUnits in BaseContent or 
     * SelectableContent SHOULD be defined as a mandatory language.
     * 
     * @param languages A collection of selectable languages.
     */
    void setOptionalLanguages(Collection<Language> languages);
    /**
     * Sets the optional language sets.
     * 
     * When the SDD contains language support, the Languages element can be defined to describe the languages supported; which 
     * languages are required and which are selectable; and how language selections are grouped.<br><br>
     * Languages defined in the Mandatory element under Languages are always in scope. Languages defined in the Optional element 
     * under Languages are in scope if selected by the deployer.<br><br>
     * The Languages element is used to declare the mandatory and optional language support available in the package. Languages 
     * whose support is deployed by LocalizationUnits in LocalizationContent MUST be defined as either a mandatory language or an 
     * optional language. In addition, languages whose support is deployed along with other content by InstallableUnits in BaseContent 
     * or SelectableContent SHOULD be defined as a mandatory language.
     * 
     * @param languageSets A collections of LanguageSets.
     */
    void setOptionalLanguageSets(Collection<LanguageSet> languageSets);
    /**
     * Sets the resulting resources.
     * 
     * The software whose deployment is described by the SDD can be described in the CompositeInstallables ResultingResource 
     * element. This software may consist of many resources that are described in the ResultingResource elements of the 
     * InstallableUnits and/or LocalizationUnits defined within the CompositeInstallable.
     * 
     * @param resources A collection of resources that result from applying the CompositeInstallable.
     */
    void setResultingResources(Collection<ResultingResource> resources);
    /**
     * Sets the resulting changes.
     * 
     * Configuration changes that result from deployment regardless of selected content or condition evaluation can be described 
     * in the CompositeInstallables ResultingChange element.<br><br>
     * Note that a ResultingChange is a change that is made to an existing resource. This is in contrast with ResultingResource, 
     * which describes newly created resources.
     * 
     * @param changes A collection of configuration changes that result from applying the CompositeInstallable.
     */
    void setResultingChanges(Collection<ResultingChange> changes);
    /**
     * Sets the base installable units.
     * 
     * The InstallableUnit element is an atomic content element that defines artifacts that install or update software and defines 
     * requirements for applying those artifacts. It may also define artifacts that undo an update or that uninstall or repair 
     * existing software.
     * 
     * @param installableUnits A collection of InstallableUnits that define base content.
     */
    void setBaseInstallableUnits(Collection<InstallableUnit> installableUnits);
    /**
     * Sets the base configuration units.
     * 
     * The ConfigurationUnit element defines artifacts that configure one or more existing resources. It also defines the requirements 
     * for applying those artifacts. It MUST NOT install, update, or uninstall resources. 
     * 
     * @param configurationUnits A collection of ConfigurationUnits that define base configuration content.
     */
    void setBaseConfigurationUnits(Collection<ConfigurationUnit> configurationUnits);
    /**
     * Sets the base composite units.
     * 
     * The CompositeUnit element is used to organize content elements within the base or selectable content hierarchies. It can 
     * define any number of InstallableUnits, ConfigurationUnits, ContainedPackages and other CompositeUnits. Composite units assist 
     * in organizing the deployment package. A composite unit can provide a convenient way to specify variables, requirements, 
     * conditions and other information that applies to every content element defined below the composite unit. Within the selectable 
     * content hierarchy, composite units can be used to group content elements that are selected by feature sets or groups. When a 
     * feature containing a composite unit is selected, all its child content elements are selected by association. Organization of 
     * content within a composite unit does not imply any relationships among the resources that result from deployment of the 
     * composite content. 
     * 
     * @param compositeUnits A collection of CompositeUnits that organize base content.
     */
    void setBaseCompositeUnits(Collection<CompositeUnit> compositeUnits);
    /**
     * Sets the base contained packages.
     * 
     * A referenced package identifies an aggregated SDD and describes the conditions of its aggregation. ReferencedPackageType 
     * provides the type definition for ContainedPackage and Requisite elements. ContainedPackage elements identify an SDD package 
     * that is treated like a content element of the defining SDD. Requisite elements identify an SDD package that can be deployed, 
     * if necessary, to satisfy resource constraints.
     * 
     * @param containedPackages A collection of SDDs whose content is considered to be base content in the context of this aggregation.
     */
    void setBaseContainedPackages(Collection<ReferencedPackage> containedPackages);
    /**
     * Sets the groups.
     * 
     * Groups can be used by the SDD author to define a convenient way for deployers to select a group of features.<br><br>
     * Typical and Custom are examples of groups that are commonly presented in installation interfaces.
     * 
     * @param groups A collection of Groups of features that can be selected as a unit.
     */
    void setGroups(Collection<Group> groups);
    /**
     * Sets the features.
     * 
     * Features can be used to organize optional functionality into meaningful selections. Features should be meaningful from the 
     * deployers point of view.
     * 
     * @param features A collection of user-selectable content.
     */
    void setFeatures(Collection<Feature> features);
    
    /**
     * Sets the multi-select lists of feature references.
     * 
     * MultiSelect in FeaturesType is a list of feature references whose selection is controlled as a multi-select list with 
     * defined minimum and maximum selections.
     * 
     * @param A collection of MultiFeatureSelects.
     * 
     */
    void setMultiFeatureSelects(Collection<MultiFeatureSelect> multiFeatureSelects);
    
    /**
     * Sets the selectable installable units.
     * 
     * The InstallableUnit element is an atomic content element that defines artifacts that install or update software and defines 
     * requirements for applying those artifacts. It may also define artifacts that undo an update or that uninstall or repair 
     * existing software.
     * 
     * @param installableUnits A collection of InstallableUnits that define selectable content.
     */
    void setSelectableInstallableUnits(Collection<InstallableUnit> installableUnits);
    /**
     * Sets the selectable configuration units.
     * 
     * The ConfigurationUnit element defines artifacts that configure one or more existing resources. It also defines the 
     * requirements for applying those artifacts. It MUST NOT install, update, or uninstall resources. 
     * 
     * @param configurationUnits A collection of ConfigurationUnits that define selectable configuration.
     */
    void setSelectableConfigurationUnits(Collection<ConfigurationUnit> configurationUnits);
    /**
     * Sets the selectable composite units.
     * 
     * The CompositeUnit element is used to organize content elements within the base or selectable content hierarchies. It can 
     * define any number of InstallableUnits, ConfigurationUnits, ContainedPackages and other CompositeUnits. Composite units assist 
     * in organizing the deployment package. A composite unit can provide a convenient way to specify variables, requirements, 
     * conditions and other information that applies to every content element defined below the composite unit. Within the selectable 
     * content hierarchy, composite units can be used to group content elements that are selected by feature sets or groups. When a 
     * feature containing a composite unit is selected, all its child content elements are selected by association. Organization of 
     * content within a composite unit does not imply any relationships among the resources that result from deployment of the 
     * composite content. 
     * 
     * @param compositeUnits A collection of CompositeUnits that organize content elements that define selectable content.
     */
    void setSelectableCompositeUnits(Collection<CompositeUnit> compositeUnits);
    /**
     * Sets the selectable contained packages.
     * 
     * A referenced package identifies an aggregated SDD and describes the conditions of its aggregation. ReferencedPackageType 
     * provides the type definition for ContainedPackage and Requisite elements. ContainedPackage elements identify an SDD package 
     * that is treated like a content element of the defining SDD. Requisite elements identify an SDD package that can be deployed, 
     * if necessary, to satisfy resource constraints.
     * 
     * @param containedPackages A collection of SDD packages whose content is selectable in the context of the aggregating SDD.
     */
    void setSelectableContainedPackages(Collection<ReferencedPackage> containedPackages);
    /**
     * Sets the localization units.
     * 
     * When there is no need to group a LocalizationUnit with other units that have common metadata, the LocalizationUnit is defined 
     * at the top level of the hierarchy. A LocalizationUnit defined at the top level of the LocalizationContent hierarchy is in scope 
     * for a particular deployment when its Condition and LocalizationBase, if any, evaluate to true and its Languages element, if any, 
     * defines a language that is in scope for the deployment. 
     * 
     * @param localizationUnits A collection of LocalizationUnits that contain artifacts that create, modify or delete language support.
     */
    void setLocalizationUnits(Collection<LocalizationUnit> localizationUnits);
    /**
     * Sets contained localization packages.
     * 
     * ContainedLocalizationPackage definitions include a list of languages supported by the contained package. The package need not 
     * be processed if none of those languages is in scope for a particular deployment.
     * 
     * @param packages A collection of SDDs whose contents are aggregated to create, modify or delete language support.
     */
    void setContainedLocalizationPackages(Collection<ReferencedPackage> packages);
    /**
     * Sets the composite localization units.
     * 
     * CompositeLocalizationUnit is a construct that allows organization of localization content in a way that is meaningful to the SDD 
     * author.<br><br>
     * One example use of a CompositeLocalizationUnit is to group a set of LocalizationUnits that provide support for a variety of languages 
     * for the same resource. This eliminates the need to define identical LocalizationBase elements in every LocalizationUnit. It can be 
     * defined once in the CompositeLocalizationUnit.<br><br>
     * If evaluation of the CompositeLocalizationUnit's Condition, Languages and LocalizationBase determines that it is not selected for 
     * deployment, none of the content elements defined below it in the hierarchy are selected.<br><br>
     * Requirements, Variables, Conditions and Completion elements common to all child content elements MAY be defined once in the 
     * CompositeLocalizationUnit rather than once in each nested element.
     * 
     * @param localizationUnits A collection of CompositeLocalizationUnits that groups localization content and defines metadata common to all the grouped 
     * content.
     */
    void setCompositeLocalizationUnits(Collection<CompositeLocalizationUnit> localizationUnits);
    
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
