/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.DisplayNameable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * A Requirement either directly defines a single set of resource constraints that MUST be met or defines one or more alternative 
 * sets of resource constraints, only one of which MUST be met.
 * 
 * When multiple Requirement elements are declared for the same operation, all MUST be met prior to processing the associated artifact.
 * The association is made between a requirement and an artifact via the operation attribute.
 */
public interface Requirement extends DisplayNameable {
    /**
     * Gets the operations.
     * 
     * A Requirement is associated with application of one or more operations by setting its operation attribute value to one of the 
     * enumerated values defined in OperationListType.<br><br>
     * If the Requirement is not a pre-requisite for application of an operation, but rather is required before the resulting resources 
     * are considered usable, then the value SHOULD be set to use. (Note that a completion action may also be required before a 
     * resulting resource is considered usable. See the CompletionType section.)<br><br>
     * The value of operation for a Requirement defined in an atomic content element MUST be set either to use or to an operation that 
     * is associated with an artifact element defined in the content elements Artifacts. The operation value(s) associate the 
     * Requirement with one or more artifact(s).<br><br>
     * When the Requirement is specified in a CompositeUnit or CompositeInstallable, the operation value MUST be set either to use or 
     * be the same top level operation as defined in the CompositeInstallable element.<br><br>
     * There is no default value for operation. The SDD author must define it explicitly.
     * 
     * @return Requirement must be met before this collection of operations is performed.
     */
    Collection<String> getOperations();
	
	/**
	 * Gets the requirement ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @return Identifier for requirement scoped to the deployment descriptor.
	 */
	String getRequirementID();
	
	/**
	 * Gets the resource constraints.
	 * 
	 * When a requirement can be satisfied in only one way, constraints MAY be defined directly under Requirement or in a single 
	 * Alternative element.<br><br>
	 * Constraints are defined using a sequence of ResourceConstraints. Every constraint in the sequence MUST be met for the 
	 * requirement to be met. 
	 * 
	 * @return A collection of constraints on one resource.
	 */
	Collection<RequirementResourceConstraint> getResourceConstraints();
	
	/**
	 * Gets the alternative requirements.
	 * 
	 * Alternative elements are used when a requirement can be satisfied in multiple ways.<br><br>
	 * As a convenience for tooling that produces SDDs, it is also possible to define a single Alternative. This is semantically 
	 * identical to directly defining ResourceConstraints under Requirements.<br><br>
	 * To satisfy a requirement, at least one of the specified alternatives MUST be satisfied.
	 * 
	 * @return A collection of alternatives that can satisfy the requirement.
	 */
	Collection<AlternativeRequirement> getAlternativeRequirements();
	          
	/**
	 * Sets the operations.
	 * 
	 * A Requirement is associated with application of one or more operations by setting its operation attribute value to one of the 
     * enumerated values defined in OperationListType.<br><br>
     * If the Requirement is not a pre-requisite for application of an operation, but rather is required before the resulting resources 
     * are considered usable, then the value SHOULD be set to use. (Note that a completion action may also be required before a 
     * resulting resource is considered usable. See the CompletionType section.)<br><br>
     * The value of operation for a Requirement defined in an atomic content element MUST be set either to use or to an operation that 
     * is associated with an artifact element defined in the content elements Artifacts. The operation value(s) associate the 
     * Requirement with one or more artifact(s).<br><br>
     * When the Requirement is specified in a CompositeUnit or CompositeInstallable, the operation value MUST be set either to use or 
     * be the same top level operation as defined in the CompositeInstallable element.<br><br>
     * There is no default value for operation. The SDD author must define it explicitly.
     * 
	 * @param operations Requirement must be met before this collection of operations is performed.
	 */
	void setOperations(Collection<String> operations);
	
	/**
	 * Sets the requirement ID.
	 * 
	 * The id attribute may be useful to software that processes the SDD, for example, for use in creating log and trace messages.
	 * 
	 * @param requirementID Identifier for requirement scoped to the deployment descriptor.
	 * @throws InvalidInputException if the parameter is null
	 */
	void setRequirementID(String requirementID);
	
	/**
	 * Sets the resource constraints.
	 * 
	 * When a requirement can be satisfied in only one way, constraints MAY be defined directly under Requirement or in a single 
	 * Alternative element.<br><br>
	 * Constraints are defined using a sequence of ResourceConstraints. Every constraint in the sequence MUST be met for the 
	 * requirement to be met.
	 * 
	 * @param resourceConstraints A collection of constraints on one resource.
	 */
	void setResourceConstraints(Collection<RequirementResourceConstraint> resourceConstraints);
	
	/**
	 * Sets the alternative requirements.
	 * 
	 * Alternative elements are used when a requirement can be satisfied in multiple ways.<br><br>
	 * As a convenience for tooling that produces SDDs, it is also possible to define a single Alternative. This is semantically 
	 * identical to directly defining ResourceConstraints under Requirements.<br><br>
	 * To satisfy a requirement, at least one of the specified alternatives MUST be satisfied.
	 * 
	 * @param alternativeRequirements A collection of alternatives that can satisfy the requirement.
	 */
	void setAlternativeRequirements(
	        Collection<AlternativeRequirement> alternativeRequirements);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	Object clone();
}