/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement;

import java.util.Collection;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;

/**
 * A RequirementFactory is used to create objects representing or directly
 * related to requirement.
 *
 */
public interface RequirementFactory {
	/**
	 * Creates a new Requirement.
	 * 
	 * @param requirementID Value to use for the id attribute of the Requirement
	 * @return A new Requirement
	 * @throws InvalidInputException if requirementID is null
	 */
	Requirement createRequirement(String requirementID) throws InvalidInputException;
	
	/**
	 * Creates a new AlternativeRequirement.
	 * 
	 * @param alternativeRequirementID Value to use for the id attribute of the AlternativeRequirement
	 * @return A new AlternativeRequirement
	 * @throws InvalidInputException if alternativeRequirementID is null
	 */
	AlternativeRequirement createAlternativeRequirement(String alternativeRequirementID) throws InvalidInputException;
	
	/**
	 * Creates a new RequirementResourceConstraint.
	 * 
	 * @param resourceConstraintID Value to use for the id attribute of the RequirementResourceConstraint
	 * @param resourceReference Value to use to for resourceRef attribute of the RequirementResourceConstraint
	 * @return A new RequirementResourceConstraint
	 * @throws InvalidInputException if resourceConstraintID or resourceReference is null
	 */
	RequirementResourceConstraint createRequirementResourceConstraint(
			String resourceConstraintID, String resourceReference) throws InvalidInputException;
	
	/**
	 * Creates a new CapacityConstraint.
	 * 
	 * 
	 * @param propertyName Value to use for the PropertyName element of the CapacityConstraint
	 * @return A new CapacityConstraint
	 * @throws InvalidInputException if propertyName is null
	 */
	CapacityConstraint createCapacityConstraint(String propertyName) throws InvalidInputException;
	
	/**
	 * Creates a new PropertyConstraint with multiple values.
	 * 
	 *  
	 * @param propertyName Value to use for the PropertyName element of the PropertyConstraint
	 * @param propertyValues Values to use for the ListOfValues element of the PropertyConstraint
	 * @return A new PropertyConstraint
	 * @throws InvalidInputException if propertyName is null, or propertyValues is null or empty
	 */
	PropertyConstraint createPropertyConstraintMultiValue(String propertyName, Collection<String> propertyValues) 
			throws InvalidInputException;
	
	/**
	 * Creates a new PropertyConstraint with single value.
	 * 
	 * 
	 * @param propertyName Value to use for the PropertyName element of the PropertyConstraint
	 * @param value Value to use for the Value element of the PropertyConstraint
	 * @return A new PropertyConstraint
	 * @throws InvalidInputException if propertyName or value is null
	 */
	PropertyConstraint createPropertyConstraintSingleValue(String propertyName, String value)
			throws InvalidInputException;

	/**
	 * Creates a new ConsumptionConstraint.
	 * 
	 * 
	 * @param propertyName Value to use for the PropertyName element of the ConsumptionConstraint
	 * @return A new ConsumptionConstraint
	 * @throws InvalidInputException if propertyName is null
	 */
	ConsumptionConstraint createConsumptionConstraint(String propertyName) throws InvalidInputException;
	
	/**
	 * Creates a new VersionConstraint.
	 * 
	 * 
	 * @return A new VersionConstraint
	 */
	VersionConstraint createVersionConstraint();

	/**
	 * Creates a new UniquenessConstraint.
	 * 
	 * 
	 * @param distinctResourceRef Value to use for the distinctResourceRef attribute of the UniquenessConstraint
	 * @return A new UniquenessConstraint
	 * @throws InvalidInputException if distinctResourceRef is null
	 */
	UniquenessConstraint createUniquenessConstraint(String distinctResourceRef) throws InvalidInputException;
	
	/**
	 * Creates a new RelationshipConstraint.
	 * 
	 * 
	 * @param type Value to use for the type attribute of the RelationshipConstraint
	 * @return A new RelationshipConstraint
	 * @throws InvalidInputException if type is null
	 */
	RelationshipConstraint createRelationshipConstraint(String type) throws InvalidInputException;
	
	/**
	 * Creates a new InternalDependency.
	 *  
	 * @param contentRef Value to use for the contentRef attribute of the InternalDependency
	 * @return A new InternalDependency
	 * @throws InvalidInputException if contentRef is null
	 */
	InternalDependency createInternalDependency(String contentRef) throws InvalidInputException;
}
