/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.smlvalidators;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Iterator;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.xpath.XPathFunctionException;

import org.apache.xerces.dom.PSVIElementNSImpl;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSNamedMap;
import org.apache.xerces.xs.XSTypeDefinition;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ConstraintNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.DOMStructure;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementLocation;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementTypeMap;
import org.eclipse.cosmos.rm.internal.validation.artifacts.EmbeddedTargetElementNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TargetElementNode;
import org.eclipse.cosmos.rm.internal.validation.artifacts.TypeDeclarationCollection;
import org.eclipse.cosmos.rm.internal.validation.common.ISMLConstants;
import org.eclipse.cosmos.rm.internal.validation.common.IValidationOutput;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidationMessages;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessage;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DataBuilderRegistry;
import org.eclipse.cosmos.rm.internal.validation.databuilders.DocumentDOMBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.ElementDeclarationBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TargetInstanceBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.TypeDeclarationBuilder;
import org.eclipse.cosmos.rm.internal.validation.databuilders.AbstractDataBuilder.AnnotationResult;
import org.eclipse.cosmos.rm.internal.validation.reference.DerefXPathFunction;
import org.eclipse.cosmos.rm.internal.validation.reference.URIReference;
import org.eclipse.cosmos.rm.internal.validation.util.ParserHelper;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.DOMException;
import org.w3c.dom.Node;

/**
 * This class is used to validate the sml:targetElement, sml:targetRequired and
 * sml:targetType constraints.
 *
 * @author sleeloy
 * @author Ali Mehregani
 * @author David Whiteman
 * @author John Arwe
 */
public class TargetValidator extends AbstractSMLValidator
{
	private Map<String, ConstraintNode> targetSchemaBuilderStructure;
	private Map<TargetElementNode, String> targetInstanceBuilderStructure;
	private DOMStructure documentDOMBuilderElems;
	private Map<String, Map<String, String>> substitutionElementElems;
	private ElementTypeMap elementTypeMapBuilderStructure;
	private TypeDeclarationBuilder typeDeclarationBuilder;
	private IValidationOutput<String, Object> validationLogger;



	public void initialize(Map<String, Object> validationAttribute)
	{
		super.initialize(validationAttribute);

		DataBuilderRegistry builderRegistry = DataBuilderRegistry.getInstanceLevelRegistry();

		builderRegistry.registerDataStructureBuilder(TargetInstanceBuilder.ID, new TargetInstanceBuilder());
		builderRegistry.registerDataStructureBuilder(DocumentDOMBuilder.ID, new DocumentDOMBuilder());
		builderRegistry.registerDataStructureBuilder(TypeDeclarationBuilder.ID, new TypeDeclarationBuilder());
		builderRegistry.registerDataStructureBuilder(ElementDeclarationBuilder.ID, new ElementDeclarationBuilder());
	}

	public boolean validate()
	{
		setTaskName(SMLValidationMessages.validationTarget);
		validationLogger = getValidationOutput();
		documentDOMBuilderElems = (DOMStructure)SMLValidatorUtil.retrieveDataStructure(DocumentDOMBuilder.ID);
		TargetInstanceBuilder targetInstanceBuilder = (TargetInstanceBuilder) DataBuilderRegistry.getInstanceLevelRegistry().getDataStructureBuilder(TargetInstanceBuilder.ID);
		targetInstanceBuilderStructure = (Map<TargetElementNode, String>) targetInstanceBuilder.getDataStructure();
		targetSchemaBuilderStructure = (Map<String, ConstraintNode>) targetInstanceBuilder.getTargetSchemaBuilderStructure();
		substitutionElementElems = (Map<String, Map<String, String>>) targetInstanceBuilder.getSubstitutionGroupStructure();
		elementTypeMapBuilderStructure = targetInstanceBuilder.getElementTypeMap();
		typeDeclarationBuilder = (TypeDeclarationBuilder)DataBuilderRegistry.getInstanceLevelRegistry().getDataStructureBuilder(TypeDeclarationBuilder.ID);

		// Check the syntax first
		if (!isSyntaxValid())
		{
			if (shouldAbortOnError())
			{
				return false;
			}
		}

		// Validate the target constraints
		return validateValues() && validateTargetInstance();
	}


	/**
	 * Verifies that the values specified for sml:targetElement and sml:targetType
	 * are valid (i.e. The values exist in the scope of the document)
	 *
	 * @return true if validation passes; false otherwise
	 */
	private boolean validateValues()
	{
		for (Iterator<String> elementNames = targetSchemaBuilderStructure.keySet().iterator(); elementNames.hasNext();)
		{
			String elementName = elementNames.next();
			ConstraintNode schemaNode = targetSchemaBuilderStructure.get(elementName);

			QName targetElement = schemaNode.getTargetElement();
			if (targetElement != null && elementTypeMapBuilderStructure.getType(targetElement.getNamespaceURI(), targetElement.getLocalPart()) == null)
			{
				ElementLocation location = schemaNode.getLocation();
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(
						location.getFilePath(), location.getLineNumber(), NLS.bind(SMLValidationMessages.targetElementInvalidValue, targetElement.getPrefix() + ":" + targetElement.getLocalPart())));
				return false;
			}

			QName targetType = schemaNode.getTargetType();
			TypeDeclarationCollection typeCollection = typeDeclarationBuilder.getDataStructure();
			if (targetType != null && typeCollection.get(targetType) == null)
			{
				ElementLocation location = schemaNode.getLocation();
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(
						location.getFilePath(), location.getLineNumber(), NLS.bind(SMLValidationMessages.targetTypeInvalidValue, targetType.getPrefix() + ":" + targetType.getLocalPart())));
				return false;
			}
		}
		return true;
	}


	protected boolean validateTargetInstance()
	{
		Iterator<TargetElementNode> iter = targetInstanceBuilderStructure.keySet().iterator();

		// Check to make sure the type declarations are consistent
		if (!typeDeclarationBuilder.isStructureValid())
		{
			validationLogger.reportMessage(typeDeclarationBuilder.getErrorMessage());
			return false;
		}

		// For each element that has an associated target* constraint
		while (iter.hasNext())
		{
			TargetElementNode targetElementNode = (TargetElementNode)iter.next();
			ConstraintNode targetSchemaNode = (ConstraintNode)targetSchemaBuilderStructure.get(targetElementNode.getName());

			// Check substitution group
			if (targetSchemaNode == null){
				Map<String, String> subGroupMap = substitutionElementElems.get(targetElementNode.getUri());
				if (subGroupMap != null){
					Object subGroup = subGroupMap.get(targetElementNode.getName());
					if (subGroup != null)
						targetSchemaNode = (ConstraintNode)targetSchemaBuilderStructure.get(ParserHelper.removeNameSpace((String)subGroup));
				}
			}

			if (targetElementNode.isRequired()) // If SML reference was governed by an element declaration with sml:targetRequired=true
			{
				if (targetElementNode.isNilref()) { // URI resolution should not be attempted for nil references, and the constraint is automatically unsatisfied
					Object[] errorParams = new String[3];
					errorParams[0] = targetElementNode.getName();
					errorParams[1] = targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
					errorParams[2] = targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
					validationLogger.reportMessage(
							ValidationMessageFactory.createErrorMessage(
									targetElementNode.getLocation().getFilePath(),
									targetElementNode.getLocation().getLineNumber(),
									NLS.bind(SMLValidationMessages.unresolvedNilRef,errorParams)
									)
							);
					return false;
				} else if ( !isURISMLDeferenceable(targetElementNode)) {	// Attempt URI resolution for non-nil SML references
					Object[] errorParams = new String[3];
					errorParams[0] = targetElementNode.getName();
					errorParams[1] = targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
					errorParams[2] = targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
					validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.unresolvedTargetURI,errorParams)));
					return false;
				}
			}

			// check if we need to check the target element
			if (targetSchemaNode.getTargetElement() != null){ // should be using targetElementNode, not putative schema node cached using the local name of the SML reference element
				if (!validateTargetElement(targetElementNode, targetSchemaNode))
					return false;
			}
			if (targetSchemaNode.getTargetType() != null){ // should be using targetElementNode, not putative schema node cached using the local name of the SML reference element
				if (!validateTargetType(targetElementNode, targetSchemaNode))
					return false;
			}

		}
		return true;
	}

	protected boolean validateTargetType(TargetElementNode targetElementNode, ConstraintNode targetSchemaNode) {

		String targetElementNamespace = targetElementNode.getUri();
		String targetElementUri;
		try {
			targetElementUri = determineTargetElementURI(targetElementNode);
		} catch (URISyntaxException e) {
			return false;
		}

		// check if schema node matches instance node
		String getTargetType = elementTypeMapBuilderStructure.getType(targetElementNode.getUri(), targetElementUri);
		String targetSchemaNodeTypeName = qNameToString(targetSchemaNode.getTargetType());
		if ((getTargetType != null)){
			if (compareElementInstance(targetSchemaNodeTypeName, getTargetType)) {
				return true;
			}
			if (checkInheritanceTree(targetSchemaNodeTypeName, targetElementUri)) {
				return true;
			}
		}

		String targetTypeUri = null;
		// the ref could be an alias
		// Search through alias data structure for actual element
		Node alias = documentDOMBuilderElems.get(targetElementUri);
		if (alias != null) {
			targetTypeUri = alias.getNodeName();
			targetElementNamespace = alias.getNamespaceURI();
		}

		if (targetTypeUri != null) {
			getTargetType = elementTypeMapBuilderStructure.getType(targetElementNamespace, targetTypeUri);
		}
		if (getTargetType != null) {
			if (compareElementInstance(getTargetType, targetSchemaNodeTypeName)) {
				return true;
			}
			if (checkInheritanceTree(targetSchemaNodeTypeName, targetElementUri)) {
				return true;
			}
		}

		// there could be a substitution group
		// DLW - commenting out for now since this looks like it would have
		// caused a CCE
		// String substitutionGroup =
		// substitutionElementElems.get(targetElementUri);
		boolean valid = false;
		// while (substitutionGroup != null){
		// getTargetType =
		// elementTypeMapBuilderStructure.getType(targetElementUri
		// ,(String)substitutionGroup);
		// if (compareElementInstance(substitutionGroup,
		// qNameToString(targetSchemaNode.getTargetType())))
		// {
		// valid = true;
		// break;
		// }
		// substitutionGroup = substitutionElementElems.get(substitutionGroup);
		// }
		// //if dangling then it's okay
		if (valid || alias == null)
			return true;

		Object[] errorParams = new String[5];
		errorParams[0] = targetElementNode.getName();
		errorParams[1] = targetElementNode.getLocation().getLineNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
		errorParams[2] = targetElementNode.getLocation().getColumnNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
		errorParams[3] = targetElementUri;
		errorParams[4] = targetSchemaNodeTypeName;
		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.targetTypeInvalid,errorParams)));

		return false;
	}

	/**
	 *
	 * @param targetSchemaNode
	 * @param targetElementUri
	 * @param getTargetType
	 * @return
	 */
	private boolean checkInheritanceTree(String targetSchemaNodeTypeName, String targetElementUri) {
		boolean valid = false;
		PSVIElementNSImpl psviElementNode = (PSVIElementNSImpl) documentDOMBuilderElems.get(targetElementUri);
		XSTypeDefinition type = (XSTypeDefinition) psviElementNode.getTypeDefinition();
		while ((type instanceof XSComplexTypeDefinition) && (((XSComplexTypeDefinition) type).getContentType() == XSComplexTypeDefinition.CONTENTTYPE_ELEMENT)) {
			if (compareElementInstance(type.getName(), targetSchemaNodeTypeName)) {
				valid = true;
				break;
			}
			type = type.getBaseType();
		}
		return valid;
	}

	protected boolean validateTargetElement(TargetElementNode targetElementNode, ConstraintNode targetSchemaNode) {

		String targetElementNamespace = targetElementNode.getUri();
		String targetElementUri;
		try {
			targetElementUri = determineTargetElementURI(targetElementNode);
		} catch (URISyntaxException e) {
			return false;
		}

		// check if schema node matches instance node
		if (compareElementInstance(qNameToString(targetSchemaNode.getTargetElement()), targetElementUri)) {
			return true;
		} else {
			// the ref could be an alias
			// Search through alias data structure for actual element
			Node alias = documentDOMBuilderElems.get(targetElementUri);
			if (alias != null){
				targetElementUri = alias.getNodeName();
				targetElementNamespace  = alias.getNamespaceURI();
			}
			if (compareElementInstance(targetElementUri, qNameToString(targetSchemaNode.getTargetElement()))){
				return true;
			}
			// there could be a substitution group
			Map<String, String> substitutionGroupMap = substitutionElementElems.get(targetElementNamespace);
			boolean valid = false;
			if (substitutionGroupMap != null){
				String substitutionGroup = substitutionGroupMap.get(targetElementUri);
				while (substitutionGroup != null){
					if (compareElementInstance(substitutionGroup, qNameToString(targetSchemaNode.getTargetElement()))){
						valid = true;
						break;
					}
					substitutionGroup = ParserHelper.removeNameSpace(substitutionGroup);
					substitutionGroup = substitutionGroupMap.get(substitutionGroup);
				}
			}
			// if dangling then it's okay
			if (valid || alias == null) return true;
		}
		Object[] errorParams = new String[5];
		errorParams[0] = targetElementNode.getName();
		errorParams[1] = targetElementNode.getLocation().getLineNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getLineNumber()+""; //$NON-NLS-1$
		errorParams[2] = targetElementNode.getLocation().getColumnNumber() < 0 ? SMLValidationMessages.commonUnknown : targetElementNode.getLocation().getColumnNumber()+""; //$NON-NLS-1$
		errorParams[3] = targetElementUri;

		QName targetElement = targetSchemaNode.getTargetElement();
		String prefix = targetElement.getPrefix();
		errorParams[4] = prefix != null && prefix.length() > 0 ? prefix + ":" + targetElement.getLocalPart() : targetElement.getLocalPart();

		validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), NLS.bind(SMLValidationMessages.invalidTargetInstance,errorParams)));
		return false;
	}

	private String determineTargetElementURI(TargetElementNode targetElementNode) throws URISyntaxException {
		String targetElementUri = targetElementNode.getTargetURI();

		try {
			Node contextNode = SMLValidatorUtil.locateDocumentNode(targetElementNode.getAliases(), targetElementNode.getDocumentPosition());
			URIReference targetURI = new URIReference(contextNode, targetElementUri, targetElementNode.getBaseURIValue());
			targetElementUri = targetURI.getDocumentReference();
		} catch (URISyntaxException exception) {
			validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(targetElementNode.getLocation().getFilePath(), targetElementNode.getLocation().getLineNumber(), exception.getLocalizedMessage()));
			throw exception;
		} catch (Exception exc) {
			exc.printStackTrace();
		}
		return targetElementUri;
	}

	private String qNameToString(QName qName)
	{
		String prefix = qName.getPrefix();
		return prefix == null || prefix.length() <= 0 ? qName.getLocalPart() : prefix + ":" + qName.getLocalPart();
	}

	/**
	 * Check to make sure the syntax of the identity constraints
	 * is correct.  We need to use the XSModel object to retrieve
	 * all identity constraints.  The reason why the identity constraint
	 * structure can't be used is because there is no guarantee
	 * for all element declarations to have an instance (i.e. You
	 * can have an element declaration E with an invalid identity
	 * constraint that has no instance)
	 *
	 * @return true if the identity constraints are syntactically
	 * correct; false otherwise
	 */
	private boolean isSyntaxValid()
	{
		XSModel xsModel = (XSModel)getAttributes().get(IValidator.ATTRIBUTE_XS_MODEL);
		if (xsModel == null)
		{
			return true;
		}

		XSNamedMap typeDefinitions = xsModel.getComponents(XSConstants.TYPE_DEFINITION);
		boolean shouldAbortOnError = shouldAbortOnError();
		boolean status = true;

		// For every type definition, check for bad declarations
		for (int i = 0, definitionCount = typeDefinitions.getLength();
			 i < definitionCount /*&& (!shouldAbortOnError || status)*/; i++)
		{
			XSTypeDefinition typeDefinition = (XSTypeDefinition)typeDefinitions.item(i);
			checkForBadDeclaration(SMLValidatorUtil.retrieveAnnotation(typeDefinition, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE, false));
			checkForBadDeclaration(SMLValidatorUtil.retrieveAnnotation(typeDefinition, ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE, false));
			checkForBadDeclaration(SMLValidatorUtil.retrieveAnnotation(typeDefinition, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE, false));
		}

		XSNamedMap elementDeclarations = xsModel.getComponents(XSConstants.ELEMENT_DECLARATION);
		// For every element declaration, check for bad values
		for (int i = 0, declarationCount = elementDeclarations.getLength();
			 i < declarationCount && (!shouldAbortOnError || status); i++)
		{
			XSElementDeclaration elementDeclarationToCheck = (XSElementDeclaration)elementDeclarations.item(i);
			AnnotationResult targetElementAnnotationResult = SMLValidatorUtil.retrieveAnnotation(elementDeclarationToCheck, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
			AnnotationResult targetTypeAnnotationResult = SMLValidatorUtil.retrieveAnnotation(elementDeclarationToCheck, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
			status = checkForInvalidTargetElementValue(elementDeclarations, elementDeclarationToCheck, targetElementAnnotationResult);
			if (!shouldAbortOnError || status) {
				status = status && checkForInvalidTargetType(typeDefinitions, elementDeclarations, elementDeclarationToCheck, targetTypeAnnotationResult);
			}
		}
		// clear cache before starting the check for inconsistent declarations and invalid derivations

		if (!shouldAbortOnError || status)
		{
			status = status && checkForInvalidDerivations();
		}
		return status;
	}

	private boolean checkForInvalidTargetType(XSNamedMap allTypeDefs, XSNamedMap globalElementDeclarations, XSElementDeclaration elementDeclarationWithTargetType, AnnotationResult result) throws DOMException {
		boolean status = true;
		if (result != null) {
			Node[] nodes = result.getNodes();
			if (nodes.length > 0) {
				String targetTypeValue = nodes[0].getNodeValue();
				boolean found = false;
				XSTypeDefinition typeDef = null;
				for (int j = 0, typeDefinitionCount = allTypeDefs.getLength();
				 j < typeDefinitionCount && (!found); j++)
				{
					typeDef = (XSTypeDefinition) allTypeDefs.item(j);
					if (found = compareElementInstance(typeDef.getName(), targetTypeValue)) {
						break;
					}
				}
				if (!found) {
					getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
							ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.targetTypeInvalidValue, targetTypeValue)));
					if (shouldAbortOnError()) {
						return false;
					}
				}

			}
		}
		return status;
	}


	protected boolean handleNewElement(XSElementDeclaration elementDeclaration, XSComplexTypeDefinition typeDefinition)
	{
		// If we haven't cached this node yet, we need to determine if it has a target* attribute
		// before we do so
		String qName = SMLValidatorUtil.createQualifiedName(elementDeclaration);
		String currentElementTargetTypeUri = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
		String currentElementTargetElementUri = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
		String currentElementTargetRequired = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE);
		if ((currentElementTargetTypeUri != null) || (currentElementTargetElementUri != null)) {
			byte targetTypeOrElement = (currentElementTargetTypeUri != null) ? TargetElementNode.TYPE_TYPE : TargetElementNode.INSTANCE_TYPE;
			String targetUri = (currentElementTargetTypeUri != null) ? currentElementTargetTypeUri : currentElementTargetElementUri;
			EmbeddedTargetElementNode embeddedTargetElementNode = new EmbeddedTargetElementNode(elementDeclaration, typeDefinition, targetUri, targetTypeOrElement);
			if (currentElementTargetRequired != null) {
				embeddedTargetElementNode.setRequired(SMLValidatorUtil.isTrue(currentElementTargetRequired));
			}
			getConstraintMap().put(qName, embeddedTargetElementNode);
		}
		return true;
	}


	protected boolean handleDiscoveredElement(XSElementDeclaration elementDeclaration, XSComplexTypeDefinition typeDefinition, Object discoveredElement)
	{
		EmbeddedTargetElementNode embeddedTargetElementNode = (EmbeddedTargetElementNode)discoveredElement;
		String currentElementTargetTypeUri = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETTYPE_ATTRIBUTE);
		String currentElementTargetElementUri = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETELEMENT_ATTRIBUTE);
		String targetUri = (currentElementTargetTypeUri != null) ? currentElementTargetTypeUri : currentElementTargetElementUri;
		// target type should be checked to see if it's the same hierarchy, because if it's not,
		// there is no error
		if (currentElementTargetTypeUri != null)
		{
			if (!checkForSameHierarchy(typeDefinition, embeddedTargetElementNode.getEnclosingType()))
			{
				return true;
			}
		}
		// if either targetElement or targetType exists, we should make sure it doesn't
		// equal the value of the cached node that has the same name
		if (targetUri != null) {
			String errorMessage = SMLValidationMessages.targetBadDerivation;
			// If they are the same exact class, then this is a situation where two element nodes
			// that aren't identical but have the same name are enclosed in the same complex type,
			// either directly or indirectly.
			if (typeDefinition == embeddedTargetElementNode.getEnclosingType()) {
				errorMessage = SMLValidationMessages.targetInconsistentDeclarations;
			}
			if (!targetUri.equals(embeddedTargetElementNode.getTargetURI())) {
				// TODO do we need file path info here?  if so, where do we get it from?
				// TODO Also would be good to have accurate location info
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
				return false;
			}
			String currentElementTargetRequired = retrieveAnnotationValue(elementDeclaration, ISMLConstants.SML_URI, ISMLConstants.TARGETREQUIRED_ATTRIBUTE);
			if ((currentElementTargetRequired != null) && (SMLValidatorUtil.isTrue(currentElementTargetRequired) != embeddedTargetElementNode.isRequired())) {
				validationLogger.reportMessage(ValidationMessageFactory.createErrorMessage(-1, NLS.bind(errorMessage, typeDefinition.getName())));
				return false;
			}
		}

		return true;
	}


	/**
	 * Answer the value of an attribute for a given declaration.
	 *
	 * @param declaration
	 * @param namespaceUri
	 * @param attributeName
	 * @return
	 */
	protected String retrieveAnnotationValue(XSElementDeclaration declaration, String namespaceUri, String attributeName) {
		String result = null;
		AnnotationResult annotation = SMLValidatorUtil.retrieveAnnotation(declaration, namespaceUri, attributeName);
		if ((annotation != null) && (annotation.getNodes().length > 0)) {
			return annotation.getNodes()[0].getNodeValue();
		}
		return result;
	}

	private boolean checkForSameHierarchy(XSComplexTypeDefinition parentComplexTypeForCurrentElement, XSComplexTypeDefinition enclosingType) {
		return ((parentComplexTypeForCurrentElement.derivedFromType(enclosingType, XSConstants.DERIVATION_EXTENSION)) || (enclosingType.derivedFromType(parentComplexTypeForCurrentElement, XSConstants.DERIVATION_EXTENSION)));
	}

	private void checkForBadDeclaration(AnnotationResult annotationResult) {
		if (annotationResult != null) {
			getValidationOutput().reportMessage(ValidationMessageFactory.createWarningMessage(
					ValidationMessage.NO_LINE_NUMBER, SMLValidationMessages.targetBadDeclaration));
		}
	}

	protected boolean compareElementInstance(String elem1, String elem2)
	{
		return (ParserHelper.removeNameSpace(elem1).equals(ParserHelper.removeNameSpace(elem2)));
	}

	private boolean checkForInvalidTargetElementValue(XSNamedMap elementDeclarations, XSElementDeclaration elementDeclaration, AnnotationResult result) throws DOMException {
		boolean status = true;
		if (result != null) {
			Node[] nodes = result.getNodes();
			if (nodes.length > 0) {
				String targetElementValue = nodes[0].getNodeValue();
				boolean found = false;
				for (int j = 0, declarationCount2 = elementDeclarations.getLength();
				 j < declarationCount2 && (!found); j++)
				{
					XSElementDeclaration elemDec2 = (XSElementDeclaration) elementDeclarations.item(j);
					if (elemDec2 != elementDeclaration) {
						found = compareElementInstance(elemDec2.getName(), targetElementValue);
					}
				}
				if (!found) {
					getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(
							ValidationMessage.NO_LINE_NUMBER, NLS.bind(SMLValidationMessages.targetElementInvalidValue, targetElementValue)));
					status = false;
				}
			}
		}
		return status;
	}

	/**
	 * Make sure that the uri can be resolved.
	 * @param uri the uri string to resolve
	 * @return true if the uri can be resolved otherwise false.
	 */
	protected static boolean isURISMLDeferenceable(TargetElementNode targetElementNode)
	{
		try
		{
			String uri = targetElementNode.getTargetURI();
			if ((uri == null) || "".equals(uri.trim())) {
				return false;
			}
			Node context = SMLValidatorUtil.locateDocumentNode(targetElementNode.getAliases(), targetElementNode.getDocumentPosition());
			return (((DerefXPathFunction)DerefXPathFunction.instance()).evaluate(context, SMLValidatorUtil.removeLineBreaks(uri), targetElementNode.getBaseURIValue()) != null);
		}
		catch (XPathFunctionException e)
		{
			return false;
		}
	}
}
