/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.util;

/**
 * Utility class for Base64 encoding
 */
public class Base64Util {

	private final static byte[] decodeMap = 
	{
		-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 
		-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 
		-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 
		-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 
		-1, -1, -1, 62, -1, -1, -1, 63, 52, 53, 
		54, 55, 56, 57, 58, 59, 60, 61, -1, -1, 
		-1, -1, -1, -1, -1,  0,  1,  2,  3,  4, 
		 5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 
		15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 
		25, -1, -1, -1, -1, -1, -1, 26, 27, 28, 
		29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 
		39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 
		49, 50, 51, -1, -1, -1, -1, -1
	};
	
	// This class should not be instantiated. 
	private Base64Util() {
	}
	
	/**
	 * Remove all whitespaces in a string. 
	 * @param input any string
	 * @return a string free of whitespaces
	 */
	public static String removeWhiteSpaces(String input) {
		return input.replaceAll("\\s", "");
	}
	
	/** 
	 * Decode a base64-encoded byte array.  
	 * @param input a base64-encoded byte array
	 * @return decoded byte array
	 */
	public static byte[] decode(byte[] input) throws Exception {
		int iLen = input.length;
		
		// length of input byte array must be a multiple of 4
		if (iLen % 4 != 0) {
			throw new Exception("Length of Base64 encoded input string is not a multiple of 4.");
		}
		
		// strip trailing '=' signs (padding)
		while (iLen > 0 && input[iLen - 1] == '=') {
			iLen--;
		}
		
		// length of output
		int oLen = (iLen * 3) / 4;

		byte[] out = new byte[oLen];
		int ip = 0;
		int op = 0;
		while (ip < iLen) {
			byte a0 = input[ip++];
			byte a1 = input[ip++];
			byte a2 = ip < iLen ? input[ip++] : (byte) 'A';
			byte a3 = ip < iLen ? input[ip++] : (byte) 'A';
			// First bit must not be 1.
			// This check is to make sure the index into the decodeMap is valid. 
			if (((a0 | a1 | a2 | a3) & 0x80) == 1) {
				throw new IllegalArgumentException("Illegal character in Base64 encoded data.");
			}

			int b0 = decodeMap[a0];
			int b1 = decodeMap[a1];
			int b2 = decodeMap[a2];
			int b3 = decodeMap[a3];
			if (b0 == -1 || b1 == -1 || b2 == -1 || b3 == -1) {
				throw new IllegalArgumentException("Illegal character in Base64 encoded data.");
			}

			int c0 = (b0 << 2) | (b1 >>> 4);
			int c1 = ((b1 & 0xf) << 4) | (b2 >>> 2);
			int c2 = ((b2 & 3) << 6) | b3;
			
			out[op++] = (byte) c0;
			if (op < oLen) {
				out[op++] = (byte) c1;
			}
			if (op < oLen) {
				out[op++] = (byte) c2;
			}
		}
		return out;
	}
}
 