/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit;

import java.util.Collection;
import java.util.Properties;

import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Describable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;

/**
 * InstallableUnit and ConfigurationUnit content elements can include zero or more ResultingChange elements that describe 
 * the key resources whose configuration is modified when the content elements artifacts are processed. ResultingChange 
 * elements refer to resources in Topology and define characteristics of those resources that will become true when the 
 * content element is applied.
 */
public interface ResultingChange extends Describable {

    /**
     * Gets the name.
     * 
     * The Name corresponds with the name of the changed resource as known in the deployment environment. Name SHOULD be 
     * defined in Topology and not in ResultingChange, because the name is not changed by processing the content elements 
     * artifacts. If Name is defined in both places, the values MUST match.
     *  
     * @return Name of the resulting resource as known in the deployment environment
     */
    String getName();
    
	/**
	 * Gets the properties.
	 * 
	 * Property elements MAY be included to identify property values of the identified resource as they will exist after 
	 * applying the content element.<br><br>
	 * Properties defined in ResultingChange MUST be properties that are modified by processing the content elements artifacts.
	 * 
	 * @return A resulting property setting of the changing resource.
	 */
	Properties getProperties();
	
	/**
	 * Gets the condition.
	 * 
	 * A Condition is used when the resulting change will be performed by applying the content element only when certain 
	 * conditions exist in the deployment environment.
	 *  
	 * @return A condition that determines if the resulting change definition is relevant to a particular deployment.
	 */
	Condition getCondition();
	
	/**
	 * Gets teh relationships.
	 * 
	 * When application of the content element results in the creation or modification of relationships, the Relationship 
	 * elements SHOULD be included to identify relationships as they will exist after application of the content element. 
	 * 
	 * @return A collection of relationships with another resource that will result from this deployment.
	 */
	Collection<Relationship> getRelationships();
	
	/**
	 * Sets the name.
	 * 
	 * The Name corresponds with the name of the changed resource as known in the deployment environment. Name SHOULD be 
     * defined in Topology and not in ResultingChange, because the name is not changed by processing the content elements 
     * artifacts. If Name is defined in both places, the values MUST match.
     * 
	 * @param name Name of the resulting resource as known in the deployment environment
	 */
	void setName(String name);
	
	/**
	 * Sets the properties.
	 * 
	 * Property elements MAY be included to identify property values of the identified resource as they will exist after 
	 * applying the content element.<br><br>
	 * Properties defined in ResultingChange MUST be properties that are modified by processing the content elements artifacts.
	 * 
	 * @param newResultingChangeProperties A resulting property setting of the changing resource.
	 */
	void setProperties(Properties newResultingChangeProperties);
	
	/**
	 * Sets the condition.
	 * 
	 * A Condition is used when the resulting change will be performed by applying the content element only when certain 
	 * conditions exist in the deployment environment.
	 * 
	 * @param condition A condition that determines if the resulting change definition is relevant to a particular deployment.
	 */
	void setCondition(Condition condition);
	
	/**
	 * Sets the relationships.
	 * 
	 * When application of the content element results in the creation or modification of relationships, the Relationship 
	 * elements SHOULD be included to identify relationships as they will exist after application of the content element. 
	 * 
	 * @param relationships A collection of relationships with another resource that will result from this deployment.
	 */
	void setRelationships(Collection<Relationship> relationships);
	
	/**
	 * Creates a copy of this object.
	 * 
	 * @return A copy of this object. 
	 */
	public Object clone();
}
