/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.cosmos.dc.broker.internal.persistence;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.apache.axiom.om.OMElement;
import org.apache.axiom.om.impl.builder.StAXOMBuilder;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.Group;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.Groups;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.Service;
import org.eclipse.cosmos.dc.broker.provisional.datamodel.Services;
import org.eclipse.cosmos.dc.broker.provisional.persistence.BrokerDatabase;
import org.eclipse.cosmos.dc.broker.provisional.persistence.BrokerDatabaseException;

public class BrokerDatabaseImpl implements BrokerDatabase {
	
	private static File databaseDir = null;
	private static File databaseFile = null;
	private static Groups groups = null;
	private static Object writeLock = new Object();

	private Logger logger = null;
	private String ResourceBundleName = "org.eclipse.cosmos.dc.broker.internal.persistence.brokerMessages"; 

	private static BrokerDatabaseImpl instance = new BrokerDatabaseImpl(); 
	
	public static BrokerDatabaseImpl getInstance() {
		return instance;
	}

	private BrokerDatabaseImpl() {
		// init logger
		logger = Logger.getLogger(getClass());
		logger.setResourceBundle(ResourceBundle.getBundle(ResourceBundleName));
	}
	
	public void init(Map<String, Object> initMap) {
		String dbdir = initMap.get("dbdir").toString();
		setDatabaseDir(dbdir);
	}
	
	private void setDatabaseDir(String dbdirpath) {
		File dbdir = new File(dbdirpath);
		if (dbdir.exists() == false) {
			dbdir.mkdir();
			logger.l7dlog(Level.INFO, "CREATED_DB_INFO", new Object[] {dbdirpath}, null);
		}
		databaseDir = dbdir;
	}
	
	public Groups getAllGroups() throws BrokerDatabaseException {
		if (groups == null) {
			loadxml();
		}
		return groups;
	}
	
	/**
	 * Get all groups that include a service with the given EPR.
	 */
	public Groups getGroupsByNamespace(String ns) throws BrokerDatabaseException {
		if (groups == null) {
			loadxml();
		}
		
		Groups result = new Groups();
		
		for (Group group : groups.getGroups()) {
			Services services = group.getServices();
			if (services != null) {
				List<Service> serviceList = services.getServices();
				if (serviceList != null) {
					for (Service service : serviceList) {
						if (service.getNamespace().equals(ns)){
							result.addGroup(group);
							break;
						}
					}
				}
			}
		}
		return result;
	}
	
	public Group getGroup(String hostname, String groupName) throws BrokerDatabaseException {
		if (groups == null) {
			loadxml();
		}
		for (Group group : groups.getGroups()) {
			if (group.getHost().equals(hostname) && 
					group.getName().equals(groupName)) {
				return group;
			}
		}
		return null;
	}
	
	public Group getGroupById(String id) throws BrokerDatabaseException {
		if (groups == null) {
			loadxml();
		}
		for (Group group : groups.getGroups()) {
			String groupId = group.getId();
			if ( groupId != null && groupId.equals(id) ) {
				return group;
			}
		}
		return null;
	}
	
	public Groups getGroupsByRecordTypeNamespace(String recordTypeNamespace) throws BrokerDatabaseException {
		if (groups == null) {
			loadxml();
		}
		Groups result = new Groups();
		
		for (Group group : groups.getGroups()) {
			String groupRecordTypeNamespace = group.getRecordTypeNamespace();
			if ( groupRecordTypeNamespace != null && groupRecordTypeNamespace.equals(recordTypeNamespace) ) {
				result.addGroup(group);
			}
		}
		return result;
	}
	
	public boolean addGroup(Group newgroup) throws BrokerDatabaseException {
		boolean dup = false;
		
		if (groups == null) {
			loadxml();
		}
		synchronized(writeLock) {
			// Check if a group with the same ID already exist.
			// if so, return false
			String host = newgroup.getHost();
			String name = newgroup.getName();
			List<Group> groupList = groups.getGroups();
			for (Group group : groupList) {
				if (group.getHost().equals(host) && 
						group.getName().equals(name)) {
					dup = true;
					break;
				}
			}
			if (dup == false) {
				groupList.add(newgroup);
				save();
				return true;
			} else {
				return false;
			}
		}
	}
	
	public boolean deleteGroup(String hostname, String groupName) throws BrokerDatabaseException {
		boolean groupDeleted = false;
		if (groups == null) {
			loadxml();
		}
		// find group with the specified ID and delete the group
		synchronized (writeLock) {
			List<Group> groupList = groups.getGroups();
			for (Group group : groupList) {
				if (group.getHost().equals(hostname) && 
						group.getName().equals(groupName)) {
					groupDeleted = groupList.remove(group);
					break;
				}
			}
			save();
		}
		return groupDeleted;
	}
	
	public boolean deleteGroupById(String id) throws BrokerDatabaseException {
		boolean groupDeleted = false;
		if (groups == null) {
			loadxml();
		}
		// find group with the specified ID and delete the group
		synchronized (writeLock) {
			List<Group> groupList = groups.getGroups();
			for (Group group : groupList) {
				String groupId = group.getId();
				if ( groupId != null && groupId.equals(id) ) {
					groupDeleted = groupList.remove(group);
					break;
				}
			}
			save();
		}
		return groupDeleted;
	}
	
	private void loadxml() throws BrokerDatabaseException {
		if (databaseDir == null) {
			throw new BrokerDatabaseException("Database directory not set");
		}
		
		if (databaseFile == null) {
			databaseFile = new File(databaseDir, "brokerdb.xml");
			if (databaseFile.exists() == false) {
				// Create database XML if it does not exist in database directory
				try {
					databaseFile.createNewFile();
				} catch (IOException e) {
					logger.l7dlog(Level.ERROR, "BROKER_DB_CREATION_ERROR", new Object[] {databaseDir}, e);
					throw new BrokerDatabaseException("Failed to create broker database at " + databaseDir);
				}
				groups = new Groups();
				save();
				return;
			}
		}

		try {
	    	InputStream stream = new FileInputStream(databaseFile);
	    	//create the parser
	    	XMLStreamReader parser = XMLInputFactory.newInstance().createXMLStreamReader(stream);
	    	//create the builder
	    	StAXOMBuilder builder = new StAXOMBuilder(parser);
	
	    	//get the root element (in this case the envelope)
	    	OMElement docRoot = builder.getDocumentElement();
	    	groups = Groups.getGroups(docRoot);
		} catch (Exception e) {
			logger.l7dlog(Level.ERROR, "LOAD_BROKER_DB_ERROR", e);
			throw new BrokerDatabaseException("Failed to load broker database.", e);
		}
	}
	
	private synchronized void save() throws BrokerDatabaseException {
		try {
			FileWriter fw = new FileWriter(databaseFile);
			
			OMElement om = groups.toOM();
			XMLStreamWriter writer = XMLOutputFactory.newInstance().createXMLStreamWriter(fw);
			om.serialize(writer);
			writer.flush();
			writer.close();
		} catch (Exception e) {
			logger.l7dlog(Level.ERROR, "SAVE_BROKER_DB_ERROR", e);
			throw new BrokerDatabaseException("Failed to save broker database xml. ", e);
		}
	}
	
}
