/*******************************************************************************
 * Copyright (c) 2009 SAS Institute, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute - initial API and implementation
 *     CA, Inc. - Bug 268158
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.o10r.impl;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.eclipse.cosmos.me.sdd.cr.ResolverByQname;
import org.eclipse.cosmos.me.sdd.schema.AdditionalContentType;
import org.eclipse.cosmos.me.sdd.schema.AlternativeRequirementType;
import org.eclipse.cosmos.me.sdd.schema.ArgumentType;
import org.eclipse.cosmos.me.sdd.schema.BooleanParameterType;
import org.eclipse.cosmos.me.sdd.schema.ConditionType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalDerivedVariableExpressionType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ContentType;
import org.eclipse.cosmos.me.sdd.schema.DerivedVariableType;
import org.eclipse.cosmos.me.sdd.schema.InstallableUnitType;
import org.eclipse.cosmos.me.sdd.schema.IntegerParameterType;
import org.eclipse.cosmos.me.sdd.schema.ParametersType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.RequirementType;
import org.eclipse.cosmos.me.sdd.schema.RequirementsType;
import org.eclipse.cosmos.me.sdd.schema.ResourcePropertyType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.ResultingResourceType;
import org.eclipse.cosmos.me.sdd.schema.SubstitutionType;
import org.eclipse.cosmos.me.sdd.schema.URIParameterType;
import org.eclipse.cosmos.me.sdd.schema.VariablesType;
import org.eclipse.cosmos.me.sdd.schema.ext.ResolutionObject;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;

/**
 * Class to walk the InstallableUnit definition.
 * 
 * @author jehamm
 *
 */
public class InstallableUnitWalker {
	SDDContext _ctxt;
	ResourceConstraintResolver _resourceResolver;
	
	/**
	 * Constructor
	 * @param installableUnit
	 * @param contentsType 
	 */
	public InstallableUnitWalker(SDDContext ctxt, ResolverByQname resolver) {
		_ctxt = ctxt;
		_resourceResolver = new ResourceConstraintResolver(resolver);
	}
	
	/**
	 * Walk the different types that could be here.
	 */
	public void walk(ResourceType resource) {
		
		// Walk the variables first.
		InstallableUnitType iu = _ctxt.getDeploymentDescriptor().getInstallableUnit();
		
		// Iterate over the list of resources.
		VariablesType variables = iu.getVariables();
		if (variables != null) {
			for (Object listObject : variables.getParametersOrResourcePropertyOrDerivedVariable()) {
				walkVariable(listObject);
			}
		}
		
		// Iterate over the list of resources.
		RequirementsType requirements = iu.getRequirements();
		if (requirements != null) {
			for (RequirementType requirement : requirements.getRequirement()) {
				walkRequirement(requirement);	
			}
		}
			
		//
		// Resulting Resources
		//
		for (ResultingResourceType resultingResource : iu.getResultingResource()) {
			walkResultingResource(resultingResource);
		}
		
		if(iu.getArtifacts().getInstallArtifact() != null ) {
			walkInstallableUnit(iu);
		}
	}

	/**
	 * Walk the variables. We want to load these variables.
	 * 
	 * @param variableTypeObject
	 */
	private void walkVariable(Object type) {
		
		// ignore arbitrary xml
		try {
			((JAXBElement<?>)type).getValue();
		} catch (ClassCastException e) {
			return;
		}
		
		Object listObject = ((JAXBElement<?>)type).getValue();
		if(listObject instanceof DerivedVariableType) {
			
			DerivedVariableType derivedType = (DerivedVariableType)listObject;
			
			List<ResolutionObject> resolutionList = new ArrayList<ResolutionObject>();
			
			for (ConditionalDerivedVariableExpressionType conditionalType : derivedType.getConditionalExpression()) {
				ResolutionObject resolution = new ResolutionObject();
				ConditionType condition = conditionalType.getCondition();
				if(condition != null) {
					
					// The expression is the the value that the derived variable should resolve
					// to as a value if this condition is met.
					String expression = conditionalType.getExpression();
					
					// Set the value of resolution to the expression of this conditional
					resolution.setValue(expression);
					List<ConditionalResourceConstraintType> conditionalConstraintList = 
						condition.getResourceConstraint();
				
					for (ConditionalResourceConstraintType c : conditionalConstraintList) {
						//
						// TODO -- Check the model FIRST before re-running the resolution.
						//
						resolutionList.add(_resourceResolver.resolve(c, resolution));
					}
				}
			}
	
			// So at this time we should have resolved our conditional variable.		
			// Examine the model and determine which constraint was resolved or which was not.
			for (ResolutionObject r : resolutionList) {
				if(r.isResolved()) {
					r.setId(derivedType.getId());
					_ctxt.getResolutionModel().addResolutionObject(derivedType.getId(), r);
				}
			}
		}
		else if(listObject instanceof ResourcePropertyType) {
			ResourcePropertyType resourceType =(ResourcePropertyType)listObject;
			System.out.println("ResourcePropertyType for variables are not implemented...." + resourceType.getId());
		}
		else if(listObject instanceof ParametersType) {
			ParametersType parametersType = (ParametersType)listObject;
			// Get the type of parameter we are handling.
			List<Object> parametersList = parametersType.getIntegerParameterOrStringParameterOrBooleanParameter();
			for (Object param : parametersList) {
				ResolutionObject resolution = new ResolutionObject();
				
				Object parameterObject = ((JAXBElement<?>)param).getValue();
				if(parameterObject instanceof URIParameterType) {
					URIParameterType uiParameter = (URIParameterType)parameterObject;
					resolution.setId(uiParameter.getId());
					resolution.setValue(uiParameter.getDefaultValue());
				}
				else if(parameterObject instanceof IntegerParameterType) {
					IntegerParameterType integerParameter = (IntegerParameterType)parameterObject;
					resolution.setId(integerParameter.getId());
					resolution.setValue(integerParameter.getDefaultValue());
				}
				else if(parameterObject instanceof BooleanParameterType) {
					BooleanParameterType booleanParameter = (BooleanParameterType)parameterObject;
					resolution.setId(booleanParameter.getId());
					resolution.setValue(booleanParameter.getDefaultValue());
				}
				else {
					// Unable to process because we don't have this type defined.
					System.out.println("walkVariables() - Don't know this parameter type" + parameterObject.toString());
				}
				
				
				//
				// TODO -- Check the model FIRST before re-running the resolution.
				//
				
				
				// OK -- so we probably want a property resolver to be able to handle this but I am 
				// coding it in this place now in order to have a a handle to any thing that has been 
				// resolved in the model above.
				if(resolution.getValue() != null) {
					String resolutionValue = resolution.getValue();
						
					// See if this value has been resolved in our model.
					String envValue = resolveEnvironmentVariable(resolutionValue);
					ResolutionObject resolved = _ctxt.getResolutionModel().getResolutionObject(envValue);
					
					String resolvedParameterValue;
					if(resolved != null) {
						resolvedParameterValue = resolved.getValue();
						resolutionValue = resolutionValue.replace("$(" + envValue + ")", resolvedParameterValue);
						resolution.setValue(resolutionValue);
						resolution.resolve();
					}
				}			
				
				// Add this parameter to our model.
				_ctxt.getResolutionModel().addResolutionObject(resolution.getId(), resolution);
			}
		}
		else {
			//Invalid variable type.
			System.out.println("walkVariables() - Don't know this variable type" + listObject.toString());
		}
	}

	/**
	 * Walk the requirement.
	 * @param requirement
	 */
	private void walkRequirement(RequirementType requirement) {
		
		// Get the list of alternatives.
		for (AlternativeRequirementType alternative : requirement.getAlternative()) {
			// Now that we have each alternative, we need to solve each.
			for (RequirementResourceConstraintType resourceConstraint : alternative.getResourceConstraint()) {
				ResolutionObject resolution = new ResolutionObject();
				resolution.setId(resourceConstraint.getId());
				
				ResourceType type = (ResourceType) resourceConstraint.getResourceRef();				
				resolution.setReferenceId(type.getId());
				resolution.setValue(type.getName());
				
				String resourceName = type.getName();
				if(resourceName != null) {
					String value = resolveEnvironmentVariable(resourceName);
					ResolutionObject resolved = _ctxt.getResolutionModel().getResolutionObject(value);
					
					String resolvedParameterValue;
					if(resolved != null) {
						resolvedParameterValue = resolved.getValue();
						value = resourceName.replace("$(" + value + ")", resolvedParameterValue);
						resolution.setValue(value);
					}
				}
				
				//
				//TODO - Check the model FIRST before re-running the resolution.
				//
				
				//
				// TODO - This is broke!!!  We actually need  a way of looking at the returns to make
				//        sure which alternative was resolved. If we had one or more then good; otherwise bad.!
				//
	
				_ctxt.getResolutionModel().addResolutionObject(resourceConstraint.getId(),
						_resourceResolver.resolve(resourceConstraint, resolution));
			}
		}
		
		// Handle resource constraints on the requirement.
		for (RequirementResourceConstraintType resourceConstraint : requirement.getResourceConstraint()) {
			ResolutionObject resolution = new ResolutionObject();
			resolution.setId(resourceConstraint.getId());
			
			ResourceType type = (ResourceType) resourceConstraint.getResourceRef();				
			resolution.setReferenceId(type.getId());
			
			// Note -- this means that you cannot have multiple substitutions on a variable.
			String resourceName = type.getName();
			if(resourceName != null) {
				String value = resolveEnvironmentVariable(resourceName);
				ResolutionObject resolved = _ctxt.getResolutionModel().getResolutionObject(value);
				
				String resolvedParameterValue;
				if(resolved != null) {
					resolvedParameterValue = resolved.getValue();
					value = resourceName.replace("$(" + value + ")", resolvedParameterValue);
					resolution.setValue(value);
				}
			}
			
			//
			//TODO - Check the model FIRST before re-running the resolution.
			//
			
			_ctxt.getResolutionModel().addResolutionObject(resourceConstraint.getId(),
					_resourceResolver.resolve(resourceConstraint, resolution));
		}
	}

	private void walkResultingResource(ResultingResourceType resultingResource) {
		// walk the resource through.
		ResolutionObject obj = new ResolutionObject();
		obj.setReferenceId(((ResourceType)resultingResource.getResourceRef()).getId());
		obj.setId(resultingResource.getName());
		obj.setValue(resultingResource.getVersion());
		obj.resolve();
		
		// Add this resolution to our model.
		_ctxt.getResolutionModel().addResultingResource(obj.getReferenceId(), obj);
	}

	private void walkInstallableUnit(InstallableUnitType iu) {
		int i = 0;
		for (ArgumentType argument : iu.getArtifacts().getInstallArtifact().getArguments().getArgument()) {
			ResolutionObject obj = new ResolutionObject();

			obj.setId(argument.getName());
			String resolutionValue = argument.getValue();
			
			// See if this value has been resolved in our model.
			String envValue = resolveEnvironmentVariable(resolutionValue);
			ResolutionObject resolved = _ctxt.getResolutionModel().getResolutionObject(envValue);
			
			String resolvedParameterValue;
			if(resolved != null) {
				resolvedParameterValue = resolved.getValue();
				resolutionValue = resolutionValue.replace("$(" + envValue + ")", resolvedParameterValue);
				obj.setValue(resolutionValue);
				obj.resolve();
			}
			
			_ctxt.getResolutionModel().addArtifact("argument_" + i++, obj);
		}
		
		for (AdditionalContentType additionalContent : iu.getArtifacts().getInstallArtifact().getAdditionalContent()) {
			String contentReference = additionalContent.getContentRef();
			String contentName = null;
			
		    List<ContentType> packageFileArray = _ctxt.getPackageDescriptor().getContents().getContent();
		    for (ContentType packageContent : packageFileArray) {
		        if (contentReference.equalsIgnoreCase(packageContent.getId())) {
		        	contentName = packageContent.getPathname();
		        }
		    }
		    
		    // If this is null then it means that something was referenced from the 
		    // SDD that was not defined in the package descriptor.
		    if(contentName == null) {
		    	// This is an error and should be reported as such.
		    }
		    
		    ResolutionObject contentResolution = new ResolutionObject();
		    contentResolution.setReferenceId(contentName);
		    
		    // Look at our substitution data.
			List<SubstitutionType> substitutionList = additionalContent.getSubstitution();
			int v = 0;
			for (SubstitutionType substitution : substitutionList) {
				
				// need a resolution object for this substitution.
				ResolutionObject obj = new ResolutionObject();
				obj.setReferenceId(contentName);
				
				// Get the substitution object and the data.
				obj.setId(substitution.getPattern());
				
				String resolutionValue = substitution.getValue();
				
				// See if this value has been resolved in our model.
				String envValue = resolveEnvironmentVariable(resolutionValue);
				ResolutionObject resolved = _ctxt.getResolutionModel().getResolutionObject(envValue);
				
				String resolvedParameterValue;
				if(resolved != null) {
					resolvedParameterValue = resolved.getValue();
					resolutionValue = resolutionValue.replace("$(" + envValue + ")", resolvedParameterValue);
					obj.setValue(resolutionValue);
					obj.resolve();
				}
				
				_ctxt.getResolutionModel().addArtifact("substitution_" + v++, obj);
				
				// Open the uri indicated by the contentPath and perform the substitution
				String parentDir = _ctxt.getPackageDescriptorParentUri();
				File f = new File(parentDir + File.separator + "additionalcontent" + File.separator + contentName );
				if(f.exists()) {
					BufferedReader reader = null;
					BufferedWriter writer = null;
					try {
						reader = new BufferedReader(new FileReader(f));
						
						StringBuffer buffer = new StringBuffer();
						
						// Open and read the file.
						String line;							
						while ((line = reader.readLine()) != null) {
					        buffer.append(line.replace(obj.getId(), obj.getValue())).append("\n");
					    } 
						
						reader.close();
						
						writer = new BufferedWriter(new FileWriter(f));
						//Now write our augmented data out.
						writer.write(buffer.toString());
						
					} catch (FileNotFoundException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					} catch (IOException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
					finally {
						try {
							writer.close();
						} catch (IOException e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
						}
					}
				}
			}
		}
	}

	private String resolveEnvironmentVariable(String value) {
		String envValue = value;
		
		if(envValue.contains("$")) {
			// Strip out and then form value substitute if we can.
			envValue = value.substring(value.indexOf("$"), value.indexOf(")") + 1);
			
			envValue = envValue.replace("$","");
			envValue = envValue.replace("(", "");
			envValue = envValue.replace(")", "");
		}
		return envValue;
	}
}
