/*******************************************************************************
 * Copyright (c) 2009 SAS Institute, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.o10r.impl;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBElement;

import org.eclipse.cosmos.me.sdd.cr.Constraint;
import org.eclipse.cosmos.me.sdd.cr.ResolverByQname;
import org.eclipse.cosmos.me.sdd.schema.CapacityConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalPropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConsumptionConstraintValueType;
import org.eclipse.cosmos.me.sdd.schema.MaxVersionType;
import org.eclipse.cosmos.me.sdd.schema.PropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.PropertyValueListType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintType;
import org.eclipse.cosmos.me.sdd.schema.VersionConstraintValueType;
import org.eclipse.cosmos.me.sdd.schema.VersionRangeType;
import org.eclipse.cosmos.me.sdd.schema.VersionValueType;
import org.eclipse.cosmos.me.sdd.schema.ext.ResolutionObject;

/**
 * This class is responsible for resolving any resource constraint found. It will attempt to resolve
 * any and all properties, versions, capacities, and consumption defined for a particular resource and return 
 * a resolution object for each.
 * 
 * @author jehamm
 *
 */
public class ResourceConstraintResolver {
	private ResolverByQname _resolver;
	
	public ResourceConstraintResolver(ResolverByQname r) {
		_resolver = r;
	}
	
	/**
	 * Attempt to resolve this resource constraint.
	 * @param constraint
	 * @param resolution 
	 */
	public ResolutionObject resolve(RequirementResourceConstraintType constraint, ResolutionObject resolution) {
		List<Constraint> constraints = new ArrayList<Constraint>();
		
		// Loop over the types that we may defined here.
		for(Object type : constraint.getResourceConstraintGroup()) {
			
			// Create an object to contain our data for each pass of the resolvers.
			Constraint c = new Constraint();
			
			Object constraintType = ((JAXBElement<?>)type).getValue();
			
			// If this is a version constraint.
			if(constraintType instanceof VersionConstraintType) {
				VersionConstraintType versionConstraint = (VersionConstraintType)constraintType;
				
				c.setType("version");
				
				// Get the list of supported versions.
				VersionConstraintValueType supported = versionConstraint.getSupported();
				if(supported != null) {
					// Supported versions have ranges.
					for (VersionRangeType range : supported.getRange()) {
						if(range != null) {
							String minimumValue = range.getMinVersion();
							c.setValue("minimum", minimumValue);
						}
						
						// Get maximum value if specified.
						MaxVersionType maxType = range.getMaxVersion();
						if(maxType != null) {
							String maximumValue = maxType.getValue();
							c.setValue("maximum", maximumValue);
						}
					}
					
					// Supported versions have versions.
					for (VersionValueType version : supported.getValue()) {
						c.setValue("versionvalue", version.getVersion());
					}
				}
			}
			else if(constraintType instanceof CapacityConstraintType) {
				// If this is a capacity constraint.
				c.setType("capacity");
			}
			else if(constraintType instanceof PropertyConstraintType) {
				PropertyConstraintType propertyConstraint = (PropertyConstraintType)constraintType;
				c.setType("property");
				
				String name = propertyConstraint.getPropertyName().getLocalPart();
								
				// See if we have a list of values.
				PropertyValueListType propertyValues = propertyConstraint.getListOfValues();
				if(propertyValues != null) {
					List<JAXBElement<String>> valuesList = propertyValues.getValue();
					if(valuesList != null) {
						String [] values = (String[]) valuesList.toArray();	
						c.setValues(name, values);
					}
				}
				String value = propertyConstraint.getValue();
				c.setValue(name, value);
			}
			else if(constraintType instanceof ConsumptionConstraintType) {
				ConsumptionConstraintType consumptionConstraint = (ConsumptionConstraintType)constraintType;
				c.setType("consumption");
				
				String consumptionName = consumptionConstraint.getPropertyName().getLocalPart();
				
				ConsumptionConstraintValueType consumptionValue = consumptionConstraint.getValue();
				if(consumptionValue != null) {
					c.setValue(consumptionName, consumptionValue.getValue());
				}
			}
			else {
				// Unknown type we cannot process.
				System.out.println("Tell us what you are '" + constraintType.toString() + "'");
			}
			
				
			// Make sure we found something before continuing....
			if(c.getType() != null) {
				// Set the resource value in case we need it.
				c.setResource(resolution.getValue());
				_resolver.resolve(getResource(constraint).getType(), c);
				constraints.add(c);
			}
			else {
				System.out.println("Unable to find the type of constraint we were looking for...");
			}
		}
		
		// So after this loop has run, we should have a collection of Constraints that define the
		// whether the resource is accepted or rejected.
		boolean resourceResolved = true;
		for(Constraint c : constraints) {
			if(c.isRejected()) {
				resourceResolved = false;
			}
		}	
		
		// Set out resource status based on the outcome of all the constraints against this resource.
		if(resourceResolved) {
			resolution.resolve();
		}
		else {
			resolution.reject();
		}
		
		return resolution;
	}
	
	/**
	 * Attempt to resolve this resource constraint.
	 * @param requirementResourceConstraintType
	 */
	public ResolutionObject resolve(ConditionalResourceConstraintType constraint, ResolutionObject resolution) {
		List<Constraint> constraints = new ArrayList<Constraint>();
		
		for(ConditionalPropertyConstraintType propertyConstraint : constraint.getPropertyConstraint()) {
			// Create an object to contain our data.
			Constraint c = new Constraint();
			
			c.setType("property");
			
			String name = propertyConstraint.getPropertyName().getLocalPart();
							
			// See if we have a list of values.
			PropertyValueListType propertyValues = propertyConstraint.getListOfValues();
			if(propertyValues != null) {
				List<JAXBElement<String>> valuesList = propertyValues.getValue();
				if (!valuesList.isEmpty()) {
					List<String> l = new ArrayList<String>();
					for (JAXBElement<String> j : valuesList) {
						l.add(j.getValue());
					}
					c.setValues(name, l.toArray(new String[]{}));
				}
			}
			
			// See if there was just a value specified.
			String value = propertyConstraint.getValue();
			if(value != null) {
				c.setValue(name, value);
			}
			
			// Make sure we found something before continuing....
			if(c.getType() != null) {
				_resolver.resolve(getResource(constraint).getType(), c);
				constraints.add(c);
			}
			else {
				System.out.println("Unable to find the type of constraint we were looking for...");
			}
		}
		
		// So after this loop has run, we should have a collection of Constraints that define the
		// whether the resource is accepted or rejected.
		boolean resourceResolved = true;
		for(Constraint c : constraints) {
			if(c.isRejected()) {
				resourceResolved = false;
			}
		}	
		
		// Set out resource status based on the outcome of all the constraints against this resource.
		if(resourceResolved) {
			resolution.resolve();
		}
		else {
			resolution.reject();
		}
				
		return resolution;	
	}
	
	private ResourceType getResource(RequirementResourceConstraintType rrct) {
		return (ResourceType)rrct.getResourceRef();
	}

	private ResourceType getResource(ConditionalResourceConstraintType rrct) {
		return (ResourceType)rrct.getResourceRef();
	}
}
