/***********************************************************************
 * Copyright (c) 2009 CA, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     CA, Inc. - Initial implementation
 ***********************************************************************/
package org.eclipse.cosmos.me.sdd.o10r.impl;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.eclipse.cosmos.me.sdd.cr.ResourceHandler;
import org.eclipse.cosmos.me.sdd.schema.BooleanParameterType;
import org.eclipse.cosmos.me.sdd.schema.ConditionType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalDerivedVariableExpressionType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalPropertyConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ConditionalResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.DerivedVariableType;
import org.eclipse.cosmos.me.sdd.schema.InstallableUnitType;
import org.eclipse.cosmos.me.sdd.schema.IntegerParameterType;
import org.eclipse.cosmos.me.sdd.schema.ParametersType;
import org.eclipse.cosmos.me.sdd.schema.ResourcePropertyType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.StringParameterType;
import org.eclipse.cosmos.me.sdd.schema.URIParameterType;
import org.eclipse.cosmos.me.sdd.schema.VariablesType;
import org.eclipse.cosmos.me.sdd.schema.ext.SDDContext;

public class VariableResolver {
	private SDDContext ctxt;
	private ResourceHandler handler;
	
	public VariableResolver(SDDContext ctxt, ResourceHandler handler) {
		this.ctxt = ctxt;
		this.handler = handler;
	}
	
	public void installableUnitResolver() {
		InstallableUnitType iu = ctxt.getDeploymentDescriptor().getInstallableUnit();
		
		resolve(iu.getVariables());
	}
	
	private void resolve(VariablesType vt) {
		if (vt != null) {
			List<Object> variables = vt.getParametersOrResourcePropertyOrDerivedVariable();
			for (Object o : variables) {
				Object obj = ((JAXBElement<?>)o).getValue();
				if (obj instanceof DerivedVariableType) {
					resolve((DerivedVariableType)obj);
				}
				if (obj instanceof ParametersType) {
					resolve((ParametersType)obj);
				}
				if (obj instanceof ResourcePropertyType) {
					resolve((ResourcePropertyType)obj);
				}
			}
			vt.getParametersOrResourcePropertyOrDerivedVariable().add(ctxt.getVariables());
		}
	}
	
	private void resolve(ResourcePropertyType var) {
		System.out.println("... ResourcePropertyType ...");
	}

	private String resolve(DerivedVariableType var) {
		if (ctxt.containsVariable(var.getId())) {
			System.out.printf("%s::resolve: %s - (%s) already assigned\n",
					this.getClass().getName(), var.getId(), ctxt.getVariable(var.getId()));
		} else
		if (var.getExpression() != null) {
			ctxt.setVariable(var.getId(), ctxt.substituteVariables(var.getExpression()));
		} else
		if (!var.getConditionalExpression().isEmpty()) {
			String val = resolve(var.getConditionalExpression());
			if (val != null) {
				ctxt.setVariable(var.getId(), ctxt.substituteVariables(val));
			}
		}
		return ctxt.getVariable(var.getId());
	}

	private String resolve(List<ConditionalDerivedVariableExpressionType> list) {
		List<Pair> exprs = new ArrayList<Pair>();
		for (ConditionalDerivedVariableExpressionType exp : list) {
			Pair p = resolve(exp);
			if (p != null) exprs.add(p);
		}
		Pair p = getHighestPriority(exprs);
		if (p != null) {
			return p.expression;
		}
		return null;
	}
	
	private Pair resolve(ConditionalDerivedVariableExpressionType var) {
		if (var.getCondition() != null) {
			ConditionType cond = var.getCondition();
			for (ConditionalResourceConstraintType resCon : cond.getResourceConstraint()) {
				for (ConditionalPropertyConstraintType p : resCon.getPropertyConstraint()) {
					if (handler.evaluate(ctxt, (ResourceType)resCon.getResourceRef(), p)) {
						String value = (String)handler.getPropertyValue(ctxt, (ResourceType)resCon.getResourceRef(), p.getPropertyName());
						resCon.getOtherAttributes().put( p.getPropertyName(), value);
						cond.getOtherAttributes().put(new QName("satisfied"), "true");
						return new Pair(var.getPriority(), var.getExpression());
					}
				}
			}
		}
		return null;
	}

	private void resolve(ParametersType params) {
		for (Object o : params.getIntegerParameterOrStringParameterOrBooleanParameter()) {
			Object obj = ((JAXBElement<?>)o).getValue();
			if (obj instanceof BooleanParameterType) {
				resolve((BooleanParameterType)obj);
				return;
			}
			if (obj instanceof IntegerParameterType) {
				resolve((IntegerParameterType)obj);
				return;
			}
			if (obj instanceof StringParameterType) {
				resolve((StringParameterType)obj);
				return;
			}
			if (obj instanceof URIParameterType) {
				resolve((URIParameterType)obj);
				return;
			}
		}
	}

	private void resolve(URIParameterType var) {
		if (ctxt.containsVariable(var.getId())) {
			System.out.printf("%s::resolve(URIParameterType var): %s - (%s) already assigned\n",
					this.getClass().getSimpleName(), var.getId(), ctxt.getVariable(var.getId()));
			return;
		}
		if (var.getDefaultValue() != null) {
			ctxt.setParameter(var.getId(), (var.getDefaultValue() == null) ? var.isRequired() : false, var.getDefaultValue());
			return;
		}
	}

	private void resolve(StringParameterType var) {
		System.out.printf("%s::resolve(StringParameterType var)\n",
				this.getClass().getSimpleName());
	}

	private void resolve(IntegerParameterType var) {
		System.out.printf("%s::resolve(IntegerParameterType var)\n",
				this.getClass().getSimpleName());
	}

	private void resolve(BooleanParameterType var) {
		System.out.printf("%s::resolve(BooleanParameterType var)\n",
				this.getClass().getSimpleName());
	}

	private Pair getHighestPriority(List<Pair> list) {
		if (list.isEmpty()) return null;
		BigInteger i = BigInteger.ZERO;
		for (Pair p : list) {
			BigInteger j = p.priority;
			i = i.max(j);
		}
		for (Pair p : list) {
			if (p.priority.equals(i)) {
				return p;
			}
		}
		return null;
	}

	private class Pair {
		public BigInteger priority;
		public String expression; 
		
		public Pair(BigInteger priority, String expression) {
			this.priority = priority;
			this.expression = expression;
		}
	}
}
