/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.provisional.repository.core;

import java.io.InputStream;
import java.util.Map;

import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryConnectionException;
import org.eclipse.cosmos.rm.provisional.repository.exception.RepositoryOperationException;
import org.eclipse.cosmos.rm.provisional.repository.operations.ISMLOperation;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLDocument;
import org.eclipse.cosmos.rm.provisional.repository.resource.ISMLMetadata;

/**
 * Represents a repository that can be used to connect to and fetch 
 * stored documents from.  This interface provides an abstraction from
 * the type of repository used and the query method used to retrieve stored
 * documents. 
 *  
 * 
 * @author Ali Mehregani
 */
public interface ISMLRepository
{
	/**
	 * A connection will first need to be established before 
	 * a repository can be used
	 * 
	 * @param attributes The attributes required as part of the connection.  This
	 * is specific to an implementation. 
	 * @throws RepositoryConnectionException If a connection cannot be established
	 */
    public void connect(Map<String, Object> attributes) throws RepositoryConnectionException; 
    
    
    /**
     * Returns a boolean indicating the connection status of the 
     * repository.
     * 
     * @return true if a connection has been established; false otherwise
     */
    public boolean isConnection();
    
    
    /**
     * Retrieve the property of type {@link Object} with the name passed
     * in.  The default value is returned if no such property is found
     * 
     * @param name The name of the property
     * @param defaultValue The default value
     * @return The property with the name passed in or defaultValue if no 
     * such property is found
     */
    public Object getProperty (String name, Object defaultValue);
    
    
    /**
     * Sets the property of type {@link Object} with the name and
     * value passed in
     * 
     * @param name The name of the property
     * @param value The value of the property
     */
    public void setProperty (String name, Object value);
    
    
    /**
     * Retrieve the property of type {@link String} with the name passed
     * in.  The default value is returned if no such property is found
     * 
     * @param name The name of the property
     * @param defaultValue The default value
     * @return The property with the name passed in or defaultValue if no 
     * such property is found
     */
    public String getProperty (String name, String defaultValue);
    
    
    /**
     * Sets the property of type {@link String} with the name and
     * value passed in
     * 
     * @param name The name of the property
     * @param value The value of the property
     */
    public void setProperty (String name, String value);
    
    
    /**
     * Retrieve the property of type boolean with the name passed
     * in.  The default value is returned if no such property is found
     * 
     * @param name The name of the property
     * @param defaultValue The default value
     * @return The property with the name passed in or defaultValue if no 
     * such property is found
     */
    public boolean getProperty (String name, boolean defaultValue);
    
    
    /**
     * Sets the property of type boolean with the name and
     * value passed in
     * 
     * @param name The name of the property
     * @param value The value of the property
     */
    public void setProperty (String name, boolean value);
    
    
    /**
     * Retrieve the property of type int with the name passed
     * in.  The default value is returned if no such property is found
     * 
     * @param name The name of the property
     * @param defaultValue The default value
     * @return The property with the name passed in or defaultValue if no 
     * such property is found
     */
    public int getProperty (String name, int defaultValue);
    
    
    /**
     * Sets the property of type int with the name and
     * value passed in
     * 
     * @param name The name of the property
     * @param value The value of the property
     */
    public void setProperty (String name, int value);
        
        
    /**
     * Retrieves the SML documents representing the root context
     * of a model.   This depends on the underlying model that a repository stores.
     * If there is no such notion of a root context in an underlying model, then an
     * empty list or null is expected to be returned.
     * 
     * @return The documents representing the root context of a model or
     * an empty list/null if no such thing exists
     * @throws RepositoryOperationException If the operation fails
     */
    public ISMLDocument[] retrieveRootDocuments() throws RepositoryOperationException;
    
    
    /**
     * Add a document to this repository.  The data of the SML document is read 
     * from the input stream and its meta-data (if one exists) is read from the 
     * metadata argument.
     * 
     * @param metadata The meta-data of the document.  The only relevant fields that 
     * should be read from the meta-data is the id, alias(es), and bound rule(s).  Other 
     * fields can potentially be used by implementers.  
     * @param input The input stream that contains the document
     * @throws RepositoryOperationException If the operation fails
     */
    public void addDocument(ISMLMetadata metadata, InputStream input) throws RepositoryOperationException;
    
    
    /**
     * Remove the SML document with the meta-data that is passed in.  The documents
     * removed is the returned set of fetchDocuments(metadata).  This operation has no
     * effect if no stored document matches the meta-data passed in.
     * 
     * @param metadata The meta-data matching the stored documents that will be removed
     * @throws RepositoryOperationException If the operation fails
     */
    public void removeDocuments(ISMLMetadata metadata) throws RepositoryOperationException;
    
    
    /**
     * Returns the ID of the supported operations of this repository.
     * 
     * @return The supported operations of this repository
     */
    public String[] getSupportedOperations();
    
    
    /**
     * Returns the operation matching the operation id passed in.
     * 
     * @param id The operation id
     * @return The supported operation or null if no such operation is
     * supported
     */
    public ISMLOperation getOperation(String id);
    
    
    /**
     * Returns the set of documents that match the meta-data that is passed in.
     * An empty list is returned if the meta-data does not match any stored documents
     * 
     * @param metadata The meta-data used to perform the query
     * @return The set of stored documents that match the meta-data passed in
     * @throws RepositoryOperationException If the operation fails
     */
    public ISMLDocument[] fetchDocuments (ISMLMetadata metadata) throws RepositoryOperationException;
    
    
    /**
     * Returns the set of active documents of this repository
     * 
     * @return The set of active documents or an empty list if no such
     * list exists.
     */
    public ISMLDocument[] getActiveDocuments();
    
    
    /**
     * Sets the set of active documents of this repository
     * 
     * @param documents The active documents of this repository
     */
    public void setActiveDocuments(ISMLDocument[] documents);
        
    
    /**
     * Disconnects this repository
     */
    public void disconnect();      
}
