/**********************************************************************
 * Copyright (c) 2008 IBM Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.cosmos.rm.internal.validation.smlvalidators;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.SAXParser;

import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ElementSchemaModel;
import org.eclipse.cosmos.rm.internal.validation.artifacts.ValidationSet;
import org.eclipse.cosmos.rm.internal.validation.common.SMLValidatorUtil;
import org.eclipse.cosmos.rm.internal.validation.common.AbstractValidationOutput.ValidationMessageFactory;
import org.eclipse.cosmos.rm.internal.validation.core.AbstractSMLValidator;
import org.eclipse.cosmos.rm.internal.validation.core.IValidator;
import org.eclipse.cosmos.rm.internal.validation.databuilders.SMLValidatingBuilder;
import org.xml.sax.InputSource;
import org.xml.sax.SAXParseException;

/**
 * Validates the schemas included in a validation set.  The
 * schemas are validated against each other and the "schema of
 * schema".
 * 
 * @author Ali Mehregani
 */
public class SchemaValidator extends AbstractSMLValidator
{
	/**
	 * Error prefix
	 */
	private static final String VALIDATION_ERROR_PREFIX = "cvc";

	/**
	 * States of input stream
	 */
	private static final int RESET = 0x00;
	private static final int CLOSE = 0x01;
	
	
	/**
	 * The validation set
	 */
	private ValidationSet validationSet;
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.core.AbstractValidator#initialize(java.util.Map)
	 */
	public void initialize(Map<String, Object> init)
	{		
		super.initialize(init);
		this.validationSet = (ValidationSet)init.get(IValidator.ATTRIBUTE_VALIDATION_SET);
	}
	
	
	/**
	 * @see org.eclipse.cosmos.rm.internal.validation.core.IValidator#validate()
	 */
	public boolean validate()
	{
		// Validate the definitions included in the validation
		// set against the instance documents
		ElementSchemaModel[] definitions = validationSet.getDefinitions();
		InputStream[] schemas = getSchemas();
		boolean status = true;
		try
		{	
			for (int i = 0; i < definitions.length; i++)
			{
				modifyStreamState(schemas, RESET);			
				status = status && validateSchema(definitions[i], getSchemas());
				if (!status && shouldAbortOnError())
				{
					break;
				}
			}
		}
		finally
		{
			modifyStreamState(schemas, CLOSE);	
		}
		
		return status;
	}
	
	
	/**
	 * Constructs a set of input streams using the definition
	 * documents of the validation set + the "schema of schema"
	 *  
	 * @return The schemas of the validation set
	 */
	private InputStream[] getSchemas()
	{
		List<InputStream> schemas = new ArrayList<InputStream>();
		schemas.add (this.getClass().getClassLoader().getResourceAsStream("validation-resources/XMLSchema.xsd"));
		ElementSchemaModel[] definitions = validationSet.getDefinitions();
		for (int i = 0; i < definitions.length; i++)
		{
			schemas.add(new ByteArrayInputStream(definitions[i].getSource().getBytes()));
		}
		
		return schemas.toArray(new InputStream[schemas.size()]);
	}


	/**
	 * Modifies the state of the input streams
	 * passed in.
	 * 
	 * @param streams The input stream
	 * @param state The state 
	 */
	private void modifyStreamState(InputStream[] streams, int state)
	{
		if (streams == null)
		{
			return;
		}
		
		try
		{
			for (int i = 0; i < streams.length; i++)
			{
				switch (state)
				{
					case RESET:
						if (streams[i].available() <= 0)
						{
							streams[i].reset();
						}
						break;
					case CLOSE:
						streams[i].close();
						break;				
				}
			}
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
	}


	/**
	 * Recursive method validates schema against all 
	 * prerequisite schemas
	 * 
	 * @param schema The schema to be checked
	 */
	private boolean validateSchema(ElementSchemaModel schema, InputStream[] schemas)
	{
		if (schema.hasBeenValidated()) 
		{
			// No need to validate a schema twice
			return true;
		}
		
		
		ArrayList<ElementModel> schemasForValidating = new ArrayList<ElementModel>();
		ArrayList<ElementModel> instanceSchemas = new ArrayList<ElementModel>();
		
		instanceSchemas.add(schema);
		String[] namespaces = schema.getNamespaceContext().getNamespaces() ;
		for (int i = 0; i < namespaces.length; i++)
		{
			if (namespaces[i].equals(schema.getTargetNamespace()))
			{
				continue;
			}
			
			ElementSchemaModel prereqModel = validationSet.getDefinitionByNamespace(namespaces[i]);
			if (prereqModel != null) 
			{
				schemasForValidating.add(prereqModel);
				validateSchema(prereqModel, schemas);
			}
		}

		try
		{				
			SAXParser schemaParser = SMLValidatorUtil.createSAXParser(schemas, true, null, getValidationOutput());
			schemaParser.parse(new InputSource(new ByteArrayInputStream(schema.getSource().getBytes())), new SMLValidatingBuilder(null));
			return true;
		}
		catch (SAXParseException e) 
		{
			reportError (schema, e);			
		} 
		catch (Exception e) 
		{
			getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(-1, e.getLocalizedMessage()));
		} 
		finally 
		{
			schema.markValidated();			
		}
		
		return false;
	}
	
	
	/**
	 * Report an error based on the expcetion passed in
	 * 
	 * @param elementModel The element model causing the error
	 * @param e The exception
	 */
	private void reportError(ElementModel elementModel, SAXParseException e) 
	{
		boolean validationError = elementModel instanceof ElementSchemaModel || 
								  e.getMessage().startsWith(VALIDATION_ERROR_PREFIX);
		
		int lineNumber = -1;
		ElementModel errorLocation = null;
		
		if (validationError) 
		{
			errorLocation = elementModel;
			lineNumber = elementModel.getLineNumber() + e.getLineNumber();
		} 
		else 
		{
			errorLocation = elementModel;
			lineNumber = elementModel.getLineNumber();
		}
		
		getValidationOutput().reportMessage(ValidationMessageFactory.createErrorMessage(errorLocation == null ? 
				null : errorLocation.getFilePath(), 
				lineNumber, e.getColumnNumber(), e.getLocalizedMessage()));
	}

}
