/*******************************************************************************
 * Copyright (c) 2009 SAS Institute, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.cim.profile;

import java.io.File;

import org.eclipse.cosmos.me.sdd.cr.Constraint;
import org.eclipse.cosmos.me.sdd.cr.IResolver;
import org.eclipse.cosmos.me.sdd.schema.ext.ResolutionObject;

/**
 * Directory resolver class
 * 
 * @author jehamm
 *
 */
public class DirectoryResolver implements IResolver {


	/**
	 * Resolve this class based on Constraint defined. The Constraint may define a property or
	 * a version.
	 * 
	 * @param c
	 */
	public ResolutionObject resolve(Constraint c) {
		// Create a container for resolution
		ResolutionObject resolution = new ResolutionObject();
		if(c.getType().equalsIgnoreCase("property")) {
			return resolveProperty(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("version")) {
			return resolveVersion(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("capacity")) {
			return resolveCapacity(c, resolution);
		}
		else if(c.getType().equalsIgnoreCase("consumption")) {
			return resolveConsumption(c, resolution);
		}
		return resolution;
	}
	
	/**
	 * Resolve a consumption constraint
	 * 
	 * @param c
	 * @param resolution
	 * @return
	 */
	private ResolutionObject resolveConsumption(Constraint c, ResolutionObject resolution) {
		System.out.println("DirectoryResolver.resolveConsumption() - not implemented.");
		return resolution;
	}

	/**
	 * Resolve a capacity constraint.
	 * 
	 * @param c
	 * @param resolution
	 * @return
	 */
	private ResolutionObject resolveCapacity(Constraint c, ResolutionObject resolution) {
		System.out.println("DirectoryResolver.resolveCapacity() - not implemented.");
		return resolution;
	}


	/**
	 * Resolve version data that can be associated with CIM_OperatingSystem
	 * @param c
	 * 		Constraint instance that defines the SDD property constraints.
	 * @param resolution
	 * 		Resolution instance that will contain the results of the resolution.
	 * @return
	 * 		the completed resolution if we found a 
	 */
	private ResolutionObject resolveVersion(Constraint c, ResolutionObject resolution) {
		System.out.println("DirectoryResolver.resolveVersion() - not implemented.");
		return resolution;
	}

	/**
	 * Resolve property that can be associated with CIM_OperatingSystem
	 * @param c
	 * 		Constraint instance that defines the SDD property constraints.
	 * @param resolution
	 * 		Resolution instance that will contain the results of the resolution.
	 * @return
	 * 		the completed resolution if we found a 
	 */
	private ResolutionObject resolveProperty(Constraint c, ResolutionObject resolution) {

		//Readable
		//Writable.
		for(String name : c.getNames()) {
			if(name.endsWith("Readable")) {
				return resolveDirectoryReadable(name, c, resolution);
			}
			else if(name.endsWith("Writable")) {
				return resolveDirectoryWritable(name, c, resolution);
			}
		}
		return null;
	}
	
	/**
	 * Particular resolver to solve if a directory is writable or not.
	 * @param name
	 * @param c
	 * @param resolution
	 * @return
	 */
	private ResolutionObject resolveDirectoryWritable(String name, Constraint c,
			ResolutionObject resolution) {
				
		File directory = new File(c.getResource());
		directory.mkdirs();
		
		// Get the value that we are after.
		String [] values = c.getValues(name);
		boolean compareTo = Boolean.parseBoolean(values[0]);
		
		// /See if what we are hoping to match actually matches.
		if(directory.canWrite() == compareTo) {
			resolution.accept();
			c.accept();
		}
		else {
			resolution.reject();
			c.reject();
		}
		
		return resolution;
	}
	
	/**
	 * Particular resolver to solve if a directory is readable or not.
	 * @param name
	 * @param c
	 * @param resolution
	 * @return
	 */
	private ResolutionObject resolveDirectoryReadable(String name, Constraint c,
			ResolutionObject resolution) {
		File directory = new File(c.getResource());
		
		// what are we looking for
		String [] values = c.getValues(name);
		boolean compareTo = Boolean.getBoolean(values[0]);
		
		if(directory.canRead() == compareTo) {
			resolution.accept();
			c.accept();
		}
		else {
			resolution.reject();
			c.reject();
		}
		
		return resolution;
	}	
	

}
