/*******************************************************************************
 * Copyright (c) 2009 SAS Institute, Inc., and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     SAS Institute - initial API and implementation
 *     SAS Institute - Bug 275699
 *     SAS Institute - Bug 275673
 ******************************************************************************/
package org.eclipse.cosmos.me.sdd.cli;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;

import org.eclipse.cosmos.me.sdd.advisor.IAdvisor;
import org.eclipse.cosmos.me.sdd.schema.BooleanParameterType;
import org.eclipse.cosmos.me.sdd.schema.IntegerParameterType;
import org.eclipse.cosmos.me.sdd.schema.RequirementResourceConstraintType;
import org.eclipse.cosmos.me.sdd.schema.ResourceType;
import org.eclipse.cosmos.me.sdd.schema.StringParameterType;
import org.eclipse.cosmos.me.sdd.schema.URIParameterType;

public class CLIAdvisor implements IAdvisor {

	private static final String ONE = "1";
	private static final String ZERO = "0";
	private static final String TRUE = "true";
	private static final String FALSE = "false";
	
	private BufferedReader br;

	public CLIAdvisor() {
		// initialize br
		br = new BufferedReader(new InputStreamReader(System.in));
	}

	public void displayMessage(String message) {
		System.out.println(message);
	}

	public Integer requestParameter(IntegerParameterType ipt) {
		String value = readVariable(ipt.getDisplayName() != null ? ipt
				.getDisplayName().getValue() : null, ipt.getDefaultValue());
		
		Integer returnVal = null;
		// try to parse Integer from string, if unable to do so, repeat
		try {
			returnVal = Integer.parseInt(value);
		} catch (Exception e) {
			displayMessage(Messages.getString("CLIAdvisor.ProvideEntry")); 
			return (requestParameter(ipt));
		}
		return returnVal;
	}

	public String requestParameter(StringParameterType spt) {
		return readVariable(spt.getDisplayName() != null ? spt
				.getDisplayName().getValue() : null, spt.getDefaultValue());
	}

	public Boolean requestParameter(BooleanParameterType bpt) {
		String value = readVariable(bpt.getDisplayName() != null ? bpt
				.getDisplayName().getValue() : null, bpt.getDefaultValue());

		// try to parse boolean from string, if unable to do so, repeat
		if (value.equalsIgnoreCase(CLIAdvisor.ONE))
			return true;
		else if (value.equalsIgnoreCase(CLIAdvisor.ZERO))
			return false;
		else if (value.equalsIgnoreCase(CLIAdvisor.TRUE))
			return true;
		else if (value.equalsIgnoreCase(CLIAdvisor.FALSE))
			return false;
		else {
			displayMessage(Messages.getString("CLIAdvisor.ProvideEntry")); 
			return (requestParameter(bpt));
		}
	}

	public URI requestParameter(URIParameterType upt) {
		URI returnVal = null;
		// try to parse URI from string, if unable to do so, repeat
		try {
			returnVal = URI.create(readVariable(upt.getDisplayName() != null ? upt
					.getDisplayName().getValue() : null, upt.getDefaultValue()));
		} catch (Exception e) {
			return requestParameter(upt);
		}
		return returnVal;
	}

	// method simply takes in a name and a default value
	// and returns the users input
	private String readVariable(String name, String defaultVal) {
		
		boolean hasDefault = false;

		if(name == null)
			name = Messages.getString("CLIAdvisor.MissingDisplayName");
			
		// if there is a defaultValue, display it and give the option to use it
		if ((defaultVal != null) && (defaultVal.length() > 0)) {
			hasDefault = true;
			System.out.print(Messages.getString("CLIAdvisor.UseDefault", name, defaultVal));
		}
		else {
			System.out.print(Messages.getString("CLIAdvisor.ProvideValue", name));
		}

		String value = null;
		try {
			value = br.readLine();
		} catch (IOException e) {
			e.printStackTrace();
		}

		// if they press enter and there is a default value, return it,
		// otherwise, ask again.
		if ((value == null) || (value.length() == 0)) {
			if (hasDefault)
				value = defaultVal;
			else
				value = readVariable(name, defaultVal);
		}

		return value;
	}

	public void displayError(RequirementResourceConstraintType resource) {
		ResourceType rsrc = (ResourceType) resource.getResourceRef();
		displayMessage(Messages.getString("CLIAdvisor.FailedResolution", resource.getId() , rsrc.getName()));
	}
}
