/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
/**
 * 
 */
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.tooling.btg;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.tooling.btg.BTGApplication;
import org.eclipse.cosmos.me.internal.deployment.sdd.tooling.btg.util.PropertiesReader;
import org.eclipse.cosmos.me.provisional.deployment.sdd.tooling.btg.BTGProperty;

/**
 *  
 *
 */
public class PropertiesReader_Test extends TestCase {

	BTGApplication btgApp;
	
	/**
	 * @param name
	 */
	public PropertiesReader_Test(String name) {
		super(name);
	}

	/* (non-Javadoc)
	 * @see junit.framework.TestCase#setUp()
	 */
	protected void setUp() throws Exception {
		super.setUp();
		btgApp = new BTGApplication();
	}

	/* (non-Javadoc)
	 * @see junit.framework.TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
		super.tearDown();
	}
	
	public void testInvalidFileName() throws ParseException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader("badFileName");
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("Did not throw FileNotFoundException when invalid file name is given.");
		}catch (FileNotFoundException e){
			assertTrue(true);
		}
	}
	
	public void testNoBTGProperties() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.NO_BTG_PROP);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("Did not throw ParseException when no BTG properties section is defined.");
		}catch (ParseException e){
			assertTrue(true);
		}
	}
	
	public void testInvalidBTGPluginID() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.INVALID_BTG_ID);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("Did not throw ParseException when invalid BTG ID is defined.");
		}catch (ParseException e){
			assertTrue(true);
		}
	}
	
	public void testBTGNoOutputFileNameBaseValue() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.BTG_NO_OUTPUTFILENAMEBASEVALUE);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("Did not throw ParseException when BTG property OutputFileNameBase has no value.");
		}catch (ParseException e){
			assertTrue(true);
		}
	}
	
	public void testBTGNoOutputFileNameBase() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.BTG_NO_OUTPUTFILENAMEBASE);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("Did not throw ParseException when BTG properties does not contain OutputFileNameBase.");
		}catch (ParseException e){
			assertTrue(true);
		}
	}
	
	public void testDuplicateBTGProperties() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.BTG_DUPLICATE_PROPERTIES);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("Did not throw ParseException when duplicate BTG properties were found.");
		}catch (ParseException e){
			assertTrue(true);
		}
	}
	
	public void testBTGWithUnsupportedProperty_Enabled() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.BTG_UNSUPPORTED_PROP_ENABLED);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			for (Properties p : props) {

				if (PropertiesReader.BTG_PLUGINID.equals(p.get(PropertiesReader.PLUGINID_PROPERTY))) {
					// Validate the BTG properties
					propsReader.checkSupportedProperties(p, btgApp.getSupportedProperties());
					
				}
			}
			
			fail("Did not throw ParseException when unsupported BTG property (Enabled) was found.");
		}catch (ParseException e){
			assertTrue(true);
		}
	}
	
	public void testBTGWithUnsupportedProperty_Other() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.BTG_UNSUPPORTED_PROP_OTHER);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			for (Properties p : props) {

				if (PropertiesReader.BTG_PLUGINID.equals(p.get(PropertiesReader.PLUGINID_PROPERTY))) {
					// Validate the BTG properties
					propsReader.checkSupportedProperties(p, btgApp.getSupportedProperties());
					
				}
			}
			
			fail("Did not throw ParseException when unsupported BTG property (Enabled) was found.");
		}catch (ParseException e){
			assertTrue(true);
		}
	}
	
	public void testUnknownPluginID() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.UNKNOWN_PLUGIN_ID);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			assertTrue(true);
		}catch (ParseException e){		
			fail("ParseException was thrown with unknown plugin ID.");
		}
	}
	
	public void testInvalidPropertySyntax_NoEqualSymbol() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.INVALID_PROP_SYNTAX);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown with invalid property syntax.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testInvalidPropertySyntax_NoPropertyName() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.INVALID_PROP_SYNTAX2);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown with invalid property syntax.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testInvalidPropertySyntax_NoPropertyValue() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.INVALID_PROP_SYNTAX3);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown with invalid property syntax.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testInvalidFirstLine() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.INVALID_FIRST_LINE);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown with invalid first line in file.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testDuplicatePropertyInSection() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.DUPLICATE_PROPERTY_IN_SECTION);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("Did not throw ParseException when duplicate property was found in same section.");
		}catch (ParseException e){
			assertTrue(true);
		}
	}
	
	public void testInvalidNoProperty() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.INVALID_NO_PROPERTY);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown when no property was found in file.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testRPMNoEnabledProperty() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.RPM_NO_ENABLED_PROPERTY);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown when RPM plugin section has no enabled property.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testRPMNoFileNameProperty() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.RPM_NO_FILENAME_PROPERTY);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			for (Properties p : props) {
				String pluginID = "org.eclipse.cosmos.me.deployment.sdd.tooling.btg.plugin.rpmreader";
				Collection<BTGProperty> properties = new ArrayList<BTGProperty>();
				properties.add(new BTGProperty("FileName", true)); //$NON-NLS-1$
				
				if (pluginID.equals(p.get(PropertiesReader.PLUGINID_PROPERTY))) {
					// Validate the BTG properties
					propsReader.checkSupportedProperties(p, properties);
					
				}
			}
			fail("ParseException was not thrown when RPM plugin section has no filename property.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testAggregatorNoEnabledProperty() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.AGGREGATOR_NO_ENABLED_PROPERTY);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown when Aggregator plugin section has no enabled property.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testAggregatorNoRuleLocationProperty() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.AGGREGATOR_NO_RULELOCATION_PROPERTY);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			for (Properties p : props) {
				String pluginID = "org.eclipse.cosmos.me.deployment.sdd.tooling.btg.aggregator";
				Collection<BTGProperty> properties = new ArrayList<BTGProperty>();
				properties.add(new BTGProperty("RulesLocation", true)); //$NON-NLS-1$
				
				if (pluginID.equals(p.get(PropertiesReader.PLUGINID_PROPERTY))) {
					// Validate the BTG properties
					propsReader.checkSupportedProperties(p, properties);
					
				}
			}
			fail("ParseException was not thrown when Aggregator plugin section has no RuleLocation property.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testSDDReaderNoEnabledProperty() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.SDDREADER_NO_ENABLED_PROPERTY);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown when SDDReader plugin section has no enabled property.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testSDDReaderNoFileNameProperty() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.SDDREADER_NO_FILENAME_PROPERTY);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			for (Properties p : props) {
				
				String pluginID = "org.eclipse.cosmos.me.deployment.sdd.tooling.btg.plugin.sddreader";
				Collection<BTGProperty> properties = new ArrayList<BTGProperty>();
				properties.add(new BTGProperty("FileName", true)); //$NON-NLS-1$
				
				if (pluginID.equals(p.get(PropertiesReader.PLUGINID_PROPERTY))) {
					// Validate the BTG properties
					propsReader.checkSupportedProperties(p, properties);
					
				}
			}
			fail("ParseException was not thrown when SDDReader plugin section has no filename property.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testLastSectionNoProperties() throws FileNotFoundException, IOException{
		// Read in the list of properties from properties file
		PropertiesReader propsReader = new PropertiesReader(AllTests.LAST_SECTION_NO_PROPERTIES);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			fail("ParseException was not thrown when last section has no properties.");
		}catch (ParseException e){		
			assertTrue(true);
		}
	}
	
	public void testValidPropertiesFile() throws FileNotFoundException, IOException, ParseException {
		PropertiesReader propsReader = new PropertiesReader(AllTests.VALID_PROP);
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			assertTrue(true);
		}catch (ParseException e){		
			fail("Valid Properties file has ParseException: " + e.getMessage());	
			
		}
	}
	
	public void testBTGSectionOnly() throws FileNotFoundException, IOException, ParseException {
		PropertiesReader propsReader = new PropertiesReader(AllTests.BTG_SECTION_ONLY);
		
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			assertTrue(true);
		}catch (ParseException e){		
			fail("Properties file has ParseException: " + e.getMessage());	
			
		}
	}
	
	public void testEnabledPluginAdded() throws FileNotFoundException, IOException, ParseException {
		PropertiesReader propsReader = new PropertiesReader(AllTests.VALID_ONE_PLUGIN_ENABLED);
		
		try {
			Collection<Properties> props = propsReader.getPropertiesList();
			String pluginID = "org.eclipse.cosmos.me.deployment.sdd.tooling.btg.plugin.rpmreader";
			assertEquals(PropertiesReader.BTG_PLUGINID, props.toArray(new Properties[0])[0].get(PropertiesReader.PLUGINID_PROPERTY));
			assertEquals(pluginID, props.toArray(new Properties[0])[1].get(PropertiesReader.PLUGINID_PROPERTY));
			assertEquals(2, props.size());
			
		}catch (ParseException e){		
			fail("Properties file has ParseException: " + e.getMessage());	
		}
		
	}

}
