/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.validation;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.cosmos.me.internal.deployment.sdd.resources.common.validation.Messages;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.ValidationRule;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.XMLValidationError;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

/**
 * 
 * @author Eric S Rose (esrose@us.ibm.com) and Alan Watkins
 *
 */
public class SchemaValidationRule implements ValidationRule, ErrorHandler {
	// List of Schema Files to use in validation
	private Collection<File> schemaFiles = new Vector<File>();
	// Mapping between InputStreams and associated temporary Files
	private Map<InputStream, File> streamsToFiles = new HashMap<InputStream, File>();

	// List of validation errors
	private Collection<XMLValidationError> xmlValidationErrorList = new Vector<XMLValidationError>();

	public static String SCHEMA_LANGUAGE = "http://java.sun.com/xml/jaxp/properties/schemaLanguage"; //$NON-NLS-1$
	public static String XML_SCHEMA = "http://www.w3.org/2001/XMLSchema"; //$NON-NLS-1$
	public static String SCHEMA_SOURCE = "http://java.sun.com/xml/jaxp/properties/schemaSource"; //$NON-NLS-1$

	/**
	 * Construct a SchemaValidationRule from a Collection of Schema sources
	 * 
	 * @param schemaSources Collection of Schema sources, either Files or InputStreams
	 * 
	 * @throws IOException if there's an error reading from a given File or InputStream
	 * @throws FileNotFoundException if one of the given Files is missing
	 */
	public <T> SchemaValidationRule(Collection<T> schemaSources) throws IOException, FileNotFoundException {
		for (T schemaSource : schemaSources) {
			if (schemaSource instanceof File) {
				File schemaFile = (File)schemaSource;
				addSchemaFile(schemaFile);
			}
			
			if (schemaSource instanceof InputStream) {
				InputStream schemaStream = (InputStream)schemaSource;
				addSchemaInputStream(schemaStream);
			}
		}
	}

	/**
	 * Construct a SchemaValidationRule from a Schema File
	 * 
	 * @param schemaFile Schema File
	 * 
	 * @throws IOException if there's an error reading from the given File
	 * @throws FileNotFoundException if the given File is missing
	 */
	public SchemaValidationRule(File schemaFile) throws FileNotFoundException, IOException {
		addSchemaFile(schemaFile);
	}

	/**
	 * Construct a SchemaValidationRule from a Schema InputStream
	 * 
	 * @param schemaStream Schema InputStream
	 * 
	 * @throws IOException if there's an error reading from the given InputStream
	 */	
	public SchemaValidationRule(InputStream schemaStream) throws IOException {
		addSchemaInputStream(schemaStream);
	}

	public Collection<XMLValidationError> validate(InputStream xmlStream) {
		xmlValidationErrorList = new Vector<XMLValidationError>();
		// Go through the validation process.  Validate xmlStream against schemaFile

		// Create a factory
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		// Set attributes of the factory
		factory.setNamespaceAware(true);
		factory.setValidating(true);

		try {
			factory.setAttribute(SCHEMA_LANGUAGE, XML_SCHEMA);
			factory.setAttribute(SCHEMA_SOURCE, schemaFiles.toArray(new File[0]));

		} catch (IllegalArgumentException e) {
			String errorMsg = Messages.getString("SchemaValidationRule.2"); //$NON-NLS-1$
			XMLValidationError validationError = new XMLValidationError(errorMsg, this, e);
			xmlValidationErrorList.add(validationError);

			return xmlValidationErrorList;
		}

		// Now, create the parser...
		DocumentBuilder builder;
		try {
			builder = factory.newDocumentBuilder();

		} catch (ParserConfigurationException e) {
			String errorMsg = Messages.getString("SchemaValidationRule.3"); //$NON-NLS-1$
			XMLValidationError validationError = new XMLValidationError(errorMsg, this, e);
			xmlValidationErrorList.add(validationError);

			return xmlValidationErrorList;
		}

		// Set up the error handler...
		builder.setErrorHandler(this);

		try {
			// Parse the document...
			builder.parse(xmlStream);

		} catch (IOException e) {
			String errorMsg = Messages.getString("SchemaValidationRule.4"); //$NON-NLS-1$
			XMLValidationError validationError = new XMLValidationError(errorMsg, this, e);
			xmlValidationErrorList.add(validationError);

			return xmlValidationErrorList;

		} catch (SAXException e) {
			String errorMsg = Messages.getString("SchemaValidationRule.5"); //$NON-NLS-1$
			XMLValidationError validationError = new XMLValidationError(errorMsg, this, e);
			xmlValidationErrorList.add(validationError);

			return xmlValidationErrorList;

		} catch (IllegalArgumentException e) {
			String errorMsg = Messages.getString("SchemaValidationRule.6"); //$NON-NLS-1$
			XMLValidationError validationError = new XMLValidationError(errorMsg, this, e);
			xmlValidationErrorList.add(validationError);

			return xmlValidationErrorList;
		}

		// Return the error list (which may be empty)
		return xmlValidationErrorList;
	}

	/**
	 * Get a Collection of the Schema Files associated with the Rule
	 * 
	 * @return Collection of Schema Files associated with the ValidationRule instance
	 */
	public Collection<File> getSchemaFiles() {
		return schemaFiles;
	}

	/**
	 * Get a Collection of the Schema InputStreams associated with the Rule
	 * 
	 * @return Collection of Schema InputStreams associated with the ValidationRule instance
	 */
	public Collection<InputStream> getSchemaInputStreams() {
		return streamsToFiles.keySet();
	}

	/**
	 * Add a Schema File to the Rule's list
	 * 
	 * @param schemaFile Schema File to add to the validation list
	 */
	public void addSchemaFile(File schemaFile) throws IOException, FileNotFoundException {
		// Make sure Schema file exists
		if (!schemaFile.exists()) {
			throw new FileNotFoundException(Messages.getString("SchemaValidationRule.0", schemaFile.getName())); //$NON-NLS-1$
		}
		
		// Make sure Schema file is readable				
		if (!schemaFile.canRead()) {
			throw new IOException(Messages.getString("SchemaValidationRule.1", schemaFile.getName())); //$NON-NLS-1$				
		}
		
		schemaFiles.add(schemaFile);
	}

	/**
	 * Remove a Schema File from the Rule's list
	 * 
	 * @param schemaFile Schema File to remove from the validation list
	 */
	public void removeSchemaFile(File schemaFile) {
		schemaFiles.remove(schemaFile);

		// Check if an entry in the streams Hash points to this file
		if (streamsToFiles.containsValue(schemaFile)) {
			// Remove the associated entry from the Hash
			for (InputStream stream : streamsToFiles.keySet()) {
				if (streamsToFiles.get(stream).equals(schemaFile)) {
					streamsToFiles.remove(stream);
				}
			}
		}
	}

	/**
	 * Add a Schema InputStream to the Rule's list
	 * 
	 * @param schemaStream Schema InputStream to add to the validation list
	 * 
	 * @throws IOException if there's an error reading from the InputStream
	 */
	public void addSchemaInputStream(InputStream schemaStream) throws IOException {
		// Create a temporary file for this InputStream
		File tempFile = File.createTempFile("SchemaFileFromStream", null);

		// Copy the contents of the stream to the temp file
		OutputStream out = new FileOutputStream(tempFile);
		byte[] copyBuffer = new byte[1024];
		int bytesRead = 0;
		while ((bytesRead = schemaStream.read(copyBuffer)) != -1) {
			out.write(copyBuffer, 0, bytesRead);
		}

		// Add the InputStream and File to the Hash
		streamsToFiles.put(schemaStream, tempFile);
		// Add the temp File to the list
		schemaFiles.add(tempFile);

		// Delete the temp File on exit
		tempFile.deleteOnExit();				
	}

	/**
	 * Remove a Schema InputStream from the Rule's list
	 * 
	 * @param schemaStream Schema InputStream to remove from the validatoin list
	 */
	public void removeSchemaInputStream(InputStream schemaStream) {
		// Search for the given InputStream in the Hash
		if (streamsToFiles.containsKey(schemaStream)) {
			// Remove the File associated with this InputStream from the files list (also removes it from the Hash)
			removeSchemaFile((File)streamsToFiles.get(schemaStream));
		}
	}

	// The following 3 methods implement the ErrorHandler interface
	public void error(SAXParseException e) {
		addXMLValidationError(e, XMLValidationError.ERROR);
	}

	public void fatalError(SAXParseException e) {
		addXMLValidationError(e, XMLValidationError.FATAL_ERROR);
	}

	public void warning(SAXParseException e) {
		addXMLValidationError(e, XMLValidationError.WARNING);
	}

	private void addXMLValidationError(SAXParseException e, int severity) {
		XMLValidationError validationError = new XMLValidationError(e
				.getMessage(), e.getLineNumber(), e.getColumnNumber(), this,
				severity);
		xmlValidationErrorList.add(validationError);
	}
}