/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.examples.common.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.SDDManagerImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.util.ValidatorUtils;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.SDDManager;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/*
 *  SDDManager Sample
 *  
 *  Eric S Rose, esrose@us.ibm.com
 *  
 *  The provided SDDManager implementation encapsulates an entire Solution Deployment Descriptor (SDD)
 *  within a single instance of the class.  For the purposes of the SDDManager, an SDD consists of the following:
 *  some number of Package Descriptors, some number of Deployment Descriptors, a XML_DAS instance representing
 *  each XML file and a set of ValidationRules to validate the descriptors.
 *  
 *  This sample shows how to perform the following steps using the SDDManager:
 *  
 *  - Create an SDDManager and add a Package Descriptor and a Deployment Descriptor to it
 *  - Create ValidationRules and associate them with the Descriptors
 *  - Make changes to the DataGraphs representing the descriptors
 *  - Validate and write the changed DataGraphs back out to XML files
 *  
 */


public class SDDManager_Sample {
	// Use platform independent file separator
	private static final String FSEP = System.getProperty("file.separator");
	
	// Path to the location of files needed for the sample to run
	private static final String filesPath = "src" + FSEP + "org" + FSEP + "eclipse" + FSEP + "cosmos" + FSEP + "me"
		+ FSEP + "internal" + FSEP + "deployment" + FSEP + "sdd" + FSEP + "examples" + FSEP + "common" + FSEP
		+ "validation" + FSEP;

	// This sample depends on the following files
	private static final File deploymentDescriptor = new File(filesPath + "DeploymentDescriptor.xml");
	private static final File packageDescriptor = new File(filesPath + "PackageDescriptor.xml");
	

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// Copy over the original XML files so we start with known descriptors
		prepFiles();
		
		// Create a new instance of the SDDManagerImpl
		SDDManager mySDDManager = new SDDManagerImpl();
		
		// We want to store the Documents the SDDManager loads for us from the XML files
		Document ddDocument = null;
		Document pdDocument = null;
		
		try {
			// Add a Deployment Descriptor and Package Descriptor to the SDDManager from XML files
			ddDocument = mySDDManager.addDescriptorFromFile(deploymentDescriptor);
			pdDocument = mySDDManager.addDescriptorFromFile(packageDescriptor);
		} catch (IllegalArgumentException e) {
			// Thrown if descriptor file is null (should not be thrown here since files are not null)
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		} catch (FileNotFoundException e) {
			// Thrown if one of the files is missing
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		} catch (IOException e) {
			// Thrown if one of the files in unreadable
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		} catch (XMLValidationException e) {
			// Thrown if either file contains invalid XML
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		}
		
		/*
		 * If we get here no exception was thrown and the SDDManager instance contains two valid descriptors,
		 * one deployment descriptor and one package descriptor.
		 * 
		 * Since the SDDManager uses SDD_DAS instances for reading/writing descriptors, there's automatically
		 * a SchemaValidationRule associated with each descriptor.  Additional ValidationRules could be added here. 
		 */

		
		/*
		 * Now we can make changes to the descriptors and verify that they're still valid.
		 * 
		 * The following section appears in the Deployment Descriptor:
		 * 
		 * <sdd-dd:Completion type="restartRequired" resourceRef="framework" operation="install">
		 *   <sdd-common:Description>CompletionAction Desc</sdd-common:Description>
		 * </sdd-dd:Completion>
		 * <sdd-dd:Completion type="restartRequired" resourceRef="framework" operation="uninstall"/>
		 * 
		 * We've decided that no completion action is necessary so we want to remove the entire section.
		 */
		// Find the Completion element
		Element completionElement = ValidatorUtils.findElement(ddDocument, "Completion");
		// Remove it
		completionElement.getParentNode().removeChild(completionElement);
		
		// Validate the descriptors and write them back out to the XML files
		try {
			mySDDManager.writeAllDescriptors(true);
		} catch (XMLValidationException e) {
			// Thrown if the changes made caused one of the descriptors to become invalid
			System.err.println("Error validating a changed descriptor:\n");
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		} catch (IOException e) {
			// Thrown if there's a problem writing to one of the files
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		}
		
		// At this point, you can check DeploymentDescriptor.xml and PackageDescriptor.xml to see the changes
		// as compared to the originals.
	}
	
	private static void prepFiles() {
		final File ddOriginal = new File(filesPath + "DeploymentDescriptor.xml_Original");
		final File pdOriginal = new File(filesPath + "PackageDescriptor.xml_Original");
		
		// Overwrite the descriptor files with the originals so we start from a known state
		try {
			
			// Copy the Deployment Descriptor
			FileChannel ddSrcChannel = new FileInputStream(ddOriginal).getChannel();
			FileChannel ddDestChannel = new FileOutputStream(deploymentDescriptor).getChannel();
			ddDestChannel.transferFrom(ddSrcChannel, 0, ddSrcChannel.size());
			
			// Copy the Package Descriptor
			FileChannel pdSrcChannel = new FileInputStream(pdOriginal).getChannel();
			FileChannel pdDestChannel = new FileOutputStream(packageDescriptor).getChannel();
			pdDestChannel.transferFrom(pdSrcChannel, 0, pdSrcChannel.size());
			
		} catch (FileNotFoundException e) {
			// Thrown if one of the original files is missing
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		} catch (IOException e) {
			// Thrown if there's a problem reading or writing one of the files
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		}
	}
}
