/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.examples.common.validation;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.SDD_DASImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.validation.util.ValidatorUtils;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.SDD_DAS;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.XMLValidationError;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation.exception.XMLValidationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/*
 *  SDD_DAS Sample
 *  
 *  Eric S Rose, esrose@us.ibm.com
 *  
 *  The SDD Data Access Service (DAS) provides an interface for loading XML into a Document Object Model (DOM)
 *  Document, validating the contents of the Document against a set of ValidtionRules and writing a Document
 *  back out in XML.  The SDD_DAS implementation provided has the ability to use InputStreams and OutputStreams
 *  as the back end XML store.  It is intended specifically for working with Solution Deployment Descriptors instead
 *  of generic XML.  See the XML_DAS Sample for usage of the generic XML_DAS.
 *  
 *  This sample shows how to perform the following steps using the provided XML_DAS implementation:
 *  
 *  - Load the contents of an SDD XML file into a Document
 *  - Make a change to the Document
 *  - Validate the Document against an XML Schema
 *  - Write the Document back out to a (different) XML file
 */


public class SDD_DAS_Sample {
	// Use platform independent file separator
	private static final String FSEP = System.getProperty("file.separator");

	// Path to the location of files needed for the sample to run
    private static final String filesPath = "src" + FSEP + "org" + FSEP + "eclipse" + FSEP + "cosmos" + FSEP + "me"
    	+ FSEP + "internal" + FSEP + "deployment" + FSEP + "sdd" + FSEP + "examples" + FSEP + "common" + FSEP
    	+ "validation" + FSEP;
	
	// This sample depends on the following files
	private static final String XML_INPUT_FILE = filesPath + "DeploymentDescriptor.xml";
	
	// This file is created by the sample
	private static final String XML_OUTPUT_FILE = filesPath + "SDD_DAS_Sample-Output.xml";
	
	
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		// Copy over the original XML files so we start with known descriptors
		prepFiles();

		// Create an SDD_DAS instance representing the XML file
		SDD_DAS deploymentDescriptorDAS = new SDD_DASImpl();
		
		// Load the XML into a Document
		Document document = null;
		try {
			document = deploymentDescriptorDAS.loadDocument(new FileInputStream(XML_INPUT_FILE));
		} catch (FileNotFoundException e) {
			// Thrown if the XML file is missing
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		} catch (IOException e) {
			// Thrown if the XML file is present but unreadable
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		} catch (XMLValidationException e) {
			// Thrown if the XML is not well-formed
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);
		}
		
		/*
		 * The sample DeploymentDescriptor.xml file has a resource entry of type OperatingSystem with a name
		 * of Windows.  We want to change the name to Linux.  Here's the section of XML to change:
		 * 
		 * <dd:Topology>
		 *    <dd:Resource type="OperatingSystem" id="os">
		 *       <sddcommon:Name>Windows</sddcommon:Name>   <- We want to change this to Linux
		 *       ...
		 *    </dd:Resource>
		 * </dd:Topology>
		 */
		
		// We'll get the name by getting the Resource element, then getting the Name child element.
		// Then we can change Name's value.
		Element resourceElement = ValidatorUtils.findElement(document, "Resource");
		Element nameElement = ValidatorUtils.findElement(resourceElement, "Name");
		nameElement.setTextContent("Linux");
		
		/*
		 * Now that we've made a change, let's make sure out Document is still valid.  Every instance of
		 * an SDD_DAS automatically contains a SchemaValidationRule with the SDD Schema files.  We'll
		 * now run that rule via the DAS to validate the changed Document.
		 */
		// Validate the updated Document
		Collection<XMLValidationError> validationErrors = deploymentDescriptorDAS.validate(document);
		
		if (validationErrors.size() > 0) {
			// Document isn't valid
			for (XMLValidationError error : validationErrors) {
				System.err.println(error.getErrorMessage());
			}

			// No point in continuing - bail out
			System.exit(1);
		}
		
		// If we get here, the Document is valid.
		// Write it to a different XML file by passing the SDD_DAS an OutputStream pointing to a different file
		try {
			deploymentDescriptorDAS.saveDocument(document, new FileOutputStream(XML_OUTPUT_FILE), null, false);
			/*
			 * NOTE:
			 * We could have saved the Document to the file and performed validation in a single step by
			 * changing the last parameter in the method call above to true.
			 */
		} catch (XMLValidationException e) {
			// Since we didn't validate during save, this exception should never be thrown
		} catch (IOException e) {
			// Thrown if the output file isn't writable
			e.printStackTrace();
		}
		
		/*
		 * Check the output file to see the results of the Document changes
		 */
	}
	
	private static void prepFiles() {
		final File ddOriginal = new File(filesPath + "DeploymentDescriptor.xml_Original");
		
		// Overwrite the descriptor files with the originals so we start from a known state
		try {
			
			// Copy the Deployment Descriptor
			FileChannel ddSrcChannel = new FileInputStream(ddOriginal).getChannel();
			FileChannel ddDestChannel = new FileOutputStream(XML_INPUT_FILE).getChannel();
			ddDestChannel.transferFrom(ddSrcChannel, 0, ddSrcChannel.size());
			
		} catch (FileNotFoundException e) {
			// Thrown if one of the original files is missing
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		} catch (IOException e) {
			// Thrown if there's a problem reading or writing one of the files
			e.printStackTrace();
			// No point in continuing - bail out
			System.exit(1);			
		}
	}
}
