/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.provisional.deployment.sdd.common.validation;



/**
 * 
 * @author Eric S. Rose (esrose@us.ibm.com)
 *
 */
public class XMLValidationError {
	
	// Types of errors
	public static final int FATAL_ERROR = 1;
	public static final int ERROR = 2;
	public static final int WARNING = 3;
	public static final int EXCEPTION =  4;
	
	private int xmlErrorRow;
	private int xmlErrorCol;
	private int errorType;
	private ValidationRule errorSource = null;
	private String errorMessage = null;
	private Exception rootException = null;
	
	
	/**
	 * Constructor (for Validation Error resulting from some type of non-validation exception)
	 * 
	 * @param errorMsg Error message to present to the user (should be translated)
	 * @param source ValidationRule that was being executed when the exception occurred
	 * @param exception Exception that caused the error
	 */
	public XMLValidationError(String errorMsg, ValidationRule source, Exception exception) {
		errorMessage = errorMsg;
		xmlErrorRow = -1;
		xmlErrorCol = -1;	
		errorSource = source;
		rootException = exception;
		errorType = EXCEPTION;
	}
	
	/**
	 * Constructor (for Validation Error resulting from a Rule violation)
	 * 
	 * @param errorMsg Error message to present to the used (should be translated)
	 * @param errorRow Row of the validation error in the XML
	 * @param errorCol Column of the validation error in the XML
	 * @param source ValidationRule that resulted in the validation error
	 * @param severity One of FATAL_ERROR, ERROR or WARNING
	 */
	public XMLValidationError(String errorMsg, int errorRow, int errorCol, ValidationRule source, int severity) {
		errorMessage = errorMsg;
		xmlErrorRow = errorRow;
		xmlErrorCol = errorCol;	
		errorSource = source;
		errorType = severity;
	}
	
	/**
	 * Get the source rule that caused the validation error.
	 * 
	 * @return The source rule that caused the Exception
	 */
	public ValidationRule getErrorSource() {
		return (errorSource);
	}
	
	/**
	 * Get the type of the Error.
	 * Gives the severity of the error if it results from a rule violation (FATAL_ERROR, ERROR, WARNING) 
	 * or EXCEPTION if it results from an Exception.
	 * 
	 * @return Type of the error (FATAL_ERROR, ERROR, WARNING, EXCEPTION)
	 */
	public int getErrorType() {
		return (errorType);
	}
	
	/**
	 * Get the column in the XML file where the validation error occurred.
	 * 
	 * @return Column in XML file where validation error occurred.
	 */
	public int getXmlErrorCol() {
		return (xmlErrorCol);
	}
	
	/**
	 * Get the row in the XML file where the validation error occurred.
	 * 
	 * @return Row in XML file where validation error occurred.
	 */
	public int getXmlErrorRow() {
		return (xmlErrorRow);
	}
	
	/**
	 * Get the message associated with the error
	 * 
	 * @return The error message
	 */
	public String getErrorMessage() {
		return (errorMessage);
	}

	/**
	 * Get the Exception that caused this Error to occur.
	 * NOTE: This will be null if the XMLValidationError is the result of a rule violation instead of an exception.
	 * 
	 * @return The Exception that caused this error (or null if not cause by an exception)
	 */
	public Exception getRootException() {
		return (rootException);
	}
}
