/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.util.ConversionUtil;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Descriptor;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.w3c.dom.Document;

public abstract class DescriptorImpl extends SPIDataObject implements Descriptor {
	Document document = null;
	
	public DescriptorImpl(Document doc, SPISessionImpl session, String sddSchemaType) {
		super(doc.getDocumentElement(), session, sddSchemaType);
		
		document = doc;
	}
	
	public byte[] getDescriptorID() {
		byte[] descriptorID = null;		
		String idAttribute = getAttribute("descriptorID");
		
		if (idAttribute != null) {
			descriptorID = ConversionUtil.hexToBytes(idAttribute);
		}
		
		return descriptorID;
	}

	public Date getLastModified() {
		Date date = null;
		
		if (isSetAttribute("lastModified")) {
			String dateString = getAttribute("lastModified");
			DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ssZ");
			
			String timeZone = null;
			
			if (dateString.length() == 25) {
			    timeZone = dateString.substring(19, 22) + dateString.substring(23);
			}
			else if (dateString.endsWith("Z")) {
			    timeZone = "-0000";
			    dateString = dateString.substring(0, dateString.length() -1);
			}
			else {
			    DateFormat tzf = new SimpleDateFormat("Z");
			    timeZone = tzf.format(Calendar.getInstance().getTime());
			}
			
	        try {
                date = df.parse(dateString + timeZone);
            }
            catch (ParseException e) {
                // TODO: Do we need to throw an exception to the user?
            }
        }
	    
	    return date;
	}

	public String getSchemaVersion() {
		return getAttribute("schemaVersion");
	}

	public void setDescriptorID(byte[] descriptorID) {
		session.testParameter(descriptorID, 1, null);
		
		if (descriptorID != null && descriptorID.length != 16) {
			throw new InvalidInputException(1, InvalidInputException.INVALID_VALUE);
		}

		setAttribute("descriptorID", ConversionUtil.bytesToHex(descriptorID));
	}

	public void setLastModified(Date lastModified) {
		session.testParameter(lastModified, 1, null);
		
		String formattedDate  = null;
		
		if (lastModified != null) {
			DateFormat tzf = new SimpleDateFormat("Z");
			String timeZone = tzf.format(lastModified);

            DateFormat df = new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ss");
			formattedDate = df.format(lastModified) + timeZone.substring(0, 3) + ":" + timeZone.substring(3);
		}
		
		setAttribute("lastModified", formattedDate);
	}

	public void setSchemaVersion(String schemaVersion) {
		session.testParameter(schemaVersion, 1, null);

		setAttribute("schemaVersion", schemaVersion);
	}

	public String toString() {
		String description = new String("Descriptor ID: " + getDescriptorID() +
				" - Schema Version: " + getSchemaVersion() +
				" - LastModified: " + getLastModified() + "\n");
		
		return description;
	}
	
	public Document getOwnerDocument() {
		return document;
	}
}
