/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Properties;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.util.ConversionUtil;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Resource;
import org.w3c.dom.Element;

public class ResourceImpl extends DescribableObject implements Resource {
	private static final String RESOURCE_TYPE = "sdd-dd:ResourceType";
	private static final String PROPERTY_TYPE = "sdd-dd:PropertyType";
	
	private static String indent = new String("    ");

	public ResourceImpl(Element rdo, SPISessionImpl session) {
		super(rdo, session, RESOURCE_TYPE);		
	}
	
    public String getName() {
    	if (isSetElement("Name")) {
    		Element name = getChild("Name");
    		
    		return name.getTextContent();
    	}
    	else {
    		return null;
    	}
    }

    public String getResourceID() {
   		return getAttribute("id");
	}

	public String getResourceType() {
   		return getAttribute("type");
	}

	public Properties getProperties() {
    	Collection<Element> propertyElements = getChildren("Property");
    	
    	return ConversionUtil.getPropertiesFromPropertyElementList(propertyElements);
	}
	
	public Collection<Resource> getHostedResources() {
		Collection<Element> hostedResourceElements = getChildren("HostedResource");
		Collection<Resource> hostedResources = new ArrayList<Resource>();
		
			
		for (Element element : hostedResourceElements) {
			Resource resource = new ResourceImpl(element, session);
			hostedResources.add(resource);
		}
			
		return hostedResources;
	}
	
	public void setHostedResources(Collection<Resource> newHostedResources) {
        if (newHostedResources != null) {
            removeContent(getChildren("HostedResource", false));

            addContent(newHostedResources);
        }
        else {
        	unSetElement("HostedResource");
        }
	}

	public void setName(String name) {
        if (name == null) {
            unSetElement("Name");
        }
        else {
            Element nameElement = session.createDeploymentElement("Name");
            nameElement.setTextContent(name);
            setContent(nameElement);
        }
	}
	
	public void setResourceID(String resourceID) {
		session.testParameter(resourceID, 1, null);

		setAttribute("id", resourceID);
	}

	public void setResourceType(String resourceType) {
		session.testParameter(resourceType, 1, null);

		setAttribute("type", resourceType);
	}

	public void setProperties(Properties props) {
    	if (props != null) {
    		Collection<Element> properties = new ArrayList<Element>();
    		removeContent(getChildren("Property", false));
    		
    		for (Enumeration<Object> e = props.keys(); e.hasMoreElements(); ) {
    			String key = (String)e.nextElement();
    			String value = props.getProperty(key);
    			
    			Element propertyElement = session.createDeploymentElement("Property");
    			Element propertyNameElement = session.createDeploymentElement("PropertyName");
    			Element valueElement = session.createDeploymentElement("Value");
    			
    			propertyNameElement.setTextContent(key);
    			valueElement.setTextContent(value);
    			session.insertNode(propertyElement, propertyNameElement, PROPERTY_TYPE);
    			session.insertNode(propertyElement, valueElement, PROPERTY_TYPE);
    			
    			properties.add(propertyElement);
    		}
    		
			addContent(properties);
    	}
    	else {
    		unSetElement("Property");
    	}
	}
	
	public Object clone() {
		return new ResourceImpl((Element)getElement().cloneNode(true), session);
	}
	
	String toString(String offset) {
		String resourceContents = new String(offset + "Hosted Resource - ID: " + getResourceID() + 
				" , Type: "	+ getResourceType() + "\n");
		
		Properties resourceProperties = getProperties();
		for (Object key : resourceProperties.keySet()) {
		    resourceContents += offset + " Resource Property - Key = \"" + key + "\" , value = \""
		    + resourceProperties.getProperty((String) key) + "\"\n";
        }
		
		Collection<Resource> hostedResources = getHostedResources();
		for (Resource hostedResource : hostedResources) {
		    resourceContents += ((ResourceImpl) hostedResource).toString(indent + offset);
        }

		return resourceContents;
	}
	
	public String toString() {
		String resourceContents = new String("Resource - ID: " + getResourceID() + " , Type: "
				+ getResourceType() + "\n");
		
		Properties resourceProperties = getProperties();
        for (Object key : resourceProperties.keySet()) {
            resourceContents += " Resource Property - Key = \"" + key + "\" , value = \""
            + resourceProperties.getProperty((String) key) + "\"\n";
        }
        
        Collection<Resource> hostedResources = getHostedResources();
        for (Resource hostedResource : hostedResources) {
            resourceContents += ((ResourceImpl)hostedResource).toString(indent);
        }

		return resourceContents;
	}
}
