/*******************************************************************************
 * Copyright (c) 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.SAXException;

public class SDDTypeResolver {
	private static final String ddFileName = "sdd-deploymentDescriptor-1.0.xsd";
	private static final String pdFileName = "sdd-packageDescriptor-1.0.xsd";
	private static final String commonFileName = "sdd-common-1.0.xsd";
	private static final String xmldsigFileName = "xmldsig-core-schema.xsd";
	
	private static final String ddPrefix = "sdd-dd:";
	private static final String pdPrefix = "sdd-pd:";
	private static final String commonPrefix = "sdd-common:";
	private static final String xmldsigPrefix = "ds:";
	
	private Document ddDocument = null;
	private Document pdDocument = null;
	private Document commonDocument = null;
	private Document xmldsigDocument = null;
	
	public SDDTypeResolver() throws FileNotFoundException, IOException, SAXException {
		// Load SDD Schema files into Documents
		ddDocument = loadDocumentFromSchema(ddFileName);
		pdDocument = loadDocumentFromSchema(pdFileName);
		commonDocument = loadDocumentFromSchema(commonFileName);
		xmldsigDocument = loadDocumentFromSchema(xmldsigFileName);
	}
	
	public Collection<Element> resolveSDDType(String sddTypeName) {
		Element rootElement = null;
		
		// Find the appropriate Schema for the given type
		if (sddTypeName.startsWith(ddPrefix)) {
			rootElement = ddDocument.getDocumentElement();
		}
		else if (sddTypeName.startsWith(pdPrefix)) {
			rootElement = pdDocument.getDocumentElement();
		}
		else if (sddTypeName.startsWith(commonPrefix)) {
			rootElement = commonDocument.getDocumentElement();
		}
		else if (sddTypeName.startsWith(xmldsigPrefix)) {
			rootElement = xmldsigDocument.getDocumentElement();
		}
		
		// Give up if the type didn't have a valid prefix
		if (rootElement == null) {
			return null;
		}
		
		// Get the Element for the given type from the Schema
		Element typeElement = findComplexType(rootElement, stripPrefix(sddTypeName));
		
		// Give up if they type wasn't found in the Schema
		if (typeElement == null) {
			return null;
		}
		
		// Get a list of the type element's children (including expanding any groups)
		Collection<Element> childElements = findChildElements(typeElement);
		
		return childElements;
	}
	
	public String stripPrefix(String typeName) {
		int index = typeName.indexOf(':');
		
		if (index != -1) {
			return typeName.substring(index + 1);
		}
		else {
			return typeName;
		}
	}
	
	/*
	 * Load a Schema file into a DOM Document
	 */
	private Document loadDocumentFromSchema(String schemaFileName)
			throws FileNotFoundException, IOException, SAXException {
		// Load the Schema file from inside the jar file
		InputStream inStream = loadSchemaFromJar(schemaFileName);
		
		// Create a DocumentBuilder and parse the InputStream into a Document
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = null;
		try {
			builder = factory.newDocumentBuilder();
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		}
		Document xmlDocument = builder.parse(inStream);

		removeWhitespaceNodes(xmlDocument.getDocumentElement());
		
		return xmlDocument;		
	}
	
	/*
	 * Remove all whitespace-only Text nodes from a Document Element
	 */
	private void removeWhitespaceNodes(Element e) {
		NodeList children = e.getChildNodes();
		
		for (int i = children.getLength() - 1; i >= 0; i--) {
			Node child = children.item(i);
			
			if (child instanceof Text && ((Text)child).getData().trim().length() == 0) {
				e.removeChild(child);
			}
			else if (child instanceof Element) {
				removeWhitespaceNodes((Element)child);
			}
		}
	}
	
	/*
	 * Find the <xsd:complexType> element that matches the given type name
	 */
	private Element findComplexType(Element rootElement, String typeName) {
		// Check the given element
		if (rootElement.getNodeName().equals("xsd:complexType")) {
			String elementType = rootElement.getAttribute("name");
			
			// Does this element's name attribute match the desired type?
			if (elementType != null && typeName.equals(elementType)) {
				return rootElement;
			}
		}
		
		// Look through this Element's children
		NodeList children = rootElement.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			if (children.item(i) instanceof Element) {
				Element element = findComplexType((Element)children.item(i), typeName);
				
				// If the element in question was found, return it
				if (element != null) {
					return element;
				}
			}
		}
		
		// Couldn't find the given element
		return null;
	}
	
	/*
	 * Find the <xsd:element> Nodes under the given Element (including those in referenced groups)
	 */
	private Collection<Element> findChildElements(Element rootElement) {
		Collection<Element> elements = new ArrayList<Element>();
		
		// Is this a <xsd:group> element?
		if (rootElement.getNodeName().equals("xsd:group")) {
			Collection<Element> groupElements = expandGroupReference(rootElement);
			
			// Add the elements from the group
			if (groupElements != null) {
				elements.addAll(groupElements);
			}
		}			
		
		// Check the given element
		if (rootElement.getNodeName().equals("xsd:element")) {
			elements.add(rootElement);
		}
		
		// Look through this element's children
		NodeList children = rootElement.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			if (children.item(i) instanceof Element) {
				elements.addAll(findChildElements((Element)children.item(i)));
			}
		}
		
		// Return the list of elements
		return elements;		
	}
	
	/*
	 * Expand the given group reference element into a list of the group's contents
	 */
	private Collection<Element> expandGroupReference(Element refElement) {
		Element docElement = null;
		
		if (refElement.getAttribute("ref") != null) {
			if (refElement.getAttribute("ref").startsWith(ddPrefix)) {
				docElement = ddDocument.getDocumentElement();
			}
			else if (refElement.getAttribute("ref").startsWith(pdPrefix)) {
				docElement = pdDocument.getDocumentElement();
			}
			else if (refElement.getAttribute("ref").startsWith(commonPrefix)) {
				docElement = commonDocument.getDocumentElement();
			}
			else if (refElement.getAttribute("ref").startsWith(xmldsigPrefix)) {
				docElement = xmldsigDocument.getDocumentElement();
			}
			else {
				// The referenced group isn't in one of the SDD Schema
				return null;
			}
		}
		else {
			// The given root element is not a group reference element
			return null;
		}
		
		// Find the referenced group
		String groupName = refElement.getAttribute("ref");
		groupName = groupName.substring(groupName.indexOf(':') + 1);
		Element groupElement = findGroup(docElement, groupName);
		
		// Return all the <xsd:element> nodes under the group element
		return findChildElements(groupElement);
	}
	
	/*
	 * Find the group referenced by the refElement somewhere under the rootElement
	 */
	private Element findGroup(Element rootElement, String groupName) {
		// Check the given element
		if (rootElement.getNodeName().equals("xsd:group")) {
			String elementName = rootElement.getAttribute("name");
			
			// Does this element's name attribute match the desired group name?
			if (elementName != null && elementName.equals(groupName)) {
				return rootElement;
			}
		}
		
		// Look through this Element's children
		NodeList children = rootElement.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			if (children.item(i) instanceof Element) {
				Element element = findGroup((Element)children.item(i), groupName);
				
				// If the group in question was found, return it
				if (element != null) {
					return element;
				}
			}
		}
		
		// Couldn't find the given group
		return null;
	}
	
	/*
	 * Load a Schema file from inside the jar, or from the file system if not in the jar
	 */
	private InputStream loadSchemaFromJar(String schemaToLoad) throws FileNotFoundException {
		// Try to load the file from inside the jar file
		InputStream in = SDDTypeResolver.class.getResourceAsStream("/sdd_schema/" + schemaToLoad); //$NON-NLS-1$
        
		// If the jar is missing, load from the file system
		if (in == null) {
			in = new FileInputStream(new File("sdd_schema/" + schemaToLoad)); //$NON-NLS-1$
		}
		
		return in;
	}
}
