/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi;


import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact.ArtifactFactoryImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.condition.ConditionFactoryImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit.ContentUnitFactoryImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.requiredbase.RequiredBaseFactoryImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.requirement.RequirementFactoryImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.util.SchemaTypePattern;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.variable.VariableFactoryImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.BaseFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Descriptor;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.ArtifactFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.ConditionFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentUnitFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requiredbase.RequiredBaseFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.RequirementFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.VariableFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class SPISessionImpl implements SPISession {
	
	private static SPISessionImpl defaultSession = null;
	private Document ddDocument = null;
	private Document pdDocument = null;
	private Document commonDocument = null;
	private Document xmlDsigDocument = null;
	
	private DocumentBuilder builder = null;
	
	private boolean performHardErrorChecking = true;
	
	private boolean useCopies = true;
	
	private final Logger log;
	
	private SDDTypeResolver sddTypeResolver = null;
	
	
	public static SPISessionImpl getDefaultSession() {
		if (defaultSession == null)
			defaultSession = new SPISessionImpl();
		
		return defaultSession;
	}
	
	SPISessionImpl() {
		log = Logger.getLogger("org.eclipse.cosmos.me.deployment.sdd.common.spi.default");
		log.setUseParentHandlers(false);
		
		try {
			builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
		} catch (ParserConfigurationException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		ddDocument = builder.newDocument();
		pdDocument = builder.newDocument();
		commonDocument = builder.newDocument();
		xmlDsigDocument = builder.newDocument();
		
		try {
			sddTypeResolver = new SDDTypeResolver();
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (SAXException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	public ArtifactFactory createArtifactFactory() {
		return new ArtifactFactoryImpl(this);
	}

	public BaseFactory createBaseFactory() {
		return new BaseFactoryImpl(this);
	}

	public ConditionFactory createConditionFactory() {
		return new ConditionFactoryImpl(this);
	}

	public ContentUnitFactory createContentUnitFactory() {
		return new ContentUnitFactoryImpl(this);
	}

	public RequiredBaseFactory createRequiredBaseFactory() {
		return new RequiredBaseFactoryImpl(this);
	}

	public RequirementFactory createRequirementFactory() {
		return new RequirementFactoryImpl(this);
	}

	public VariableFactory createVariableFactory() {
		return new VariableFactoryImpl(this);
	}
	
	public Logger getLogger() {
		return log;
	}
	
	// Methods not included in the SPISession interface

	// Not at all recommended for use, but if it proves to be absolutely necessary... it's here.
    public Document getDdDocument() {
        return ddDocument;
    }

    public Document getPdDocument() {
        return pdDocument;
    }
    
    public Document getCommonDocument() {
        return commonDocument;
    }
    
	public void setHardErrorChecking(boolean hardErrorChecking) {
		performHardErrorChecking = hardErrorChecking;
	}
	
	public boolean isErrorCheckingEnabled() {
		return performHardErrorChecking;
	}
	
	public boolean isUseCopy()
	{
	    return useCopies;
	}
	
	public void setUseCopy(boolean copies)
	{
	    useCopies = copies;
	}
	
	public void testParameter(Object param, int paramNumber, SchemaTypePattern pattern) {
		boolean isValid = true;
		Level logLevel = performHardErrorChecking ? Level.SEVERE : Level.WARNING;
		InvalidInputException e = null;
		
		if (param == null) {
			e = new InvalidInputException(paramNumber,InvalidInputException.NULL_ARGUMENT);
			isValid = false;
			log.log(logLevel, e.toString(), e);
		}
		
		// This test needs to be this way, so that a valid byte[] isn't checked
		// by the next else-if clause, which will generate an error by trying to
		// cast byte[] to Object[].
		else if (param instanceof byte[]) {
			if (((byte[])param).length == 0) {
				e = new InvalidInputException(paramNumber,InvalidInputException.EMPTY_ARRAY);
				isValid = false;
				log.log(logLevel, e.toString(), e);
			}
		}
		else if (param.getClass().isArray() && (((Object[])param).length == 0)) {
			e = new InvalidInputException(paramNumber,InvalidInputException.EMPTY_ARRAY);
			isValid = false;
			log.log(logLevel, e.toString(), e);
		}
		else if (param instanceof Collection && ((Collection)param).isEmpty()) {
            e = new InvalidInputException(paramNumber,InvalidInputException.EMPTY_LIST);
            isValid = false;
            log.log(logLevel, e.toString(), e);
		}
			
		if (performHardErrorChecking && ! isValid) {
			throw e;
		}
	}
	
	public void testIntParameter(int param, int paramNumber, int[] validValues) {
		boolean isValid = false;
		InvalidInputException e = null;

		for(int i = 0; (i < validValues.length) && (! isValid) ; i++) {
			if (param == validValues[i])
				isValid = true;
		}
		
		if (! isValid) {
			e = new InvalidInputException(
					paramNumber,InvalidInputException.INVALID_VALUE);
			handleInputException(e);
		}
	}
	
	/**
	 * 
	 * @param param
	 * @param paramNumber
	 * @param minSize
	 * @param maxSize The max size of the array. If there is no max size, use a negative number.
	 */
	public void testCollectionParameter(Collection param, int paramNumber, int minSize, int maxSize) {
	    boolean isValid = false;
	    InvalidInputException e = null;

	    if (param != null && (!(maxSize > -1) || param.size() <= maxSize) && param.size() >= minSize) {
	        isValid = true;
	    }
	    
	    if (performHardErrorChecking && ! isValid) {
            e = new InvalidInputException(paramNumber, InvalidInputException.INVALID_VALUE);
            handleInputException(e);
	    }
	}
	
	public void handleInputException(InvalidInputException e) {
		Level logLevel = performHardErrorChecking ? Level.SEVERE : Level.WARNING;

		log.log(logLevel, e.toString(), e);
		
		if (performHardErrorChecking) {
			throw e;
		}
	}

	public Element createDeploymentElement(String tagName) {
	    Element e = ddDocument.createElementNS("http://docs.oasis-open.org/sdd/ns/deploymentDescriptor",
	    		"sdd-dd:" + tagName);
		return e;
	}

	public Element createPackageElement(String tagName) {
		Element e = pdDocument.createElementNS("http://docs.oasis-open.org/sdd/ns/packageDescriptor",
				"sdd-pd:" + tagName);
		return e;
	}

    public Element createCommonElement(String tagName) {
        Element e = commonDocument.createElementNS("http://docs.oasis-open.org/sdd/ns/common", "sdd-common:" + tagName);
        return e;
    }
	
	public Element createXmlDsigElement(String tagName) {
		Element e = xmlDsigDocument.createElementNS("http://www.w3.org/2000/09/xmldsig#", "ds:" + tagName);
		return e;
	}
	
	public void insertNode(Node parent, Node node, String parentSchemaType) {
	    if (parent instanceof Document) {
	        ((Document) parent).adoptNode(node);
	        parent.appendChild(node);
	    }
	    else {
	        parent.getOwnerDocument().adoptNode(node);
		    
		    Collection<Element> allowedChildNodes = sddTypeResolver.resolveSDDType(parentSchemaType);
		    Collection<Element> children = getChildren(parent);
		    String nodeName = sddTypeResolver.stripPrefix(node.getNodeName());
		    
		    boolean foundNode = false;
		    // Look for the given node name in the list of children
		    Iterator<Element> allowedIter = allowedChildNodes.iterator();
		    while (allowedIter.hasNext()) {
		    	Element nextAllowedElement = allowedIter.next();
		    	if (nextAllowedElement.getAttribute("name").equals(nodeName)) {
		    		// Found the element to insert, look for where to insert it
		    		foundNode = true;
		    		if (allowedIter.hasNext()) {
		    			boolean insertedNode = false;
		    			// Element goes in the middle somewhere, find out where
		    			while (allowedIter.hasNext() && !insertedNode) {
		    				nextAllowedElement = allowedIter.next();
		    				
		    				// Check if this allowed node is a child of the parent
		    				Iterator<Element> childrenIter = children.iterator();
		    				while (childrenIter.hasNext() && !insertedNode) {
		    					Element child = childrenIter.next();
		    					String childName = sddTypeResolver.stripPrefix(child.getNodeName());
		    					if (nextAllowedElement.getAttribute("name").equals(childName)) {
		    						// Found the node in question, so insert before it
		    						insertedNode = true;
		    						parent.insertBefore(node, child);
		    					}
		    				}
		    			}
		    			
		    			if (!insertedNode) {
		    				// Didn't find a place to insert the node, so append it
		    				parent.appendChild(node);
		    			}
		    		}
		    		else {
		    			// Element goes at the end, append it
		    			parent.appendChild(node);
		    		}
		    	}
		    }
		    
		    // Didn't find the given node name in the list of children, append it to the end and log a warning
		    if (!foundNode) {
		    	parent.appendChild(node);
		    	log.warning("Adding element " + node.getNodeName() + " as a child of " + parent.getNodeName() +
		    			", but " + node.getNodeName() + " is not in the list of allowed children.");
		    }
	    }
	}
	
	public void insertNode(Node parent, SPIDataObject node, String parentSchemaType) {
	    insertNode(parent, node.getElement(), parentSchemaType);
	}
	
	public void insertNode(SPIDataObject parent, Node node, String parentSchemaType) {
	    insertNode(parent.getElement(), node, parentSchemaType);
	}
	
	public void insertNode(SPIDataObject parent, SPIDataObject node, String parentSchemaType) {
	    insertNode(parent.getElement(), node.getElement(), parentSchemaType);
	}
	
	public Collection<Element> getChildren(Node parent, String nodeName) {
	    Collection<Element> nodes = new ArrayList<Element>();
	    
	    Collection<Element> nodeList = getChildren(parent);
	    
	    for (Element element : nodeList) {
            if ((element.getLocalName() != null && (element.getLocalName().equals(nodeName) ||
            		nodeName.endsWith(":" + element.getLocalName()))) || element.getNodeName().equals(nodeName) ||
            		element.getNodeName().endsWith(":" + nodeName)) {
                nodes.add(element);
            }
        }
	    
	    return nodes;
	}
	
	public Collection<Element> getChildren(Node parent) {
	    Collection<Element> nodes = new ArrayList<Element>();
	    
	    NodeList nodeList = parent.getChildNodes();
	    
	    for (int i = 0; i < nodeList.getLength(); i++) {
	        Node node = nodeList.item(i);
	        
	        if (node instanceof Element) {
	            nodes.add((Element) node);
	        }
	    }
	    
	    return nodes;
	}
	
	public Element getChild(Node parent, String nodeName) {
	    Element element = null;
	    Collection<Element> nodes = getChildren(parent, nodeName);
	    
	    if (!nodes.isEmpty()) {
	        element = nodes.iterator().next();
	    }
	    
	    return element;
	}
	
	public Element getChild(Node parent) {
        Collection<Element> nodes = getChildren(parent);
        return nodes.iterator().next();
	}
	
	public void removeChildren(Node parent, String nodeName) {
		Collection<Element> nodes = getChildren(parent, nodeName);
		
		for (Element e : nodes) {
			parent.removeChild(e);
		}
	}
	
	public Descriptor buildDescriptorFromDocument(Document doc) throws InvalidInputException {
		log.entering(getClass().getName(), "getDescriptor");
		
		this.testParameter(doc, 1, null);
		
		Descriptor d = null;		
		if (doc.getDocumentElement().getNodeName().equals("sdd-dd:DeploymentDescriptor")) {
			d = new DeploymentDescriptorImpl(doc, this);
		}
		else if (doc.getDocumentElement().getNodeName().equals("sdd-pd:PackageDescriptor")) {
			d = new PackageDescriptorImpl(doc, this);
		}
		else {
			throw new InvalidInputException(1, InvalidInputException.INVALID_VALUE);
		}
		
		log.exiting(getClass().getName(), "getDescriptor");
		
		return d;
	}
	
	public DocumentBuilder getDocumentBuilder() {
		return builder;
	}
}