/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.AdditionalContent;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.Artifact;
import org.w3c.dom.Element;


public class ArtifactImpl extends SPIDataObject implements Artifact {
	private static final String ARTIFACT_TYPE = "sdd-dd:ArtifactType";
	private static final String ARGUMENT_LIST_TYPE = "sdd-dd:ArgumentListType";
	private static final String OUTPUT_VARIABLE_LIST_TYPE = "sdd-dd:OutputVariableListType";
        
	public ArtifactImpl(Element artifactObject, SPISessionImpl session) {
        super(artifactObject, session, ARTIFACT_TYPE);
    }
    
    public String getContentReference() {
    	return isSetAttribute("contentRef") ? getAttribute("contentRef") : null;
    }

    public String getPackageType() {
    	return isSetAttribute("type") ? getAttribute("type") : null;
    }

    public Collection<AdditionalContent> getAdditionalContents() {
    	Collection<Element> additionalContentList = getChildren("AdditionalContent");
		Collection<AdditionalContent> crcList = new ArrayList<AdditionalContent>();
		
		for (Element element : additionalContentList) {
			AdditionalContent temp = new AdditionalContentImpl(element, session);
			crcList.add(temp);
		}
		
		return crcList;
    }

    public BigInteger getWeight() {
    	return isSetAttribute("weight") ? new BigInteger(getAttribute("weight")) : null;
    }

    public Collection<ArtifactArgument> getArguments() {
    	Collection<ArtifactArgument> arguments = new ArrayList<ArtifactArgument>();
    	
        if (isSetElement("Arguments")) {
        	Element argumentsElement = getChild("Arguments");
            
        	Collection<Element> list = session.getChildren(argumentsElement, "Argument");
        	for (Element arg : list) {
                String name = arg.getAttribute("name");
                String value = arg.getAttribute("value");
                String required = arg.getAttribute("required");

                ArtifactArgument artarg = new ArtifactArgument(name.length()>0 ? name : null,
                		value.length()>0 ? value : null, required.length()>0 ? Boolean.valueOf(required) : null);
                arguments.add(artarg);
            }
        }
        
        return arguments;
    }

    public Collection<ArtifactOutputVariable> getOutputVariables() {
    	Collection<ArtifactOutputVariable> outputVariables = new ArrayList<ArtifactOutputVariable>();
        
        if (isSetElement("OutputVariables")) {
        	Element ovlElement = getChild("OutputVariables");
        	
        	Collection<Element> list = session.getChildren(ovlElement, "OutputVariable");

        	for (Element arg : list) {
        		String id = arg.getAttribute("id");
        		String outputParameterName = arg.getAttribute("outputParameterName");
        		
        		ArtifactOutputVariable artOV = new ArtifactOutputVariable(id.length()>0 ? id : null, outputParameterName.length()>0 ? outputParameterName : null);
        		outputVariables.add(artOV);
        	}
        }
        
        return outputVariables;
    }

    public String getResourceReference() {
    	return isSetAttribute("resourceRef") ? getAttribute("resourceRef") : null;
    }

    public void setPackageType(String packageType) {
        // TODO Handle bad input
    	if (packageType == null) {
			unSetAttribute("type");
    	}
		else {
			setAttribute("type", packageType);
		}
    }

    public void setContentReference(String packageContentReference) {
        // TODO Handle bad input
    	if (packageContentReference == null) {
			unSetAttribute("contentRef");
    	}
		else {
			setAttribute("contentRef", packageContentReference);
		}
    }
    
    public void setAdditionalContents(Collection<AdditionalContent> newAdditionalContents) {
        
		if (newAdditionalContents != null) {
            removeContent(getChildren("AdditionalContent", false));

            addContent(newAdditionalContents);
        }
        else {
        	unSetElement("AdditionalContent");
        }
    }

    public void setWeight(BigInteger weight) {
    	if (weight == null) {
			unSetAttribute("weight");
    	}
		else {
			setAttribute("weight", weight.toString());
		}
    }

    public void setArguments(Collection<ArtifactArgument> arguments) {
        if (arguments == null || arguments.isEmpty()) {
            unSetElement("Arguments");
        }
        else {
            Element argumentsElement = session.createDeploymentElement("Arguments");
            for (ArtifactArgument arg : arguments) {
            	Element argElement = session.createDeploymentElement("Argument");
            	argElement.setAttribute("name" , arg.getName());
            	argElement.setAttribute("value" , arg.getValue());
            	argElement.setAttribute("required" , arg.getRequired().toString());
            	session.insertNode(argumentsElement, argElement, ARGUMENT_LIST_TYPE);
            }
                
            setContent(argumentsElement);
        }
    }

    public void setOutputVariables(Collection<ArtifactOutputVariable> outputVariables) {
        if (outputVariables == null || outputVariables.isEmpty()) {
            unSetElement("OutputVariables");
        }
        else {
        	Element outputVariablesElement = session.createDeploymentElement("OutputVariables");
            for (ArtifactOutputVariable artOV : outputVariables) {
            	Element outputV = session.createDeploymentElement("OutputVariable");
            	outputV.setAttribute("id" , artOV.getOutputVariableID());
            	outputV.setAttribute("outputParameterName" , artOV.getArtifactParameter());
            	
            	session.insertNode(outputVariablesElement, outputV, OUTPUT_VARIABLE_LIST_TYPE);
            }
                
            setContent(outputVariablesElement);
        }
    }

    public void setResourceReference(String resourceReference) {
    	if (resourceReference == null) {
			unSetAttribute("resourceRef");
    	}
		else {
			setAttribute("resourceRef", resourceReference);
		}
    }

    public String toString() {
        String tmp = new String("Content Ref: \"" + getAttribute("contentRef") + "\"\n");

        tmp += "Resource Ref: \"" + getAttribute("resourceRef") + "\"\n";
        
        return tmp;
    }
}
