/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact.ArtifactImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.requirement.RequirementImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.Artifact;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Completion;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ConfigurationUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ResultingChange;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;
import org.w3c.dom.Element;

public class ConfigurationUnitImpl extends ContentUnitImpl implements ConfigurationUnit {
	private static final String CONFIGURATION_UNIT_TYPE = "sdd-dd:ConfigurationUnitType";
	private static final String CONFIGURATION_ARTIFACTS_TYPE = "sdd-dd:ConfigurationArtifactsType";
	private static final String REQUIREMENTS_TYPE = "sdd-dd:RequirementsType";
	
	public ConfigurationUnitImpl(Element data, SPISessionImpl session) {
		super(data, session, CONFIGURATION_UNIT_TYPE);
	}

	public Artifact getConfigurationArtifact() {
		Artifact configArtifact = null;
        
        if (isSetElement("Artifacts")) {
        	Collection<Element> configArtifactList = session.getChildren(getChild("Artifacts"), "ConfigArtifact");
        	if (!configArtifactList.isEmpty()) {
        		Element configArtifactElement = configArtifactList.iterator().next();
        		configArtifact = new ArtifactImpl(configArtifactElement, session);
        	}
        }

        return configArtifact;
	}
	
	public Collection<Completion> getCompletion() {
		Collection<Element> completionElements = getChildren("Completion");
        Collection<Completion> completions = new ArrayList<Completion>();

        for (Element element : completionElements) {
            completions.add(new CompletionImpl(element, session));
        }

        return completions;
	}

	public Collection<Requirement> getRequirements() {
		Collection<Requirement> requirements = new ArrayList<Requirement>();

        if (isSetElement("Requirements")) {
            Collection<Element> requirementElementList = session.getChildren(getChild("Requirements"), "Requirement");

            for (Element element : requirementElementList) {
                requirements.add(new RequirementImpl(element, session));
            }
        }

        return requirements;
	}

	public Collection<ResultingChange> getResultingChanges() {
        Collection<Element> resultingChangeElements = getChildren("ResultingChange");
        Collection<ResultingChange> resultingChanges = new ArrayList<ResultingChange>();

        for (Element element : resultingChangeElements) {
            resultingChanges.add(new ResultingChangeImpl(element, session));
        }

        return resultingChanges;
	}

	public String getTargetResourceReference() {
		return getAttribute("targetResourceRef");
	}

	public void setCompletion(Collection<Completion> configurationUnitCompletion) {
		 if (configurationUnitCompletion != null && !configurationUnitCompletion.isEmpty()) {
             //Remove all Completion elements
             removeContent(getChildren("Completion", false));

             addContent(configurationUnitCompletion);
         }
         else {
             //Remove all Completion elements
             removeContent(getChildren("Completion", false));
         }
	}

	public void setConfigurationArtifact(Artifact configurationArtifact) {
		session.testParameter(configurationArtifact, 1, null);

		Element cat = getChild("Artifacts");
		if (configurationArtifact == null) {
		    Element artifact = session.getChild(cat, "ConfigArtifact");
			if (artifact != null) {
				cat.removeChild(artifact);
            }
        }
        else {
            session.insertNode(cat, (SPIDataObject) configurationArtifact, CONFIGURATION_ARTIFACTS_TYPE);
            setContent(cat);
        }
	}

	public void setRequirements(Collection<Requirement> newRequirements) {
		if (newRequirements != null && !newRequirements.isEmpty()) {
            Element requirementsDataObject;

            if (isSetElement("Requirements")) {
                requirementsDataObject = getChild("Requirements");
            }
            else {
                requirementsDataObject = session.createDeploymentElement("Requirements");
            }

            //Remove all Requirement elements
            Collection<Element> requirementNodeList = session.getChildren(requirementsDataObject, "Requirement");

            for (Element element : requirementNodeList) {
                requirementsDataObject.removeChild(element);
            }
            
            for (Requirement requirement : newRequirements) {
                session.insertNode(requirementsDataObject, (SPIDataObject) requirement, REQUIREMENTS_TYPE);
            }

            setContent(requirementsDataObject);
        }
        else {
            unSetElement("Requirements");
        }
	}

	public void setResultingChanges(Collection<ResultingChange> resultingChanges) {
    	if (resultingChanges != null && !resultingChanges.isEmpty()) {
            //Remove all ResultingResource elements
            removeContent(getChildren("ResultingResource", false));

            addContent(resultingChanges);
        }
        else {
            //Remove all ResultingResource elements
            removeContent(getChildren("ResultingResource", false));
        }
	}

	public void setTargetResourceReference(String targetResourceReference) {
		session.testParameter(targetResourceReference, 1, null);
		
		if (targetResourceReference == null) {
			unSetAttribute("targetResourceRef");
		}
		else {
			setAttribute("targetResourceRef", targetResourceReference);
		}
	}	
}