/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.artifact.ArtifactImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.requiredbase.RequiredBaseImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.requirement.RequirementImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.Artifact;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.InstallUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ResultingResource;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.exception.InvalidInputException;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requiredbase.RequiredBase;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;
import org.w3c.dom.Element;

public abstract class InstallUnitImpl extends ContentUnitImpl implements InstallUnit {
	private static final String INSTALLATION_ARTIFACTS_TYPE = "sdd-dd:InstallationArtifactsType";
	private static final String REQUIREMENTS_TYPE = "sdd-dd:RequirementsType";
    
    public InstallUnitImpl(Element data, SPISessionImpl session, String sddSchemaType) {
        super(data, session, sddSchemaType);
    }
    
    public Artifact getInstallArtifact() {
        Artifact installArtifact = null;
        
        if (isSetElement("Artifacts")) {
            Element installArtifactElement = session.getChild(getChild("Artifacts"), "InstallArtifact");
        	if (installArtifactElement != null) {
        		installArtifact = new ArtifactImpl(installArtifactElement, session);
        	}
        }

        return installArtifact;    
    }

    public String getTargetResourceReference() {
        return getAttribute("targetResourceRef");
    }

    public Collection<ResultingResource> getResultingResources() {
    	Collection<Element> resultingResourceElements = getChildren("ResultingResource");
        Collection<ResultingResource> resultingResources = new ArrayList<ResultingResource>();

        for (Element element : resultingResourceElements) {
            resultingResources.add(new ResultingResourceImpl(element, session));
        }

        return resultingResources;        
    }

    public Artifact getUpdateArtifact() {
        Artifact updateArtifact = null;
        
        if (isSetElement("Artifacts")) {
            Element artifactElement = session.getChild(getChild("Artifacts"), "UpdateArtifact");
        	if (artifactElement != null) {
        		updateArtifact = new ArtifactImpl(artifactElement, session);
        	}
        }
        
        return updateArtifact;
    }

    public Artifact getUndoArtifact() {
        Artifact undoArtifact = null;
        if (isSetElement("Artifacts")) {
            Element artifactElement = session.getChild(getChild("Artifacts"), "UndoArtifact");
        	if (artifactElement != null) {
        		undoArtifact = new ArtifactImpl(artifactElement, session);
        	}
        }

        return undoArtifact;
    }

    public Artifact getUninstallArtifact() {
        Artifact uninstallArtifact = null;
        if (isSetElement("Artifacts")) {
            Element artifactElement = session.getChild(getChild("Artifacts"), "UninstallArtifact");
        	if (artifactElement != null) {
        		uninstallArtifact = new ArtifactImpl(artifactElement, session);
        	}
        }

        return uninstallArtifact;
    }
    
    public void setResultingResources(Collection<ResultingResource> newResultingResources) {
    	removeContent(getChildren("ResultingResource", false));
    	
    	if (newResultingResources != null && !newResultingResources.isEmpty()) {
            addContent(newResultingResources);
        }
    }

    public void setInstallArtifact(Artifact installArtifact) {
    	Element iat = getChild("Artifacts");
    	if (iat == null) {
    		iat = session.createDeploymentElement("Artifacts");
    	}
    	if (installArtifact == null) {
    		// Special case, handle attempts to remove the only artifact
            // currently contained by the install unit
            if (session.getChildren(iat, "InstallArtifact") != null &&
            		session.getChildren(iat, "UpdateArtifact") == null &&
            		session.getChildren(iat, "UndoArtifact") == null &&
            		session.getChildren(iat, "UninstallArtifact") == null) {
                InvalidInputException e = new InvalidInputException(1, InvalidInputException.NULL_ARGUMENT);
                session.handleInputException(e);
            }
            
            Element artifact = session.getChild(iat, "InstallArtifact");
            if (artifact != null) {
            	iat.removeChild(artifact);
            }
        }
        else {
            Element artifact = session.getChild(iat, "InstallArtifact");
            if (artifact != null) {
                iat.removeChild(artifact);
            }
        	
        	session.insertNode(iat, (SPIDataObject) installArtifact, INSTALLATION_ARTIFACTS_TYPE);
        }
    	
    	setContent(iat);
    }

    public void setTargetResourceReference(String targetResourceReference) {
        session.testParameter(targetResourceReference, 1, null);

        setAttribute("targetResourceRef", targetResourceReference);
    }

    public void setUpdateArtifact(Artifact updateArtifact) {
    	Element uat = getChild("Artifacts");
    	if (uat == null) {
    		uat = session.createDeploymentElement("Artifacts");
    	}
    	if (updateArtifact == null) {
    		// Special case, handle attempts to remove the only artifact
            // currently contained by the install unit
            if (session.getChildren(uat, "UpdateArtifact") != null &&
            		session.getChildren(uat, "InstallArtifact") == null &&
                    session.getChildren(uat, "UndoArtifact") == null &&
                    session.getChildren(uat, "UninstallArtifact") == null) {
                InvalidInputException e = new InvalidInputException(1, InvalidInputException.NULL_ARGUMENT);
                session.handleInputException(e);
            }

            Element artifact = session.getChild(uat, "UpdateArtifact");
            if (artifact != null) {
                uat.removeChild(artifact);
            }
        }
        else {
            Element artifact = session.getChild(uat, "UpdateArtifact");
            if (artifact != null) {
                uat.removeChild(artifact);
            }
        	
        	session.insertNode(uat, (SPIDataObject) updateArtifact, INSTALLATION_ARTIFACTS_TYPE);
            setContent(uat);
        }
    }
    
    public void setUndoArtifact(Artifact undoArtifact) {
    	Element uat = getChild("Artifacts");
    	if (uat == null) {
    		uat = session.createDeploymentElement("Artifacts");
    	}
    	if (undoArtifact == null) {
    		// Special case, handle attempts to remove the only artifact
            // currently contained by the install unit
            if (session.getChildren(uat, "UndoArtifact") != null &&
            		session.getChildren(uat, "InstallArtifact") == null &&
                    session.getChildren(uat, "UpdateArtifact") == null &&
                    session.getChildren(uat, "UninstallArtifact") == null) {
                InvalidInputException e = new InvalidInputException(1, InvalidInputException.NULL_ARGUMENT);
                session.handleInputException(e);
            }

            Element artifact = session.getChild(uat, "UndoArtifact");
            if (artifact != null) {
                uat.removeChild(artifact);
            }
        }
        else {
            Element artifact = session.getChild(uat, "UndoArtifact");
            if (artifact != null) {
                uat.removeChild(artifact);
            }
        	
        	session.insertNode(uat, (SPIDataObject) undoArtifact, INSTALLATION_ARTIFACTS_TYPE);
            setContent(uat);
        }
    }

    public void setUninstallArtifact(Artifact uninstallArtifact) {
    	Element uat = getChild("Artifacts");
    	if (uat == null) {
    		uat = session.createDeploymentElement("Artifacts");
    	}
    	if (uninstallArtifact == null) {
    		// Special case, handle attempts to remove the only artifact
            // currently contained by the install unit
            if (session.getChildren(uat, "UninstallArtifact") != null &&
            		session.getChildren(uat, "InstallArtifact") == null &&
                    session.getChildren(uat, "UpdateArtifact") == null &&
                    session.getChildren(uat, "UndoArtifact") == null) {
                InvalidInputException e = new InvalidInputException(1, InvalidInputException.NULL_ARGUMENT);
                session.handleInputException(e);
            }
            
            Element artifact = session.getChild(uat, "UninstallArtifact");
            if (artifact != null) {
                uat.removeChild(artifact);
            }
        }
        else {
            Element artifact = session.getChild(uat, "UninstallArtifact");
            if (artifact != null) {
                uat.removeChild(artifact);
            }

        	session.insertNode(uat, (SPIDataObject) uninstallArtifact, INSTALLATION_ARTIFACTS_TYPE);
            setContent(uat);
        }        
    }

    public String toString() {
        return super.toString();
    }

    public RequiredBase getRequiredBase() {
        RequiredBase copiedRequiredBase = null;
        
        if (isSetElement("RequiredBase")) {
        	copiedRequiredBase = new RequiredBaseImpl(getChild("RequiredBase"), session);
        }
        
        return copiedRequiredBase;
    }
    
    public Collection<Requirement> getRequirements() {
    	Collection<Requirement> requirements = new ArrayList<Requirement>();

        if (isSetElement("Requirements")) {
            Collection<Element> requirementElementList = session.getChildren(getChild("Requirements"), "Requirement");

            for (Element element : requirementElementList) {
                requirements.add(new RequirementImpl(element, session));
            }
        }
        
        return requirements;
    }

    public void setRequiredBase(RequiredBase newRequiredBase) {
        if (newRequiredBase == null) {
        	unSetElement("RequiredBase");
        }
        else {
        	setContent((SPIDataObject) newRequiredBase);
        }
    }

    public void setRequirements(Collection<Requirement> newRequirements) {
    	if (newRequirements != null && !newRequirements.isEmpty()) {
            Element requirementsElement;

            if (isSetElement("Requirements")) {
                requirementsElement = getChild("Requirements");
            }
            else {
                requirementsElement = session.createDeploymentElement("Requirements");
            }

            //Remove all Requirement elements
            Collection<Element> requirementNodeList = session.getChildren(requirementsElement, "Requirement");
            for (Element element : requirementNodeList) {
                requirementsElement.removeChild(element);
            }
            
            for (Requirement requirement : newRequirements) {
                session.insertNode(requirementsElement, (SPIDataObject) requirement, REQUIREMENTS_TYPE);
            }

            setContent(requirementsElement);
        }
        else {
            unSetElement("Requirements");
        }
    }
}