/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.contentunit;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.DisplayNameableObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.condition.ConditionImpl;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.variable.DerivedVariableImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Language;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.LanguageSet;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Multiplicity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.NestedFeature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.PackageFeatureReference;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.DerivedVariable;
import org.w3c.dom.Element;

public class NestedFeatureImpl extends DisplayNameableObject implements NestedFeature {
	private static final String NESTED_FEATURE_TYPE = "sdd-dd:NestedFeatureType";
	private static final String LANGUAGE_SELECTIONS_TYPE = "sdd-dd:LanguageSelectionsType";
	private static final String LANGUAGES_TYPE = "sdd-dd:LanguagesType";
	private static final String OPTIONAL_LANGUAGES_TYPE = "sdd-dd:OptionalLanguagesType";

	public NestedFeatureImpl(Element data, SPISessionImpl session) {
        super(data, session, NESTED_FEATURE_TYPE);
    }
	
	public NestedFeatureImpl(Element data, SPISessionImpl session, String sddSchemaType) {
        super(data, session, sddSchemaType);
    }

    public Collection<Language> getMandatoryLanguages() {
        Collection<Language> languages = new ArrayList<Language>();
        
        Element manditoryElement = session.getChild(getChild("Languages"), "Mandatory");

        if (isSetElement("Languages") && manditoryElement != null) 
        {
            Collection<Element> languageElementList = session.getChildren(manditoryElement, "Language");

            for (Element element : languageElementList)
            {
                languages.add(new LanguageImpl(element, session));
            }
        }

        return languages;
    }

    public Collection<Language> getOptionalLanguages() {
        Collection<Language> languages = new ArrayList<Language>();
        
        if (isSetElement("Languages"))
        {
        	Element optionalElement = session.getChild(getChild("Languages"), "Optional");
        	
            if (optionalElement != null) 
            {
                Collection<Element> languageElementList = session.getChildren(optionalElement, "Language");

                for (Element element : languageElementList)
                {
                    languages.add(new LanguageImpl(element, session));
                }
            }

        		
        }

        return languages;
    }

    public Collection<LanguageSet> getOptionalLanguageSets() {
        Collection<LanguageSet> languages = new ArrayList<LanguageSet>();

        Element optionalElement = session.getChild(getChild("Languages"), "Optional");
        
        if (isSetElement("Languages") && optionalElement != null) 
        {
            Collection<Element> languageElementList = session.getChildren(optionalElement, "LanguageSet");

            for (Element element : languageElementList)
            {
                languages.add(new LanguageSetImpl(element, session));
            }
        }

        return languages;
    }

    public Condition getCondition() {
        Condition condition = null;
        
        if (isSetElement("Condition")) {
        	condition = new ConditionImpl(getChild("Condition"), session);
        }
        
        return condition;
    }

    public Collection<String> getContentElements() {
    	Collection<String> contentElementRefs = new ArrayList<String>();
    	Collection<Element> contentElements = getChildren("ContentElement");
        
        for (Element contentElement : contentElements) {
        	String contentElementRef = contentElement.getAttribute("contentElementRef");
        	
        	if (contentElementRef != null) {
        		contentElementRefs.add(contentElementRef);
        	}
        }
        
        return contentElementRefs;
    }

    public Multiplicity getMultiplicity() {
        Multiplicity multiplicity = null;
        
        if (isSetElement("Multiplicity")) {
        	multiplicity = new MultiplicityImpl(getChild("Multiplicity"), session);
        }
        
        return multiplicity;
    }
    
    public String getFeatureID() {
    	return getAttribute("id");
    }

    public Collection<NestedFeature> getNestedFeatures() {
    	Collection<NestedFeature> nestedFeatures = new ArrayList<NestedFeature>();
    	Collection<Element> nestedFeatureElements = getChildren("NestedFeature");
    	
    	for (Element nestedFeatureElement : nestedFeatureElements) {
    		nestedFeatures.add(new NestedFeatureImpl(nestedFeatureElement, session));
    	}
    	
    	return nestedFeatures;
    }

    public Collection<String> getRequiredFeature() {
    	Collection<String> requiredFeatureRefs = new ArrayList<String>();
    	Collection<Element> requiredFeatureElements = getChildren("RequiredFeature");
        
        for (Element requiredFeatureElement : requiredFeatureElements) {
        	String contentElementRef = requiredFeatureElement.getAttribute("featureRef");
        	
        	if (contentElementRef != null) {
        		requiredFeatureRefs.add(contentElementRef);
        	}
        }
        
        return requiredFeatureRefs;
    }
    
    public Collection<DerivedVariable> getVariables() {
    	Collection<DerivedVariable> derivedVariables = new ArrayList<DerivedVariable>();
    	Collection<Element> derivedVariableElements = getChildren("Variable");
    	
    	for (Element derivedVariableElement : derivedVariableElements) {
    		derivedVariables.add(new DerivedVariableImpl(derivedVariableElement, session));
    	}
        
        return derivedVariables;
    }

    public void setMandatoryLanguages(Collection<Language> languages) {
        Element languageSelectionsElement;

        if (isSetElement("Languages")) {
            languageSelectionsElement = getChild("Languages");
        }
        else {
            languageSelectionsElement = session.createDeploymentElement("Languages");
        }

        if (languages != null && !languages.isEmpty()) {
            Element mandatoryElement = session.getChild(languageSelectionsElement, "Mandatory");
            
            if (mandatoryElement == null) {
                mandatoryElement = session.createDeploymentElement("Mandatory");
            }

            //Remove all Language elements
            Collection<Element> languageNodeList = session.getChildren(mandatoryElement, "Language");
            for (Element element : languageNodeList)
            {
                mandatoryElement.removeChild(element);
            }
            
            for (Language language : languages) {
                session.insertNode(mandatoryElement, (SPIDataObject) language, LANGUAGES_TYPE);
            }

            //Remove all Manditory elements
            Collection<Element> manditoryNodeList = session.getChildren(languageSelectionsElement, "Manditory");
            for (Element element : manditoryNodeList)
            {
                languageSelectionsElement.removeChild(element);
            }
            
            session.insertNode(languageSelectionsElement, mandatoryElement, LANGUAGE_SELECTIONS_TYPE);
            
            setContent(languageSelectionsElement);
        }
        else {
            //Remove all Manditory elements
            Collection<Element> manditoryNodeList = session.getChildren(languageSelectionsElement, "Manditory");
            for (Element element : manditoryNodeList)
            {
                languageSelectionsElement.removeChild(element);
            }

            if (session.getChildren(languageSelectionsElement, "Manditory").isEmpty() && session.getChildren(languageSelectionsElement, "Optional").isEmpty()) {
                unSetElement("Languages");
            }
        }
    }

    public void setOptionalLanguages(Collection<Language> languages) {
        Element languageSelectionsElement;

        if (isSetElement("Languages")) {
            languageSelectionsElement = getChild("Languages");
        }
        else {
            languageSelectionsElement = session.createDeploymentElement("Languages");
        }

        if (languages != null && !languages.isEmpty()) {
            Element optionalElement = session.getChild(languageSelectionsElement, "Optional");

            if (optionalElement == null) {
                optionalElement = session.createDeploymentElement("Optional");
            }

            //Remove all Language elements
            Collection<Element> languageNodeList = session.getChildren(optionalElement, "Language");
            for (Element element : languageNodeList)
            {
                optionalElement.removeChild(element);
            }
            
            for (Language language : languages) {
                session.insertNode(optionalElement, (SPIDataObject) language, OPTIONAL_LANGUAGES_TYPE);
            }

            //Remove all Optional elements
            Collection<Element> optionalNodeList = session.getChildren(languageSelectionsElement, "Optional");
            for (Element element : optionalNodeList)
            {
                languageSelectionsElement.removeChild(element);
            }
            
            session.insertNode(languageSelectionsElement, optionalElement, LANGUAGE_SELECTIONS_TYPE);
            
            setContent(languageSelectionsElement);
        }
        else {
            //Remove all Optional elements
            Collection<Element> optionalNodeList = session.getChildren(languageSelectionsElement, "Optional");
            for (Element element : optionalNodeList)
            {
                languageSelectionsElement.removeChild(element);
            }

            if (session.getChildren(languageSelectionsElement, "Manditory").isEmpty() && session.getChildren(languageSelectionsElement, "Optional").isEmpty()) {
                unSetElement("Languages");
            }
        }
    }

    public void setOptionalLanguageSets(Collection<LanguageSet> languageSets) {
        Element languageSelectionsElement;

        if (isSetElement("Languages")) {
            languageSelectionsElement = getChild("Languages");
        }
        else {
            languageSelectionsElement = session.createDeploymentElement("Languages");
        }

        if (languageSets != null && !languageSets.isEmpty()) {
            Element optionalElement = session.getChild(languageSelectionsElement, "Optional");

            if (optionalElement == null) {
                optionalElement = session.createDeploymentElement("Optional");
            }

            //Remove all LanguageSet elements
            Collection<Element> languageSetNodeList = session.getChildren(optionalElement, "LanguageSet");
            for (Element element : languageSetNodeList)
            {
                optionalElement.removeChild(element);
            }
            
            for (LanguageSet languageSet : languageSets) {
                session.insertNode(optionalElement, (SPIDataObject) languageSet, OPTIONAL_LANGUAGES_TYPE);
            }

            //Remove all Optional elements
            Collection<Element> optionalNodeList = session.getChildren(languageSelectionsElement, "Optional");
            for (Element element : optionalNodeList)
            {
                languageSelectionsElement.removeChild(element);
            }
            
            session.insertNode(languageSelectionsElement, optionalElement, LANGUAGE_SELECTIONS_TYPE);
            
            setContent(languageSelectionsElement);
        }
        else {
            //Remove all Optional elements
            Collection<Element> optionalNodeList = session.getChildren(languageSelectionsElement, "Optional");
            for (Element element : optionalNodeList)
            {
                languageSelectionsElement.removeChild(element);
            }

            if (session.getChildren(languageSelectionsElement, "Manditory").isEmpty() && session.getChildren(languageSelectionsElement, "Optional").isEmpty()) {
                unSetElement("Languages");
            }
        }
    }

    public void setCondition(Condition condition) {
        if (condition == null) {
            unSetElement("Condition");
        }
        else {
        	setContent((SPIDataObject)condition);
        }
    }

    public void setContentElements(Collection<String> content) {
    	if (content == null) {
    		unSetElement("ContentElement");
    	}
    	else {
    		Collection<Element> contentElements = new ArrayList<Element>();
    		removeContent(getChildren("ContentElement", false));
    		
    		for (String contentString : content) {
    			Element contentElement = session.createDeploymentElement("ContentElement");
    			contentElement.setAttribute("contentElementRef", contentString);
    			
    			contentElements.add(contentElement);
    		}
    		
    		addContent(contentElements);
    	}
    }

    public void setMultiplicity(Multiplicity multiplicity) {
        if (multiplicity == null) {
        	unSetElement("Multiplicity");
        }
        else {
        	setContent((SPIDataObject)multiplicity);
        }
    }
    
    public void setFeatureID(String id) {
        session.testParameter(id, 1, null);

        setAttribute("id", id);
    }

    public void setNestedFeatures(Collection<NestedFeature> features) {
    	if (features == null) {
    		unSetElement("NestedFeature");
    	}
    	else {
    		removeContent(getChildren("NestedFeature", false));
    		
    		addContent(features);
    	}
    }

    public void setRequiredFeature(Collection<String> features) {
    	if (features == null) {
    		unSetElement("RequiredFeature");
    	}
    	else {
    		Collection<Element> requiredFeatureElements = new ArrayList<Element>();
    		removeContent(getChildren("RequiredFeature", false));
    		
    		for (String feature : features) {
    			Element requiredFeatureElement = session.createDeploymentElement("RequiredFeature");
    			requiredFeatureElement.setAttribute("featureRef", feature);
    			
    			requiredFeatureElements.add(requiredFeatureElement);
    		}
    		
    		addContent(requiredFeatureElements);
    	}
    }

    public Object clone() {
    	return new NestedFeatureImpl((Element)getElement().cloneNode(true), session);
    }

    public Collection<PackageFeatureReference> getPackageFeatureReferences() {
    	Collection<PackageFeatureReference> pkgFeatureRefs = new ArrayList<PackageFeatureReference>();
    	Collection<Element> pkgFeatureRefElements = getChildren("PackageFeatureReference");
    	
    	for (Element pkgFeatureRefElement : pkgFeatureRefElements) {
    		PackageFeatureReference pkgFeatureRef =
    			new PackageFeatureReferenceImpl(pkgFeatureRefElement, session);
    		pkgFeatureRefs.add(pkgFeatureRef);
    	}
    	
    	return pkgFeatureRefs;
    }

    public void setPackageFeatureReferences(Collection<PackageFeatureReference> packageFeatureReferences) {
    	if (packageFeatureReferences == null) {
    		unSetElement("PackageFeatureReference");
    	}
    	else {
    		removeContent(getChildren("PackageFeatureReference", false));
    		
    		addContent(packageFeatureReferences);
    	}
    }

    public void setVariables(Collection<DerivedVariable> variables) {
    	if (variables == null) {
    		unSetElement("Variable");
    	}
    	else {
    		removeContent(getChildren("Variable", false));
    		
    		addContent(variables);
    	}
    }

    public Boolean getAddOn() {
    	Boolean addOn = null;
    	
    	if (isSetAttribute("addOn")) {
    		addOn = Boolean.parseBoolean(getAttribute("addOn"));
    	}
    	
    	return addOn;
    }

    public void setAddOn(Boolean addOn) {
        if (addOn == null) {
        	unSetAttribute("addOn");
        }
        else {
        	setAttribute("addOn", addOn.toString());
        }
    }
}
