/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.util.Arrays;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.Identity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.ReferencedPackage;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.TranslatableString;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.artifact.Artifact;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.CompositeUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ConfigurationUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.ContentUnitFactory;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.InstallableUnit;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.requirement.Requirement;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.BooleanParameter;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ConditionalDerivedVariable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ConditionalDerivedVariableExpression;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.DerivedVariable;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ParameterGroup;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.ResourceProperty;
import org.w3c.dom.Element;

public class CompositeUnit_Test extends TestCase
{
    CompositeUnit unit;
    CompositeUnit clonedUnit;
    Element element;
    SPISessionImpl session;

    public CompositeUnit_Test(String name)
    {
        super(name);
    }

    protected void setUp() throws Exception
    {
        super.setUp();
        
        unit = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createCompositeUnit("unitID");
        
        CompositeUnit cu1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createCompositeUnit("cu1ID");
        CompositeUnit cu2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createCompositeUnit("cu2ID");
        unit.setCompositeUnits(Arrays.asList(cu1, cu2));
        
        Condition condition = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        condition.setDisplayName(new TranslatableString("conName", "conNameKey"));
        unit.setCondition(condition);

        Condition con1 = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        con1.setDisplayName(new TranslatableString("con1Name", "con1NameKey"));
        ConditionalDerivedVariableExpression exp1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createConditionalDerivedVariableExpression("exp1Expression", con1);
        Condition con2 = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        con2.setDisplayName(new TranslatableString("con2Name", "con2NameKey"));
        ConditionalDerivedVariableExpression exp2 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createConditionalDerivedVariableExpression("exp2Expression", con2);
        ConditionalDerivedVariable var1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createConditionalDerivedVariable("var1ID", Arrays.asList(exp1, exp2));

        Condition con3 = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        con3.setDisplayName(new TranslatableString("con3Name", "con3NameKey"));
        ConditionalDerivedVariableExpression exp3 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createConditionalDerivedVariableExpression("exp3Expression", con3);
        Condition con4 = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        con4.setDisplayName(new TranslatableString("con4Name", "con4NameKey"));
        ConditionalDerivedVariableExpression exp4 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createConditionalDerivedVariableExpression("exp4Expression", con4);
        ConditionalDerivedVariable var2 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createConditionalDerivedVariable("var2ID", Arrays.asList(exp3, exp4));
        unit.setConditionalDerivedVariables(Arrays.asList(var1, var2));
        
        Artifact art1 = SPISession.DEFAULT_INSTANCE.createArtifactFactory().createConfigArtifact();
        art1.setContentReference("art1ContentRef");
        ConfigurationUnit cfg1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createConfigurationUnit("cfg1ID", "cfg1ResourceRef", art1);
        Artifact art2 = SPISession.DEFAULT_INSTANCE.createArtifactFactory().createConfigArtifact();
        art2.setContentReference("art2ContentRef");
        ConfigurationUnit cfg2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createConfigurationUnit("cfg2ID", "cfg2ResourceRef", art2);
        unit.setConfigurationUnits(Arrays.asList(cfg1, cfg2));
        
        ReferencedPackage package1 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createContainedPackage("package1Ref", "package1ID");
        ReferencedPackage package2 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createContainedPackage("package2Ref", "package2ID");
        unit.setContainedPackages(Arrays.asList(package1, package2));
        
        DerivedVariable dvar1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createDerivedVariable("dvar1ID", "dvar1Expression");
        DerivedVariable dvar2 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createDerivedVariable("dvar2ID", "dvar2Expression");
        unit.setDerivedVariables(Arrays.asList(dvar1, dvar2));
        
        Identity identity = SPISession.DEFAULT_INSTANCE.createBaseFactory().createIdentity(new TranslatableString("name", "nameKey"));
        unit.setIdentityData(identity);
        
        Artifact iuArt1 = SPISession.DEFAULT_INSTANCE.createArtifactFactory().createInstallArtifact();
        iuArt1.setContentReference("iuArt1ContentRef");
        InstallableUnit iu1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createInstallableUnit("iu1ID", "iu1ResourceRef", iuArt1, ContentUnitFactory.INSTALL_ARTIFACT);
        Artifact iuArt2 = SPISession.DEFAULT_INSTANCE.createArtifactFactory().createInstallArtifact();
        iuArt2.setContentReference("iuArt2ContentRef");
        InstallableUnit iu2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createInstallableUnit("iu2ID", "iu2ResourceRef", iuArt2, ContentUnitFactory.INSTALL_ARTIFACT);
        unit.setInstallableUnits(Arrays.asList(iu1, iu2));
        
        BooleanParameter param1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createBooleanParameter("param1ID");
        BooleanParameter param2 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createBooleanParameter("param2ID");
        ParameterGroup group1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createParameterGroup();
        group1.setBooleanParameters(Arrays.asList(param1, param2));
        BooleanParameter param3 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createBooleanParameter("param3ID");
        BooleanParameter param4 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createBooleanParameter("param4ID");
        ParameterGroup group2 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createParameterGroup();
        group2.setBooleanParameters(Arrays.asList(param3, param4));
        unit.setParameterGroups(Arrays.asList(group1, group2));
        
        Requirement req1 = SPISession.DEFAULT_INSTANCE.createRequirementFactory().createRequirement("req1ID");
        Requirement req2 = SPISession.DEFAULT_INSTANCE.createRequirementFactory().createRequirement("req2ID");
        unit.setRequirements(Arrays.asList(req1, req2));
        
        ResourceProperty prop1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createResourceProperty("prop1ID", "prop1Ref");
        ResourceProperty prop2 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createResourceProperty("prop2ID", "prop2Ref");
        unit.setResourceProperties(Arrays.asList(prop1, prop2));
        
        clonedUnit = (CompositeUnit) unit.clone();
        element = ((SPIDataObject) unit).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetTranslatableStrings()
    {
        Properties props = ((SPIDataObject) unit).getTranslatableStrings();
        
        assertEquals(6, props.size());
        assertEquals("name", props.getProperty("nameKey"));
        assertEquals("conName", props.getProperty("conNameKey"));
        assertEquals("con1Name", props.getProperty("con1NameKey"));
        assertEquals("con2Name", props.getProperty("con2NameKey"));
        assertEquals("con3Name", props.getProperty("con3NameKey"));
        assertEquals("con4Name", props.getProperty("con4NameKey"));
    }

    public void testGetCompositeUnits()
    {
        assertEquals(2, unit.getCompositeUnits().size());
        assertEquals("cu1ID", unit.getCompositeUnits().toArray(new CompositeUnit[0])[0].getContentUnitID());
        assertEquals("cu2ID", unit.getCompositeUnits().toArray(new CompositeUnit[0])[1].getContentUnitID());

        assertEquals(2, clonedUnit.getCompositeUnits().size());
        assertEquals("cu1ID", clonedUnit.getCompositeUnits().toArray(new CompositeUnit[0])[0].getContentUnitID());
        assertEquals("cu2ID", clonedUnit.getCompositeUnits().toArray(new CompositeUnit[0])[1].getContentUnitID());

        assertEquals(2, session.getChildren(element, "CompositeUnit").size());
        assertEquals("cu1ID", session.getChildren(element, "CompositeUnit").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("cu2ID", session.getChildren(element, "CompositeUnit").toArray(new Element[0])[1].getAttribute("id"));
    }

    public void testGetCondition()
    {
        assertEquals("conName", unit.getCondition().getDisplayName().getValue());
        assertEquals("conNameKey", unit.getCondition().getDisplayName().getTranslationKey());

        assertEquals("conName", clonedUnit.getCondition().getDisplayName().getValue());
        assertEquals("conNameKey", clonedUnit.getCondition().getDisplayName().getTranslationKey());

        assertEquals("conName", session.getChildren(session.getChildren(element, "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getTextContent());
        assertEquals("conNameKey", session.getChildren(session.getChildren(element, "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getAttribute("translationKey"));
    }

    public void testGetConfigurationUnits()
    {
        assertEquals(2, unit.getConfigurationUnits().size());
        assertEquals("cfg1ID", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getContentUnitID());
        assertEquals("cfg1ResourceRef", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getTargetResourceReference());
        assertEquals("art1ContentRef", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getConfigurationArtifact().getContentReference());
        assertEquals("cfg2ID", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[1].getContentUnitID());
        assertEquals("cfg2ResourceRef", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[1].getTargetResourceReference());
        assertEquals("art2ContentRef", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[1].getConfigurationArtifact().getContentReference());

        assertEquals(2, clonedUnit.getConfigurationUnits().size());
        assertEquals("cfg1ID", clonedUnit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getContentUnitID());
        assertEquals("cfg1ResourceRef", clonedUnit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getTargetResourceReference());
        assertEquals("art1ContentRef", clonedUnit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getConfigurationArtifact().getContentReference());
        assertEquals("cfg2ID", clonedUnit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[1].getContentUnitID());
        assertEquals("cfg2ResourceRef", clonedUnit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[1].getTargetResourceReference());
        assertEquals("art2ContentRef", clonedUnit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[1].getConfigurationArtifact().getContentReference());

        assertEquals(2, session.getChildren(element, "ConfigurationUnit").size());
        assertEquals("cfg1ID", session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("cfg1ResourceRef", session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[0].getAttribute("targetResourceRef"));
        assertEquals("art1ContentRef", session.getChildren(session.getChildren(session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[0], "Artifacts").toArray(new Element[0])[0], "ConfigArtifact").toArray(new Element[0])[0].getAttribute("contentRef"));
        assertEquals("cfg2ID", session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[1].getAttribute("id"));
        assertEquals("cfg2ResourceRef", session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[1].getAttribute("targetResourceRef"));
        assertEquals("art2ContentRef", session.getChildren(session.getChildren(session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[1], "Artifacts").toArray(new Element[0])[0], "ConfigArtifact").toArray(new Element[0])[0].getAttribute("contentRef"));
    }

    public void testGetContainedPackages()
    {
        assertEquals(2, unit.getContainedPackages().size());
        assertEquals("package1ID", unit.getContainedPackages().toArray(new ReferencedPackage[0])[0].getReferencedPackageID());
        assertEquals("package1Ref", unit.getContainedPackages().toArray(new ReferencedPackage[0])[0].getContentReference());
        assertEquals("package2ID", unit.getContainedPackages().toArray(new ReferencedPackage[0])[1].getReferencedPackageID());
        assertEquals("package2Ref", unit.getContainedPackages().toArray(new ReferencedPackage[0])[1].getContentReference());

        assertEquals(2, clonedUnit.getContainedPackages().size());
        assertEquals("package1ID", clonedUnit.getContainedPackages().toArray(new ReferencedPackage[0])[0].getReferencedPackageID());
        assertEquals("package1Ref", clonedUnit.getContainedPackages().toArray(new ReferencedPackage[0])[0].getContentReference());
        assertEquals("package2ID", clonedUnit.getContainedPackages().toArray(new ReferencedPackage[0])[1].getReferencedPackageID());
        assertEquals("package2Ref", clonedUnit.getContainedPackages().toArray(new ReferencedPackage[0])[1].getContentReference());

        assertEquals(2, session.getChildren(element, "ContainedPackage").size());
        assertEquals("package1ID", session.getChildren(element, "ContainedPackage").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("package1Ref", session.getChildren(element, "ContainedPackage").toArray(new Element[0])[0].getAttribute("contentRef"));
        assertEquals("package2ID", session.getChildren(element, "ContainedPackage").toArray(new Element[0])[1].getAttribute("id"));
        assertEquals("package2Ref", session.getChildren(element, "ContainedPackage").toArray(new Element[0])[1].getAttribute("contentRef"));
    }

    public void testGetInstallableUnits()
    {
        assertEquals(2, unit.getInstallableUnits().size());
        assertEquals("iu1ID", unit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getContentUnitID());
        assertEquals("iu1ResourceRef", unit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getTargetResourceReference());
        assertEquals("iuArt1ContentRef", unit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getInstallArtifact().getContentReference());
        assertEquals("iu2ID", unit.getInstallableUnits().toArray(new InstallableUnit[0])[1].getContentUnitID());
        assertEquals("iu2ResourceRef", unit.getInstallableUnits().toArray(new InstallableUnit[0])[1].getTargetResourceReference());
        assertEquals("iuArt2ContentRef", unit.getInstallableUnits().toArray(new InstallableUnit[0])[1].getInstallArtifact().getContentReference());

        assertEquals(2, clonedUnit.getInstallableUnits().size());
        assertEquals("iu1ID", clonedUnit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getContentUnitID());
        assertEquals("iu1ResourceRef", clonedUnit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getTargetResourceReference());
        assertEquals("iuArt1ContentRef", clonedUnit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getInstallArtifact().getContentReference());
        assertEquals("iu2ID", clonedUnit.getInstallableUnits().toArray(new InstallableUnit[0])[1].getContentUnitID());
        assertEquals("iu2ResourceRef", clonedUnit.getInstallableUnits().toArray(new InstallableUnit[0])[1].getTargetResourceReference());
        assertEquals("iuArt2ContentRef", clonedUnit.getInstallableUnits().toArray(new InstallableUnit[0])[1].getInstallArtifact().getContentReference());

        assertEquals(2, session.getChildren(element, "InstallableUnit").size());
        assertEquals("iu1ID", session.getChildren(element, "InstallableUnit").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("iu1ResourceRef", session.getChildren(element, "InstallableUnit").toArray(new Element[0])[0].getAttribute("targetResourceRef"));
        assertEquals("iuArt1ContentRef", session.getChildren(session.getChildren(session.getChildren(element, "InstallableUnit").toArray(new Element[0])[0], "Artifacts").toArray(new Element[0])[0], "InstallArtifact").toArray(new Element[0])[0].getAttribute("contentRef"));
        assertEquals("iu2ID", session.getChildren(element, "InstallableUnit").toArray(new Element[0])[1].getAttribute("id"));
        assertEquals("iu2ResourceRef", session.getChildren(element, "InstallableUnit").toArray(new Element[0])[1].getAttribute("targetResourceRef"));
        assertEquals("iuArt2ContentRef", session.getChildren(session.getChildren(session.getChildren(element, "InstallableUnit").toArray(new Element[0])[1], "Artifacts").toArray(new Element[0])[0], "InstallArtifact").toArray(new Element[0])[0].getAttribute("contentRef"));
    }

    public void testGetRequirements()
    {
        assertEquals(2, unit.getRequirements().size());
        assertEquals("req1ID", unit.getRequirements().toArray(new Requirement[0])[0].getRequirementID());
        assertEquals("req2ID", unit.getRequirements().toArray(new Requirement[0])[1].getRequirementID());
        
        assertEquals(2, clonedUnit.getRequirements().size());
        assertEquals("req1ID", clonedUnit.getRequirements().toArray(new Requirement[0])[0].getRequirementID());
        assertEquals("req2ID", clonedUnit.getRequirements().toArray(new Requirement[0])[1].getRequirementID());

        assertEquals(2, session.getChildren(session.getChildren(element, "Requirements").toArray(new Element[0])[0], "Requirement").size());
        assertEquals("req1ID", session.getChildren(session.getChildren(element, "Requirements").toArray(new Element[0])[0], "Requirement").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("req2ID", session.getChildren(session.getChildren(element, "Requirements").toArray(new Element[0])[0], "Requirement").toArray(new Element[0])[1].getAttribute("id"));
    }

    public void testGetContentUnitID()
    {
        assertEquals("unitID", unit.getContentUnitID());
        assertEquals("unitID", clonedUnit.getContentUnitID());
        assertEquals("unitID", element.getAttribute("id"));
    }

    public void testGetIdentityData()
    {
        assertEquals("name", unit.getIdentityData().getName().getValue());
        assertEquals("nameKey", unit.getIdentityData().getName().getTranslationKey());
        
        assertEquals("name", clonedUnit.getIdentityData().getName().getValue());
        assertEquals("nameKey", clonedUnit.getIdentityData().getName().getTranslationKey());

        assertEquals("name", session.getChildren(session.getChildren(element, "Identity").toArray(new Element[0])[0], "Name").toArray(new Element[0])[0].getTextContent());
        assertEquals("nameKey", session.getChildren(session.getChildren(element, "Identity").toArray(new Element[0])[0], "Name").toArray(new Element[0])[0].getAttribute("translationKey"));
    }

    public void testGetDerivedVariables()
    {
        assertEquals(2, unit.getDerivedVariables().size());
        assertEquals("dvar1ID", unit.getDerivedVariables().toArray(new DerivedVariable[0])[0].getVariableID());
        assertEquals("dvar1Expression", unit.getDerivedVariables().toArray(new DerivedVariable[0])[0].getExpression());
        assertEquals("dvar2ID", unit.getDerivedVariables().toArray(new DerivedVariable[0])[1].getVariableID());
        assertEquals("dvar2Expression", unit.getDerivedVariables().toArray(new DerivedVariable[0])[1].getExpression());

        assertEquals(2, clonedUnit.getDerivedVariables().size());
        assertEquals("dvar1ID", clonedUnit.getDerivedVariables().toArray(new DerivedVariable[0])[0].getVariableID());
        assertEquals("dvar1Expression", clonedUnit.getDerivedVariables().toArray(new DerivedVariable[0])[0].getExpression());
        assertEquals("dvar2ID", clonedUnit.getDerivedVariables().toArray(new DerivedVariable[0])[1].getVariableID());
        assertEquals("dvar2Expression", clonedUnit.getDerivedVariables().toArray(new DerivedVariable[0])[1].getExpression());

        assertEquals(4, session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").size());
        assertEquals("dvar1ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[2].getAttribute("id"));
        assertEquals("dvar1Expression", session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[2], "Expression").toArray(new Element[0])[0].getTextContent());
        assertEquals("dvar2ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[3].getAttribute("id"));
        assertEquals("dvar2Expression", session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[3], "Expression").toArray(new Element[0])[0].getTextContent());
    }

    public void testGetConditionalDerivedVariables()
    {
        assertEquals(2, unit.getConditionalDerivedVariables().size());
        assertEquals("var1ID", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getVariableID());
        assertEquals("exp1Expression", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getExpression());
        assertEquals("con1Name", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getValue());
        assertEquals("con1NameKey", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getTranslationKey());
        assertEquals("exp2Expression", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getExpression());
        assertEquals("con2Name", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getCondition().getDisplayName().getValue());
        assertEquals("con2NameKey", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getCondition().getDisplayName().getTranslationKey());
        assertEquals("var2ID", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getVariableID());
        assertEquals("exp3Expression", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getExpression());
        assertEquals("con3Name", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getValue());
        assertEquals("con3NameKey", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getTranslationKey());
        assertEquals("exp4Expression", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getExpression());
        assertEquals("con4Name", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getCondition().getDisplayName().getValue());
        assertEquals("con4NameKey", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getCondition().getDisplayName().getTranslationKey());

        assertEquals(2, clonedUnit.getConditionalDerivedVariables().size());
        assertEquals("var1ID", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getVariableID());
        assertEquals("exp1Expression", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getExpression());
        assertEquals("con1Name", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getValue());
        assertEquals("con1NameKey", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getTranslationKey());
        assertEquals("exp2Expression", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getExpression());
        assertEquals("con2Name", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getCondition().getDisplayName().getValue());
        assertEquals("con2NameKey", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getCondition().getDisplayName().getTranslationKey());
        assertEquals("var2ID", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getVariableID());
        assertEquals("exp3Expression", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getExpression());
        assertEquals("con3Name", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getValue());
        assertEquals("con3NameKey", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getTranslationKey());
        assertEquals("exp4Expression", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getExpression());
        assertEquals("con4Name", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getCondition().getDisplayName().getValue());
        assertEquals("con4NameKey", clonedUnit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[1].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[1].getCondition().getDisplayName().getTranslationKey());

        assertEquals(4, session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").size());
        assertEquals("var1ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("exp1Expression", session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[0], "ConditionalExpression").toArray(new Element[0])[0], "Expression").toArray(new Element[0])[0].getTextContent());
        assertEquals("con1Name", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[0], "ConditionalExpression").toArray(new Element[0])[0], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getTextContent());
        assertEquals("con1NameKey", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[0], "ConditionalExpression").toArray(new Element[0])[0], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getAttribute("translationKey"));
        assertEquals("exp2Expression", session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[0], "ConditionalExpression").toArray(new Element[0])[1], "Expression").toArray(new Element[0])[0].getTextContent());
        assertEquals("con2Name", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[0], "ConditionalExpression").toArray(new Element[0])[1], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getTextContent());
        assertEquals("con2NameKey", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[0], "ConditionalExpression").toArray(new Element[0])[1], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getAttribute("translationKey"));
        assertEquals("var2ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[1].getAttribute("id"));
        assertEquals("exp3Expression", session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[1], "ConditionalExpression").toArray(new Element[0])[0], "Expression").toArray(new Element[0])[0].getTextContent());
        assertEquals("con3Name", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[1], "ConditionalExpression").toArray(new Element[0])[0], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getTextContent());
        assertEquals("con3NameKey", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[1], "ConditionalExpression").toArray(new Element[0])[0], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getAttribute("translationKey"));
        assertEquals("exp4Expression", session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[1], "ConditionalExpression").toArray(new Element[0])[1], "Expression").toArray(new Element[0])[0].getTextContent());
        assertEquals("con4Name", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[1], "ConditionalExpression").toArray(new Element[0])[1], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getTextContent());
        assertEquals("con4NameKey", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[1], "ConditionalExpression").toArray(new Element[0])[1], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getAttribute("translationKey"));
    }

    public void testGetResourceProperties()
    {
        assertEquals(2, unit.getResourceProperties().size());
        assertEquals("prop1ID", unit.getResourceProperties().toArray(new ResourceProperty[0])[0].getVariableID());
        assertEquals("prop1Ref", unit.getResourceProperties().toArray(new ResourceProperty[0])[0].getResourceReference());
        assertEquals("prop2ID", unit.getResourceProperties().toArray(new ResourceProperty[0])[1].getVariableID());
        assertEquals("prop2Ref", unit.getResourceProperties().toArray(new ResourceProperty[0])[1].getResourceReference());

        assertEquals(2, clonedUnit.getResourceProperties().size());
        assertEquals("prop1ID", clonedUnit.getResourceProperties().toArray(new ResourceProperty[0])[0].getVariableID());
        assertEquals("prop1Ref", clonedUnit.getResourceProperties().toArray(new ResourceProperty[0])[0].getResourceReference());
        assertEquals("prop2ID", clonedUnit.getResourceProperties().toArray(new ResourceProperty[0])[1].getVariableID());
        assertEquals("prop2Ref", clonedUnit.getResourceProperties().toArray(new ResourceProperty[0])[1].getResourceReference());

        assertEquals(2, session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "ResourceProperty").size());
        assertEquals("prop1ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "ResourceProperty").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("prop1Ref", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "ResourceProperty").toArray(new Element[0])[0].getAttribute("resourceRef"));
        assertEquals("prop2ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "ResourceProperty").toArray(new Element[0])[1].getAttribute("id"));
        assertEquals("prop2Ref", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "ResourceProperty").toArray(new Element[0])[1].getAttribute("resourceRef"));
    }

    public void testGetParameterGroups()
    {
        assertEquals(2, unit.getParameterGroups().size());
        assertEquals(2, unit.getParameterGroups().toArray(new ParameterGroup[0])[0].getBooleanParameters().size());
        assertEquals(2, unit.getParameterGroups().toArray(new ParameterGroup[0])[1].getBooleanParameters().size());
        assertEquals("param1ID", unit.getParameterGroups().toArray(new ParameterGroup[0])[0].getBooleanParameters().toArray(new BooleanParameter[0])[0].getParameterID());
        assertEquals("param2ID", unit.getParameterGroups().toArray(new ParameterGroup[0])[0].getBooleanParameters().toArray(new BooleanParameter[0])[1].getParameterID());
        assertEquals("param3ID", unit.getParameterGroups().toArray(new ParameterGroup[0])[1].getBooleanParameters().toArray(new BooleanParameter[0])[0].getParameterID());
        assertEquals("param4ID", unit.getParameterGroups().toArray(new ParameterGroup[0])[1].getBooleanParameters().toArray(new BooleanParameter[0])[1].getParameterID());

        assertEquals(2, clonedUnit.getParameterGroups().size());
        assertEquals(2, clonedUnit.getParameterGroups().toArray(new ParameterGroup[0])[0].getBooleanParameters().size());
        assertEquals(2, clonedUnit.getParameterGroups().toArray(new ParameterGroup[0])[1].getBooleanParameters().size());
        assertEquals("param1ID", clonedUnit.getParameterGroups().toArray(new ParameterGroup[0])[0].getBooleanParameters().toArray(new BooleanParameter[0])[0].getParameterID());
        assertEquals("param2ID", clonedUnit.getParameterGroups().toArray(new ParameterGroup[0])[0].getBooleanParameters().toArray(new BooleanParameter[0])[1].getParameterID());
        assertEquals("param3ID", clonedUnit.getParameterGroups().toArray(new ParameterGroup[0])[1].getBooleanParameters().toArray(new BooleanParameter[0])[0].getParameterID());
        assertEquals("param4ID", clonedUnit.getParameterGroups().toArray(new ParameterGroup[0])[1].getBooleanParameters().toArray(new BooleanParameter[0])[1].getParameterID());

        assertEquals(2, session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").size());
        assertEquals(2, session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").toArray(new Element[0])[0], "BooleanParameter").size());
        assertEquals(2, session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").toArray(new Element[0])[1], "BooleanParameter").size());
        assertEquals("param1ID", session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").toArray(new Element[0])[0], "BooleanParameter").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("param2ID", session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").toArray(new Element[0])[0], "BooleanParameter").toArray(new Element[0])[1].getAttribute("id"));
        assertEquals("param3ID", session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").toArray(new Element[0])[1], "BooleanParameter").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("param4ID", session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").toArray(new Element[0])[1], "BooleanParameter").toArray(new Element[0])[1].getAttribute("id"));
    }

    public void testSetCompositeUnits()
    {
        CompositeUnit cu1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createCompositeUnit("cu3ID");
        unit.setCompositeUnits(Arrays.asList(cu1));

        assertEquals(1, unit.getCompositeUnits().size());
        assertEquals("cu3ID", unit.getCompositeUnits().toArray(new CompositeUnit[0])[0].getContentUnitID());

        assertEquals(1, session.getChildren(element, "CompositeUnit").size());
        assertEquals("cu3ID", session.getChildren(element, "CompositeUnit").toArray(new Element[0])[0].getAttribute("id"));
    }

    public void testSetCondition()
    {
        Condition condition = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        condition.setDisplayName(new TranslatableString("differentName", "differentNameKey"));
        unit.setCondition(condition);
        
        assertEquals("differentName", unit.getCondition().getDisplayName().getValue());
        assertEquals("differentNameKey", unit.getCondition().getDisplayName().getTranslationKey());

        assertEquals("differentName", session.getChildren(session.getChildren(element, "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getTextContent());
        assertEquals("differentNameKey", session.getChildren(session.getChildren(element, "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getAttribute("translationKey"));
    }

    public void testSetConfigurationUnits()
    {
        Artifact art1 = SPISession.DEFAULT_INSTANCE.createArtifactFactory().createConfigArtifact();
        art1.setContentReference("art3ContentRef");
        ConfigurationUnit cfg1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createConfigurationUnit("cfg3ID", "cfg3ResourceRef", art1);
        unit.setConfigurationUnits(Arrays.asList(cfg1));

        assertEquals(1, unit.getConfigurationUnits().size());
        assertEquals("cfg3ID", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getContentUnitID());
        assertEquals("cfg3ResourceRef", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getTargetResourceReference());
        assertEquals("art3ContentRef", unit.getConfigurationUnits().toArray(new ConfigurationUnit[0])[0].getConfigurationArtifact().getContentReference());

        assertEquals(1, session.getChildren(element, "ConfigurationUnit").size());
        assertEquals("cfg3ID", session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("cfg3ResourceRef", session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[0].getAttribute("targetResourceRef"));
        assertEquals("art3ContentRef", session.getChildren(session.getChildren(session.getChildren(element, "ConfigurationUnit").toArray(new Element[0])[0], "Artifacts").toArray(new Element[0])[0], "ConfigArtifact").toArray(new Element[0])[0].getAttribute("contentRef"));
    }

    public void testSetContainedPackages()
    {
        ReferencedPackage package1 = SPISession.DEFAULT_INSTANCE.createBaseFactory().createContainedPackage("package3Ref", "package3ID");
        unit.setContainedPackages(Arrays.asList(package1));
        
        assertEquals(1, unit.getContainedPackages().size());
        assertEquals("package3ID", unit.getContainedPackages().toArray(new ReferencedPackage[0])[0].getReferencedPackageID());
        assertEquals("package3Ref", unit.getContainedPackages().toArray(new ReferencedPackage[0])[0].getContentReference());

        assertEquals(1, session.getChildren(element, "ContainedPackage").size());
        assertEquals("package3ID", session.getChildren(element, "ContainedPackage").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("package3Ref", session.getChildren(element, "ContainedPackage").toArray(new Element[0])[0].getAttribute("contentRef"));
    }

    public void testSetInstallableUnits()
    {
        Artifact iuArt1 = SPISession.DEFAULT_INSTANCE.createArtifactFactory().createInstallArtifact();
        iuArt1.setContentReference("iuArt3ContentRef");
        InstallableUnit iu1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createInstallableUnit("iu3ID", "iu3ResourceRef", iuArt1, ContentUnitFactory.INSTALL_ARTIFACT);
        unit.setInstallableUnits(Arrays.asList(iu1));
        
        assertEquals(1, unit.getInstallableUnits().size());
        assertEquals("iu3ID", unit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getContentUnitID());
        assertEquals("iu3ResourceRef", unit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getTargetResourceReference());
        assertEquals("iuArt3ContentRef", unit.getInstallableUnits().toArray(new InstallableUnit[0])[0].getInstallArtifact().getContentReference());

        assertEquals(1, session.getChildren(element, "InstallableUnit").size());
        assertEquals("iu3ID", session.getChildren(element, "InstallableUnit").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("iu3ResourceRef", session.getChildren(element, "InstallableUnit").toArray(new Element[0])[0].getAttribute("targetResourceRef"));
        assertEquals("iuArt3ContentRef", session.getChildren(session.getChildren(session.getChildren(element, "InstallableUnit").toArray(new Element[0])[0], "Artifacts").toArray(new Element[0])[0], "InstallArtifact").toArray(new Element[0])[0].getAttribute("contentRef"));
    }

    public void testSetRequirements()
    {
        Requirement req1 = SPISession.DEFAULT_INSTANCE.createRequirementFactory().createRequirement("req3ID");
        unit.setRequirements(Arrays.asList(req1));
        
        assertEquals(1, unit.getRequirements().size());
        assertEquals("req3ID", unit.getRequirements().toArray(new Requirement[0])[0].getRequirementID());

        assertEquals(1, session.getChildren(session.getChildren(element, "Requirements").toArray(new Element[0])[0], "Requirement").size());
        assertEquals("req3ID", session.getChildren(session.getChildren(element, "Requirements").toArray(new Element[0])[0], "Requirement").toArray(new Element[0])[0].getAttribute("id"));
    }

    public void testSetContentUnitID()
    {
        unit.setContentUnitID("differentID");
        
        assertEquals("differentID", unit.getContentUnitID());
        assertEquals("differentID", element.getAttribute("id"));
    }

    public void testSetIdentityData()
    {
        Identity identity = SPISession.DEFAULT_INSTANCE.createBaseFactory().createIdentity(new TranslatableString("differentName", "differentNameKey"));
        unit.setIdentityData(identity);
        
        assertEquals("differentName", unit.getIdentityData().getName().getValue());
        assertEquals("differentNameKey", unit.getIdentityData().getName().getTranslationKey());
        
        assertEquals("differentName", session.getChildren(session.getChildren(element, "Identity").toArray(new Element[0])[0], "Name").toArray(new Element[0])[0].getTextContent());
        assertEquals("differentNameKey", session.getChildren(session.getChildren(element, "Identity").toArray(new Element[0])[0], "Name").toArray(new Element[0])[0].getAttribute("translationKey"));
    }

    public void testSetResourceProperties()
    {
        ResourceProperty prop1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createResourceProperty("prop3ID", "prop3Ref");
        unit.setResourceProperties(Arrays.asList(prop1));
        
        assertEquals(1, unit.getResourceProperties().size());
        assertEquals("prop3ID", unit.getResourceProperties().toArray(new ResourceProperty[0])[0].getVariableID());
        assertEquals("prop3Ref", unit.getResourceProperties().toArray(new ResourceProperty[0])[0].getResourceReference());

        assertEquals(1, session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "ResourceProperty").size());
        assertEquals("prop3ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "ResourceProperty").toArray(new Element[0])[0].getAttribute("id"));
        assertEquals("prop3Ref", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "ResourceProperty").toArray(new Element[0])[0].getAttribute("resourceRef"));
    }

    public void testSetDerivedVariables()
    {
        DerivedVariable dvar1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createDerivedVariable("dvar3ID", "dvar3Expression");
        unit.setDerivedVariables(Arrays.asList(dvar1));
        
        assertEquals(1, unit.getDerivedVariables().size());
        assertEquals("dvar3ID", unit.getDerivedVariables().toArray(new DerivedVariable[0])[0].getVariableID());
        assertEquals("dvar3Expression", unit.getDerivedVariables().toArray(new DerivedVariable[0])[0].getExpression());

        assertEquals(3, session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").size());
        assertEquals("dvar3ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[2].getAttribute("id"));
        assertEquals("dvar3Expression", session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[2], "Expression").toArray(new Element[0])[0].getTextContent());
    }

    public void testSetConditionalDerivedVariables()
    {
        Condition con1 = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        con1.setDisplayName(new TranslatableString("con5Name", "con5NameKey"));
        ConditionalDerivedVariableExpression exp1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createConditionalDerivedVariableExpression("exp3Expression", con1);
        ConditionalDerivedVariable var1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createConditionalDerivedVariable("var3ID", Arrays.asList(exp1));
        unit.setConditionalDerivedVariables(Arrays.asList(var1));
        
        assertEquals(1, unit.getConditionalDerivedVariables().size());
        assertEquals("var3ID", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getVariableID());
        assertEquals("exp3Expression", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getExpression());
        assertEquals("con5Name", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getValue());
        assertEquals("con5NameKey", unit.getConditionalDerivedVariables().toArray(new ConditionalDerivedVariable[0])[0].getConditionalDerivedExpressions().toArray(new ConditionalDerivedVariableExpression[0])[0].getCondition().getDisplayName().getTranslationKey());

        assertEquals(3, session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").size());
        assertEquals("var3ID", session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[2].getAttribute("id"));
        assertEquals("exp3Expression", session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[2], "ConditionalExpression").toArray(new Element[0])[0], "Expression").toArray(new Element[0])[0].getTextContent());
        assertEquals("con5Name", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[2], "ConditionalExpression").toArray(new Element[0])[0], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getTextContent());
        assertEquals("con5NameKey", session.getChildren(session.getChildren(session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "DerivedVariable").toArray(new Element[0])[2], "ConditionalExpression").toArray(new Element[0])[0], "Condition").toArray(new Element[0])[0], "DisplayName").toArray(new Element[0])[0].getAttribute("translationKey"));
    }

    public void testSetParameterGroups()
    {
        BooleanParameter param1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createBooleanParameter("param5ID");
        ParameterGroup group1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createParameterGroup();
        group1.setBooleanParameters(Arrays.asList(param1));
        unit.setParameterGroups(Arrays.asList(group1));

        assertEquals(1, unit.getParameterGroups().size());
        assertEquals(1, unit.getParameterGroups().toArray(new ParameterGroup[0])[0].getBooleanParameters().size());
        assertEquals("param5ID", unit.getParameterGroups().toArray(new ParameterGroup[0])[0].getBooleanParameters().toArray(new BooleanParameter[0])[0].getParameterID());

        assertEquals(1, session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").size());
        assertEquals(1, session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").toArray(new Element[0])[0], "BooleanParameter").size());
        assertEquals("param5ID", session.getChildren(session.getChildren(session.getChildren(element, "Variables").toArray(new Element[0])[0], "Parameters").toArray(new Element[0])[0], "BooleanParameter").toArray(new Element[0])[0].getAttribute("id"));
    }
    
}
