/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.cosmos.me.internal.deployment.sdd.tests.common.spi;

import java.util.Arrays;
import java.util.Collection;
import java.util.Properties;

import junit.framework.TestCase;

import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPIDataObject;
import org.eclipse.cosmos.me.internal.deployment.sdd.common.spi.SPISessionImpl;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.SPISession;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.TranslatableString;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.condition.Condition;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Feature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Language;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.LanguageSet;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.Multiplicity;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.NestedFeature;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.contentunit.PackageFeatureReference;
import org.eclipse.cosmos.me.provisional.deployment.sdd.common.spi.variable.DerivedVariable;
import org.w3c.dom.Element;

public class Feature_Test extends TestCase
{
    Feature feature;
    Feature clonedFeature;
    Element element;
    SPISessionImpl session;

    public Feature_Test(String name)
    {
        super(name);
    }

    protected void setUp() throws Exception
    {
        super.setUp();

        DerivedVariable var1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createVariable("var1ID", "var1Expression");
        DerivedVariable var2 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createVariable("var2ID", "var2Expression");
        feature = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createFeatureWithDerivedVariables("featureID", Arrays.asList(var1, var2));

        feature.setAddOn(Boolean.TRUE);
        feature.setDisplayName(new TranslatableString("displayName", "displayNameKey"));

        Condition condition = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        condition.setDescription(new TranslatableString("conditionDesc", "conditionDescKey"));
        feature.setCondition(condition);
        
        feature.setContentElements(Arrays.asList("content1", "content2"));
        feature.setDescription(new TranslatableString("desc", "descKey"));
        feature.setShortDescription(new TranslatableString("shortDesc", "shortDescKey"));
        
        Language lang1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("lang1Type");
        Language lang2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("lang2Type");
        feature.setMandatoryLanguages(Arrays.asList(lang1, lang2));
        
        Multiplicity multi = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createMultiplicity(Boolean.TRUE);
        feature.setMultiplicity(multi);
        
        NestedFeature feature1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createNestedFeatureWithDerivedVariables("nestedFeature1ID", Arrays.asList(var1, var2));
        NestedFeature feature2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createNestedFeatureWithDerivedVariables("nestedFeature2ID", Arrays.asList(var1, var2));
        feature.setNestedFeatures(Arrays.asList(feature1, feature2));

        Language optLang1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("optLang1Type");
        Language optLang2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("optLang2Type");
        feature.setOptionalLanguages(Arrays.asList(optLang1, optLang2));
        
        Language langSetLang1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("langSetLang1");
        Language langSetLang2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("langSetLang2");
        Language langSetLang3 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("langSetLang3");
        Language langSetLang4 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("langSetLang4");
        LanguageSet langSet1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguageSet(Arrays.asList(langSetLang1, langSetLang2));
        LanguageSet langSet2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguageSet(Arrays.asList(langSetLang3, langSetLang4));
        feature.setOptionalLanguageSets(Arrays.asList(langSet1, langSet2));
        
        PackageFeatureReference featureRef1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createPackageFeatureReference("contentRef1", "featureRef1");
        PackageFeatureReference featureRef2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createPackageFeatureReference("contentRef2", "featureRef2");
        feature.setPackageFeatureReferences(Arrays.asList(featureRef1, featureRef2));
        
        feature.setRequiredFeature(Arrays.asList("feature1", "feature2"));
        
        feature.setRequired(Boolean.TRUE);
        
        clonedFeature = (Feature)feature.clone();
        element = ((SPIDataObject)feature).getElement();
        session = (SPISessionImpl)SPISession.DEFAULT_INSTANCE;
    }

    public void testGetRequired()
    {
        assertTrue(feature.getRequired().booleanValue());
        assertTrue(clonedFeature.getRequired().booleanValue());
        assertTrue(Boolean.parseBoolean(element.getAttribute("required")));
    }

    public void testSetRequired()
    {
        feature.setRequired(Boolean.FALSE);
        
        assertFalse(feature.getRequired().booleanValue());
        assertFalse(Boolean.parseBoolean(element.getAttribute("required")));
    }

    public void testGetTranslatableStrings()
    {
        Properties props = ((SPIDataObject) feature).getTranslatableStrings();
        Properties clonedProps = ((SPIDataObject) clonedFeature).getTranslatableStrings();
        
        assertEquals(4, props.keySet().size());
        assertEquals(4, clonedProps.keySet().size());
        
        assertEquals("desc", props.get("descKey"));
        assertEquals("shortDesc", props.get("shortDescKey"));
        assertEquals("shortDesc", props.get("shortDescKey"));
        assertEquals("displayName", props.get("displayNameKey"));
        
        assertEquals("desc", clonedProps.get("descKey"));
        assertEquals("shortDesc", clonedProps.get("shortDescKey"));
        assertEquals("conditionDesc", clonedProps.get("conditionDescKey"));
        assertEquals("displayName", clonedProps.get("displayNameKey"));
    }

    public void testGetMandatoryLanguages()
    {
        Collection<Language> langs = feature.getMandatoryLanguages();
        Collection<Language> clonedLangs = feature.getMandatoryLanguages();
        Collection<Element> dataObjectLangs = session.getChildren(session.getChild(session.getChild(element, "Languages"), "Mandatory"), "Language");
        
        assertEquals(2, langs.size());
        assertEquals(2, clonedLangs.size());
        assertEquals(2, dataObjectLangs.size());

        assertEquals("lang1Type", langs.toArray(new Language[0])[0].getType());
        assertEquals("lang2Type", langs.toArray(new Language[0])[1].getType());

        assertEquals("lang1Type", clonedLangs.toArray(new Language[0])[0].getType());
        assertEquals("lang2Type", clonedLangs.toArray(new Language[0])[1].getType());

        assertEquals("lang1Type", (dataObjectLangs.toArray(new Element[0])[0]).getAttribute("type"));
        assertEquals("lang2Type", (dataObjectLangs.toArray(new Element[0])[1]).getAttribute("type"));
    }

    public void testGetOptionalLanguages()
    {
        Collection<Language> langs = feature.getOptionalLanguages();
        Collection<Language> clonedLangs = feature.getOptionalLanguages();
        Collection<Element> dataObjectLangs = session.getChildren(session.getChild(session.getChild(element, "Languages"), "Optional"), "Language");
        	
        assertEquals(2, langs.size());
        assertEquals(2, clonedLangs.size());
        assertEquals(2, dataObjectLangs.size());
        
        assertEquals("optLang1Type", langs.toArray(new Language[0])[0].getType());
        assertEquals("optLang2Type", langs.toArray(new Language[0])[1].getType());

        assertEquals("optLang1Type", clonedLangs.toArray(new Language[0])[0].getType());
        assertEquals("optLang2Type", clonedLangs.toArray(new Language[0])[1].getType());

        assertEquals("optLang1Type", (dataObjectLangs.toArray(new Element[0])[0]).getAttribute("type"));
        assertEquals("optLang2Type", (dataObjectLangs.toArray(new Element[0])[1]).getAttribute("type"));
    }

    public void testGetOptionalLanguageSets()
    {
        Collection<LanguageSet> langSets = feature.getOptionalLanguageSets();
        Collection<LanguageSet> clonedLangSets = feature.getOptionalLanguageSets();
        Collection<Element> dataObjectLangSets = session.getChildren(session.getChild(session.getChild(element, "Languages"), "Optional"), "LanguageSet");
        
        assertEquals(2, langSets.size());
        assertEquals(2, clonedLangSets.size());
        assertEquals(2, dataObjectLangSets.size());
        
        assertEquals("langSetLang1", langSets.toArray(new LanguageSet[0])[0].getLanguages().toArray(new Language[0])[0].getType());
        assertEquals("langSetLang2", langSets.toArray(new LanguageSet[0])[0].getLanguages().toArray(new Language[0])[1].getType());
        assertEquals("langSetLang3", langSets.toArray(new LanguageSet[0])[1].getLanguages().toArray(new Language[0])[0].getType());
        assertEquals("langSetLang4", langSets.toArray(new LanguageSet[0])[1].getLanguages().toArray(new Language[0])[1].getType());

        assertEquals("langSetLang1", clonedLangSets.toArray(new LanguageSet[0])[0].getLanguages().toArray(new Language[0])[0].getType());
        assertEquals("langSetLang2", clonedLangSets.toArray(new LanguageSet[0])[0].getLanguages().toArray(new Language[0])[1].getType());
        assertEquals("langSetLang3", clonedLangSets.toArray(new LanguageSet[0])[1].getLanguages().toArray(new Language[0])[0].getType());
        assertEquals("langSetLang4", clonedLangSets.toArray(new LanguageSet[0])[1].getLanguages().toArray(new Language[0])[1].getType());

        assertEquals("langSetLang1", (session.getChildren((dataObjectLangSets.toArray(new Element[0])[0]), "Language").toArray(new Element[0])[0]).getAttribute("type"));
        assertEquals("langSetLang2", (session.getChildren((dataObjectLangSets.toArray(new Element[0])[0]), "Language").toArray(new Element[0])[1]).getAttribute("type"));
        assertEquals("langSetLang3", (session.getChildren((dataObjectLangSets.toArray(new Element[0])[1]), "Language").toArray(new Element[0])[0]).getAttribute("type"));
        assertEquals("langSetLang4", (session.getChildren((dataObjectLangSets.toArray(new Element[0])[1]), "Language").toArray(new Element[0])[1]).getAttribute("type"));
    }

    public void testGetCondition()
    {
        assertEquals("conditionDesc", feature.getCondition().getDescription().getValue());
        assertEquals("conditionDescKey", feature.getCondition().getDescription().getTranslationKey());

        assertEquals("conditionDesc", clonedFeature.getCondition().getDescription().getValue());
        assertEquals("conditionDescKey", clonedFeature.getCondition().getDescription().getTranslationKey());

        assertEquals("conditionDesc", session.getChild(session.getChild(element, "Condition"), "Description").getTextContent());
        assertEquals("conditionDescKey", session.getChild(session.getChild(element, "Condition"), "Description").getAttribute("translationKey"));
    }

    public void testGetContentElements()
    {
        assertEquals(2, feature.getContentElements().size());
        assertEquals("content1", feature.getContentElements().toArray(new String[0])[0]);
        assertEquals("content2", feature.getContentElements().toArray(new String[0])[1]);

        assertEquals(2, clonedFeature.getContentElements().size());
        assertEquals("content1", clonedFeature.getContentElements().toArray(new String[0])[0]);
        assertEquals("content2", clonedFeature.getContentElements().toArray(new String[0])[1]);

        assertEquals(2, session.getChildren(element, "ContentElement").size());
        assertEquals("content1", (session.getChildren(element, "ContentElement").toArray(new Element[0])[0]).getAttribute("contentElementRef"));
        assertEquals("content2", (session.getChildren(element, "ContentElement").toArray(new Element[0])[1]).getAttribute("contentElementRef"));
    }

    public void testGetMultiplicity()
    {
        assertTrue(feature.getMultiplicity().getMultiplesAllowed().booleanValue());
        assertTrue(clonedFeature.getMultiplicity().getMultiplesAllowed().booleanValue());
        assertTrue(Boolean.parseBoolean((session.getChildren(element, "Multiplicity").toArray(new Element[0])[0]).getAttribute("multiplesAllowed")));
    }

    public void testGetFeatureID()
    {
        assertEquals("featureID", feature.getFeatureID());
        assertEquals("featureID", clonedFeature.getFeatureID());
        assertEquals("featureID", element.getAttribute("id"));
    }

    public void testGetNestedFeatures()
    {
        assertEquals(2, feature.getNestedFeatures().size());
        assertEquals("nestedFeature1ID", feature.getNestedFeatures().toArray(new NestedFeature[0])[0].getFeatureID());
        assertEquals("nestedFeature2ID", feature.getNestedFeatures().toArray(new NestedFeature[0])[1].getFeatureID());

        assertEquals(2, clonedFeature.getNestedFeatures().size());
        assertEquals("nestedFeature1ID", clonedFeature.getNestedFeatures().toArray(new NestedFeature[0])[0].getFeatureID());
        assertEquals("nestedFeature2ID", clonedFeature.getNestedFeatures().toArray(new NestedFeature[0])[1].getFeatureID());

        assertEquals(2, session.getChildren(element, "NestedFeature").size());
        assertEquals("nestedFeature1ID", (session.getChildren(element, "NestedFeature").toArray(new Element[0])[0]).getAttribute("id"));
        assertEquals("nestedFeature2ID", (session.getChildren(element, "NestedFeature").toArray(new Element[0])[1]).getAttribute("id"));
    }

    public void testGetRequiredFeature()
    {
        assertEquals(2, feature.getRequiredFeature().size());
        assertEquals("feature1", feature.getRequiredFeature().toArray(new String[0])[0]);
        assertEquals("feature2", feature.getRequiredFeature().toArray(new String[0])[1]);
        
        assertEquals(2, clonedFeature.getRequiredFeature().size());
        assertEquals("feature1", clonedFeature.getRequiredFeature().toArray(new String[0])[0]);
        assertEquals("feature2", clonedFeature.getRequiredFeature().toArray(new String[0])[1]);

        assertEquals(2, session.getChildren(element, "RequiredFeature").size());
        assertEquals("feature1", (session.getChildren(element, "RequiredFeature").toArray(new Element[0])[0]).getAttribute("featureRef"));
        assertEquals("feature2", (session.getChildren(element, "RequiredFeature").toArray(new Element[0])[1]).getAttribute("featureRef"));
    }

    public void testGetVariables()
    {
        assertEquals(2, feature.getVariables().size());
        assertEquals("var1ID", feature.getVariables().toArray(new DerivedVariable[0])[0].getVariableID());
        assertEquals("var1Expression", feature.getVariables().toArray(new DerivedVariable[0])[0].getExpression());
        assertEquals("var2ID", feature.getVariables().toArray(new DerivedVariable[0])[1].getVariableID());
        assertEquals("var2Expression", feature.getVariables().toArray(new DerivedVariable[0])[1].getExpression());

        assertEquals(2, clonedFeature.getVariables().size());
        assertEquals("var1ID", clonedFeature.getVariables().toArray(new DerivedVariable[0])[0].getVariableID());
        assertEquals("var1Expression", clonedFeature.getVariables().toArray(new DerivedVariable[0])[0].getExpression());
        assertEquals("var2ID", clonedFeature.getVariables().toArray(new DerivedVariable[0])[1].getVariableID());
        assertEquals("var2Expression", clonedFeature.getVariables().toArray(new DerivedVariable[0])[1].getExpression());

        assertEquals(2, session.getChildren(element, "Variable").size());
        assertEquals("var1ID", (session.getChildren(element, "Variable").toArray(new Element[0])[0]).getAttribute("id"));
        assertEquals("var1Expression", session.getChildren((session.getChildren(element, "Variable").toArray(new Element[0])[0]), "Expression").toArray(new Element[0])[0].getTextContent());
        assertEquals("var2ID", (session.getChildren(element, "Variable").toArray(new Element[0])[1]).getAttribute("id"));
        assertEquals("var2Expression", session.getChildren((session.getChildren(element, "Variable").toArray(new Element[0])[1]), "Expression").toArray(new Element[0])[0].getTextContent());
    }

    public void testGetPackageFeatureReferences()
    {
        assertEquals(2, feature.getPackageFeatureReferences().size());
        assertEquals("contentRef1", feature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[0].getContentElementRef());
        assertEquals("featureRef1", feature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[0].getPackageFeatureRef());
        assertEquals("contentRef2", feature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[1].getContentElementRef());
        assertEquals("featureRef2", feature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[1].getPackageFeatureRef());

        assertEquals(2, clonedFeature.getPackageFeatureReferences().size());
        assertEquals("contentRef1", clonedFeature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[0].getContentElementRef());
        assertEquals("featureRef1", clonedFeature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[0].getPackageFeatureRef());
        assertEquals("contentRef2", clonedFeature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[1].getContentElementRef());
        assertEquals("featureRef2", clonedFeature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[1].getPackageFeatureRef());
        
        assertEquals(2, session.getChildren(element, "PackageFeatureReference").size());
        assertEquals("contentRef1", (session.getChildren(element, "PackageFeatureReference").toArray(new Element[0])[0]).getAttribute("contentElementRef"));
        assertEquals("featureRef1", (session.getChildren(element, "PackageFeatureReference").toArray(new Element[0])[0]).getAttribute("packageFeatureRef"));
        assertEquals("contentRef2", (session.getChildren(element, "PackageFeatureReference").toArray(new Element[0])[1]).getAttribute("contentElementRef"));
        assertEquals("featureRef2", (session.getChildren(element, "PackageFeatureReference").toArray(new Element[0])[1]).getAttribute("packageFeatureRef"));
    }

    public void testGetAddOn()
    {
        assertTrue(feature.getAddOn().booleanValue());
        assertTrue(clonedFeature.getAddOn().booleanValue());
        assertTrue(Boolean.parseBoolean(element.getAttribute("addOn")));
    }

    public void testGetDisplayName()
    {
        assertEquals("displayName", feature.getDisplayName().getValue());
        assertEquals("displayNameKey", feature.getDisplayName().getTranslationKey());

        assertEquals("displayName", clonedFeature.getDisplayName().getValue());
        assertEquals("displayNameKey", clonedFeature.getDisplayName().getTranslationKey());
        
        assertEquals("displayName", (session.getChildren(element, "DisplayName").toArray(new Element[0])[0]).getTextContent());
        assertEquals("displayNameKey", (session.getChildren(element, "DisplayName").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testGetDescription()
    {
        assertEquals("desc", feature.getDescription().getValue());
        assertEquals("descKey", feature.getDescription().getTranslationKey());

        assertEquals("desc", clonedFeature.getDescription().getValue());
        assertEquals("descKey", clonedFeature.getDescription().getTranslationKey());
        
        assertEquals("desc", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getTextContent());
        assertEquals("descKey", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testGetShortDescription()
    {
        assertEquals("shortDesc", feature.getShortDescription().getValue());
        assertEquals("shortDescKey", feature.getShortDescription().getTranslationKey());

        assertEquals("shortDesc", clonedFeature.getShortDescription().getValue());
        assertEquals("shortDescKey", clonedFeature.getShortDescription().getTranslationKey());
        
        assertEquals("shortDesc", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getTextContent());
        assertEquals("shortDescKey", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetMandatoryLanguages()
    {
        Language lang1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("lang3Type");
        feature.setMandatoryLanguages(Arrays.asList(lang1));
        
        Collection<Language> langs = feature.getMandatoryLanguages();
        Collection<Element> dataObjectLangs = session.getChildren(session.getChild(session.getChild(element, "Languages"), "Mandatory"), "Language");
        
        assertEquals(1, langs.size());
        assertEquals(1, dataObjectLangs.size());

        assertEquals("lang3Type", langs.toArray(new Language[0])[0].getType());
        assertEquals("lang3Type", (dataObjectLangs.toArray(new Element[0])[0]).getAttribute("type"));
    }

    public void testSetOptionalLanguages()
    {
        Language optLang1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("optLang3Type");
        feature.setOptionalLanguages(Arrays.asList(optLang1));
        
        Collection<Language> langs = feature.getOptionalLanguages();
        Collection<Element> dataObjectLangs = session.getChildren(session.getChild(session.getChild(element, "Languages"), "Optional"), "Language");

        assertEquals(1, langs.size());
        assertEquals(1, dataObjectLangs.size());
        
        assertEquals("optLang3Type", langs.toArray(new Language[0])[0].getType());
        assertEquals("optLang3Type", (dataObjectLangs.toArray(new Element[0])[0]).getAttribute("type"));
    }

    public void testSetOptionalLanguageSets()
    {
        Language langSetLang1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("langSetLang5");
        Language langSetLang2 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguage("langSetLang6");
        LanguageSet langSet1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createLanguageSet(Arrays.asList(langSetLang1, langSetLang2));
        feature.setOptionalLanguageSets(Arrays.asList(langSet1));

        Collection<LanguageSet> langSets = feature.getOptionalLanguageSets();
        Collection<Element> dataObjectLangSets = session.getChildren(session.getChild(session.getChild(element, "Languages"), "Optional"), "LanguageSet");
        
        assertEquals(1, langSets.size());
        assertEquals(1, dataObjectLangSets.size());
        
        assertEquals("langSetLang5", langSets.toArray(new LanguageSet[0])[0].getLanguages().toArray(new Language[0])[0].getType());
        assertEquals("langSetLang6", langSets.toArray(new LanguageSet[0])[0].getLanguages().toArray(new Language[0])[1].getType());

        assertEquals("langSetLang5", (session.getChildren((dataObjectLangSets.toArray(new Element[0])[0]), "Language").toArray(new Element[0])[0]).getAttribute("type"));
        assertEquals("langSetLang6", (session.getChildren((dataObjectLangSets.toArray(new Element[0])[0]), "Language").toArray(new Element[0])[1]).getAttribute("type"));
    }

    public void testSetCondition()
    {
        Condition condition = SPISession.DEFAULT_INSTANCE.createConditionFactory().createCondition();
        condition.setDescription(new TranslatableString("differentDesc", "differentDescKey"));
        feature.setCondition(condition);
        
        assertEquals("differentDesc", feature.getCondition().getDescription().getValue());
        assertEquals("differentDescKey", feature.getCondition().getDescription().getTranslationKey());

        assertEquals("differentDesc", session.getChild(session.getChild(element, "Condition"), "Description").getTextContent());
        assertEquals("differentDescKey", session.getChild(session.getChild(element, "Condition"), "Description").getAttribute("translationKey"));
    }

    public void testSetContentElements()
    {
        feature.setContentElements(Arrays.asList("content3"));
     
        assertEquals(1, feature.getContentElements().size());
        assertEquals("content3", feature.getContentElements().toArray(new String[0])[0]);

        assertEquals(1, session.getChildren(element, "ContentElement").size());
        assertEquals("content3", (session.getChildren(element, "ContentElement").toArray(new Element[0])[0]).getAttribute("contentElementRef"));
    }

    public void testSetMultiplicity()
    {
        Multiplicity multi = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createMultiplicity(Boolean.FALSE);
        feature.setMultiplicity(multi);
        
        assertFalse(feature.getMultiplicity().getMultiplesAllowed().booleanValue());
        assertFalse(Boolean.parseBoolean((session.getChildren(element, "Multiplicity").toArray(new Element[0])[0]).getAttribute("multiplesAllowed")));
    }

    public void testSetFeatureID()
    {
        feature.setFeatureID("differentID");
        
        assertEquals("differentID", feature.getFeatureID());
        assertEquals("differentID", element.getAttribute("id"));
    }

    public void testSetNestedFeatures()
    {
        DerivedVariable var1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createDerivedVariable("var3ID", "var3Expression");
        NestedFeature feature1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createNestedFeatureWithDerivedVariables("nestedFeature3ID", Arrays.asList(var1));
        feature.setNestedFeatures(Arrays.asList(feature1));

        assertEquals(1, feature.getNestedFeatures().size());
        assertEquals(1, session.getChildren(element, "NestedFeature").size());
        
        assertEquals("nestedFeature3ID", feature.getNestedFeatures().toArray(new NestedFeature[0])[0].getFeatureID());
        assertEquals("nestedFeature3ID", (session.getChildren(element, "NestedFeature").toArray(new Element[0])[0]).getAttribute("id"));
    }

    public void testSetRequiredFeature()
    {
        feature.setRequiredFeature(Arrays.asList("feature3"));
        
        assertEquals(1, feature.getRequiredFeature().size());
        assertEquals("feature3", feature.getRequiredFeature().toArray(new String[0])[0]);

        assertEquals(1, session.getChildren(element, "RequiredFeature").size());
        assertEquals("feature3", (session.getChildren(element, "RequiredFeature").toArray(new Element[0])[0]).getAttribute("featureRef"));
    }

    public void testSetPackageFeatureReferences()
    {
        PackageFeatureReference featureRef1 = SPISession.DEFAULT_INSTANCE.createContentUnitFactory().createPackageFeatureReference("contentRef1", "featureRef1");
        feature.setPackageFeatureReferences(Arrays.asList(featureRef1));
        
        assertEquals(1, feature.getPackageFeatureReferences().size());
        assertEquals("contentRef1", feature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[0].getContentElementRef());
        assertEquals("featureRef1", feature.getPackageFeatureReferences().toArray(new PackageFeatureReference[0])[0].getPackageFeatureRef());

        assertEquals(1, session.getChildren(element, "PackageFeatureReference").size());
        assertEquals("contentRef1", (session.getChildren(element, "PackageFeatureReference").toArray(new Element[0])[0]).getAttribute("contentElementRef"));
        assertEquals("featureRef1", (session.getChildren(element, "PackageFeatureReference").toArray(new Element[0])[0]).getAttribute("packageFeatureRef"));
    }

    public void testSetVariables()
    {
        DerivedVariable var1 = SPISession.DEFAULT_INSTANCE.createVariableFactory().createVariable("var3ID", "var3Expression");
        feature.setVariables(Arrays.asList(var1));
        
        assertEquals(1, feature.getVariables().size());
        assertEquals("var3ID", feature.getVariables().toArray(new DerivedVariable[0])[0].getVariableID());
        assertEquals("var3Expression", feature.getVariables().toArray(new DerivedVariable[0])[0].getExpression());

        assertEquals(1, session.getChildren(element, "Variable").size());
        assertEquals("var3ID", (session.getChildren(element, "Variable").toArray(new Element[0])[0]).getAttribute("id"));
        assertEquals("var3Expression", session.getChildren((session.getChildren(element, "Variable").toArray(new Element[0])[0]), "Expression").toArray(new Element[0])[0].getTextContent());
    }

    public void testSetAddOn()
    {
        feature.setAddOn(Boolean.FALSE);

        assertFalse(feature.getAddOn().booleanValue());
        assertFalse(Boolean.parseBoolean(element.getAttribute("addOn")));
    }

    public void testSetDisplayName()
    {
        feature.setDisplayName(new TranslatableString("differentName", "differentNameKey"));
        
        assertEquals("differentName", feature.getDisplayName().getValue());
        assertEquals("differentNameKey", feature.getDisplayName().getTranslationKey());

        assertEquals("differentName", (session.getChildren(element, "DisplayName").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentNameKey", (session.getChildren(element, "DisplayName").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetDescription()
    {
        feature.setDescription(new TranslatableString("differentDesc", "differentDescKey"));
        
        assertEquals("differentDesc", feature.getDescription().getValue());
        assertEquals("differentDescKey", feature.getDescription().getTranslationKey());

        assertEquals("differentDesc", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentDescKey", (session.getChildren(element, "Description").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }

    public void testSetShortDescription()
    {
        feature.setShortDescription(new TranslatableString("differentShortDesc", "differentShortDescKey"));

        assertEquals("differentShortDesc", feature.getShortDescription().getValue());
        assertEquals("differentShortDescKey", feature.getShortDescription().getTranslationKey());

        assertEquals("differentShortDesc", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getTextContent());
        assertEquals("differentShortDescKey", (session.getChildren(element, "ShortDescription").toArray(new Element[0])[0]).getAttribute("translationKey"));
    }
}
